<?php

namespace Drupal\openagenda\ParamConverter;

use Drupal\Core\ParamConverter\ParamConverterInterface;
use Drupal\openagenda\OpenagendaConnectorInterface;
use Drupal\openagenda\OpenagendaEventProcessorInterface;
use Drupal\openagenda\OpenagendaHelperInterface;
use Symfony\Component\Routing\Route;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Converts parameters for upcasting event to full object.
 */
class OpenagendaEventConverter implements ParamConverterInterface {

  public function __construct(
    protected OpenagendaConnectorInterface $connector,
    protected OpenagendaEventProcessorInterface $eventProcessor,
    protected OpenagendaHelperInterface $helper,
    protected RequestStack $requestStack,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function convert($value, $definition, $name, array $defaults) {
    if (
      preg_match('/^[a-z0-9-_]+$/', $value)
      && !empty($defaults['node'])
      && $defaults['node']->hasField('field_openagenda')
    ) {
      $agenda_uid = $defaults['node']->get('field_openagenda')->uid;
      $lang = $this->helper->getPreferredLanguage($defaults['node']->get('field_openagenda')->language);
      $request = $this->requestStack->getCurrentRequest();
      $oac = $request->query->get('oac');
      $base_url = $request->getSchemeAndHttpHost() . $request->getBaseUrl() . $request->getPathInfo();

      // If an oac parameter is provided, we first try to get an event triplet
      // to get previous, current & next event in one request.
      if (!empty($oac)) {
        $context = $this->helper->decodeContext($oac);

        if (isset($context['index'])) {
          $filters = !empty($context['filters']) ? $context['filters'] : [];
          $event_triplet = $this->connector->getEventTriplet($agenda_uid, $filters, $context['index'], (bool) $defaults['node']->get('field_openagenda')->first()->get('include_embedded')->getValue());

          // We check at least a current event was found and also that its slug
          // matches with the url in case a wrong oac was given.
          if (isset($event_triplet['current']['slug'])
              && $event_triplet['current']['slug'] == $value) {
            $event = $event_triplet['current'];

            if (!empty($event_triplet['previous']) && !empty($event_triplet['previous']['slug'])) {
              $event['previousEventSlug'] = $event_triplet['previous']['slug'];
            }

            if (!empty($event_triplet['next']) && !empty($event_triplet['next']['slug'])) {
              $event['nextEventSlug'] = $event_triplet['next']['slug'];
            }
          }
        }
      }

      // Failing that, we try to get the event from its slug.
      if (empty($event)) {
        $event = $this->connector->getEventBySlug($agenda_uid, $value, (bool) $defaults['node']->get('field_openagenda')->first()->get('include_embedded')->getValue());
      }

      $event['baseUrl'] = $base_url;
      $event['timetable'] = $this->eventProcessor->processEventTimetable($event, $lang);

      return $event;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function applies($definition, $name, Route $route) {
    return (!empty($definition['type']) && $definition['type'] == 'openagenda_event');
  }

}
