<?php

namespace Drupal\openagenda\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\openagenda\OpenagendaHelperInterface;
use OpenAgendaSdk\OpenAgendaSdk;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the OpenAgenda additional field filter Block.
 *
 * @Block(
 *   id = "openagenda_additional_field_filter_block",
 *   admin_label = @Translation("OpenAgenda - Additional field filter"),
 *   category = @Translation("OpenAgenda"),
 *   context_definitions = {
 *     "node" = @ContextDefinition("entity:node", label = @Translation("Node"))
 *   },
 * )
 */
class OpenagendaAdditionalFieldFilterBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * OpenAgenda module configuration object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $moduleConfig;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    ConfigFactoryInterface $config_factory,
    protected LanguageManagerInterface $languageManager,
    protected RouteMatchInterface $routeMatch,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected OpenagendaHelperInterface $helper,
    protected OpenAgendaSdk $sdk,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->moduleConfig = $config_factory->get('openagenda.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('language_manager'),
      $container->get('current_route_match'),
      $container->get('entity_type.manager'),
      $container->get('openagenda.helper'),
      $container->get('openagenda.sdk'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account) {
    return AccessResult::allowedIfHasPermission($account, 'access content');
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $node = $this->getContextValue('node');
    $block = [];
    $config = $this->getConfiguration();

    // Check that we have an OpenAgenda node and that we are hitting the base
    // route (not an event).
    if ($node && $node->hasField('field_openagenda') && $this->routeMatch->getRouteName() == 'entity.node.canonical') {
      $preFilters = $this->helper->getPreFilters($node);
      $defaultPlaceholder = $this->moduleConfig->get('openagenda.default_search_filter_placeholder');
      // phpcs:ignore Drupal.Semantics.FunctionT.NotLiteralString
      $placeholder = !empty($this->configuration['input_placeholder']) ? $this->configuration['input_placeholder'] : $this->t($defaultPlaceholder);
      // phpcs:ignore Drupal.Semantics.FunctionT.NotLiteralString
      $aria_label = !empty($this->configuration['input_aria_label']) ? $this->configuration['input_aria_label'] : $this->t($defaultPlaceholder);
      $option_tag = !empty($this->configuration['option_tag']) ? $this->configuration['option_tag'] : 'div';

      // Only display in preFilters doesn't contain an thematique entry.
      if (!isset($preFilters['thematique'])) {
        $block = [
          '#theme' => 'block__openagenda_additional_field_filter',
          '#additional_field' => $config['additional_field'],
          '#placeholder' => $placeholder,
          '#aria_label' => $aria_label,
          '#option_tag' => $option_tag,
        ];
      }
    }

    return $block;
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);

    $config = $this->getConfiguration();

    // Get all nodes of type openagenda.
    $nodes_openagenda = $this->entityTypeManager->getStorage('node')->loadByProperties(['type' => 'openagenda']);
    $nodes_options = [];
    foreach ($nodes_openagenda as $node) {
      $nodes_options[$node->id()] = $node->label();
    }

    $form['options_agenda'] = [
      '#type' => 'select',
      '#title' => $this->t('Agenda'),
      '#description' => $this->t('Select the agenda to limit the options.'),
      '#empty_option' => $this->t('- Select an agenda -'),
      '#options' => $nodes_options,
      '#default_value' => $config['options_agenda'] ?? NULL,
      '#states' => [
        'visible' => [
          ':input[name="settings[options_limited]"]' => ['checked' => TRUE],
        ],
      ],
      '#ajax' => [
        'callback' => [$this, 'ajaxGetFieldOptionsCallback'],
        'event' => 'change',
        'wrapper' => 'settings-additional-field-wrapper',
      ],
    ];

    // Get additional fields options from the selected agenda.
    $field_options = [];

    // Get values submitted after ajax.
    $values = $form_state->getUserInput();
    // Get Node Agenda ID default selected.
    $agenda_node_id = $config['options_agenda'] ?? NULL;
    // If has agenda node selected.
    if (isset($values['settings']['options_agenda']) && !empty($values['settings']['options_agenda'])) {
      $agenda_node_id = $values['settings']['options_agenda'];
    }
    // If has agenda node selected, filter type is not empty and filter type is
    // supported.
    if (!empty($agenda_node_id) && isset($nodes_openagenda[$agenda_node_id])) {
      $node_openagenda = $nodes_openagenda[$agenda_node_id];
      // Get agenda settings API.
      $agenda_settings = $node_openagenda->get('field_openagenda')->settings ?? '{}';
      $agenda_settings = json_decode($agenda_settings, TRUE);
      // If has settings.
      if (!empty($agenda_settings)) {
        // Get current language.
        $current_language = $this->languageManager->getCurrentLanguage();
        // Get language code (e.g., 'en', 'fr')
        $langcode = $current_language->getId();
        // Each field.
        foreach ($agenda_settings['schema']['fields'] as $v_field) {
          // If field is an additional field.
          if (in_array($v_field['schemaType'], ['agenda', 'network'])) {
            // If has options index OR is type boolean.
            if (isset($v_field['options']) || $v_field['fieldType'] === 'boolean') {
              // Add to options.
              $field_label = $v_field['label'];
              if (is_array($field_label)) {
                // Multilingual label.
                if (isset($field_label[$langcode])) {
                  $field_label = $field_label[$langcode];
                }
                else {
                  // Fallback to first value.
                  $label_values = array_values($field_label);
                  $field_label = reset($label_values);
                }
              }
              $field_options[$v_field['field']] = $field_label;
            }
          }
        }
      }
    }

    $form['additional_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Additional field'),
      '#description' => $this->t('Agenda additional field name'),
      '#empty_option' => $this->t('- Select an additional field -'),
      '#options' => $field_options,
      '#default_value' => $config['additional_field'] ?? '',
      '#required' => TRUE,
      '#prefix' => '<div id="settings-additional-field-wrapper">',
      '#suffix' => '</div>',
    ];
    $form['input_placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Placeholder'),
      '#description' => $this->t('Placeholder for the input field.'),
      '#default_value' => $config['input_placeholder'] ?? $this->moduleConfig->get('openagenda.default_search_filter_placeholder'),
    ];
    $form['input_aria_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Aria label'),
      '#description' => $this->t('Aria label for the input field.'),
      '#default_value' => $config['input_aria_label'] ?? $this->moduleConfig->get('openagenda.default_search_filter_placeholder'),
    ];
    $form['option_tag'] = [
      '#type' => 'select',
      '#title' => $this->t('Tag for options'),
      '#options' => [
        'div' => $this->t('Default'),
        'label' => $this->t('Label'),
      ],
      '#default_value' => $this->configuration['option_tag'] ?? 'label',
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * Callback ajax to get filter Options from agenda and filter type.
   *
   * @param mixed $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function ajaxGetFieldOptionsCallback($form, FormStateInterface $form_state) {
    // Update field options selected.
    return $form['settings']['additional_field'];
  }

  /**
   * {@inheritdoc}
   */
  public function blockValidate($form, FormStateInterface $form_state) {
    /*
    // $customField = $form_state->getValue('additional_field');
    // $exists = FALSE;
    // ! SDK getter API not updated. Response of request is different
    // of conditions check.
    // foreach ($this->sdk->getMyAgendasUids() as $uid) {
    //   $agendaCustomFields = $this->sdk->getAgendaAdditionalFields($uid);
    //   foreach ($agendaCustomFields as $agendaCustomField) {
    //     if ($agendaCustomField == $customField) {
    //       $exists = TRUE;
    // break 2;
    //     }
    //   }
    // }.
    // If (!$exists) {
    //   $form_state->setErrorByName('additional_field', $this->t(
    //     "This additional field doesn't belong to any of your agenda(s)!"
    //   ));
    // }
     */
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $values = $form_state->getValues();
    $this->configuration['options_agenda'] = $values['options_agenda'];
    $this->configuration['additional_field'] = $values['additional_field'];
    $this->configuration['input_placeholder'] = $values['input_placeholder'];
    $this->configuration['input_aria_label'] = $values['input_aria_label'];
    $this->configuration['option_tag'] = $values['option_tag'];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    return 0;
  }

}
