<?php

namespace Drupal\openagenda\Plugin\Field\FieldWidget;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\openagenda\OpenagendaHelperInterface;
use Drupal\openagenda\OpenagendaConnectorInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Widget for the OpenAgenda field type.
 *
 * @FieldWidget(
 *   id = "openagenda_widget",
 *   label = @Translation("OpenAgenda widget"),
 *   field_types = {
 *     "openagenda",
 *   },
 * )
 */
class OpenagendaWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  /**
   * OpenAgenda module configuration object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    protected OpenagendaHelperInterface $helper,
    protected OpenagendaConnectorInterface $connector,
    ConfigFactoryInterface $config_factory,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->config = $config_factory->get('openagenda.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('openagenda.helper'),
      $container->get('openagenda.connector'),
      $container->get('config.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element += [
      '#type' => 'fieldset',
    ];

    $element['uid'] = [
      '#type' => 'textfield',
      '#title' => $this->t('OpenAgenda UID'),
      '#description' => $this->t('UID of the OpenAgenda you want to display.'),
      '#default_value' => $items[$delta]->uid ?? '',
      '#size' => 10,
      '#maxlength' => 10,
      '#element_validate' => [
        [static::class, 'validateUid'],
      ],
    ];

    $element['events_per_page'] = [
      '#type' => 'number',
      '#title' => $this->t('Events per page'),
      '#description' => $this->t('Number of events displayed per page. Enter 0 to show all events.'),
      '#default_value' => $items[$delta]->events_per_page ?? $this->config->get('openagenda.events_per_page'),
      '#size' => 3,
      '#min' => 0,
      '#max' => 300,
    ];

    $element['current'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Only current and upcoming events'),
      '#description' => $this->t('Display only current and upcoming events. If relative date filter block is configured to be displayed on agenda page, it will be disabled on every openagenda node page.'),
      '#return_value' => TRUE,
      '#default_value' => $items[$delta]->current ?? ($this->config->get('openagenda.current') ?? FALSE),
    ];

    $element['general_prefilter'] = [
      '#type' => 'textfield',
      '#title' => $this->t('General pre-filter'),
      '#description' => $this->t('Present a subset of the agenda on your page by defining this pre-filter. Set the desired filter either on the calendar events administration page on openagenda.com or on the public calendar page on your Drupal site. The URL of the page then contains the value of the filter to define. Retrieve the full URL defined in the address bar of your browser and place it in this field.'),
      '#size' => 60,
      '#maxlength' => 255,
      '#default_value' => $items[$delta]->general_prefilter ?? '',
    ];

    $language_options = ['default' => $this->t("Use site's language")] + $this->helper->getAvailableLanguages();

    $element['language'] = [
      '#type' => 'select',
      '#title' => $this->t('Language'),
      '#description' => $this->t('The language to use for this OpenAgenda.'),
      '#options' => $language_options,
      '#default_value' => $items[$delta]->language ?? $this->config->get('openagenda.default_language'),
    ];

    $element['include_embedded'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include embedded content'),
      '#description' => $this->t('Include embedded HTML content in event descriptions.'),
      '#return_value' => TRUE,
      '#default_value' => $items[$delta]->include_embedded ?? $this->config->get('openagenda.include_embedded'),
    ];

    return $element;
  }

  /**
   * Form validation handler for the quantity element.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function validateUid(array $element, FormStateInterface $form_state) {
    if ($element['#value'] !== '') {
      // Get agenda settings from the OpenAgenda API.
      $agenda_settings = \Drupal::service('openagenda.connector')->getAgendaSettings($element['#value']);
      // Store to form state for later use.
      $form_state->set('openagenda_agenda_settings', $agenda_settings);
      // If settings not found.
      if (empty($agenda_settings)) {
        // UID not exist.
        $form_state->setError($element, t('The UID does not exist.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    // Get agenda settings.
    $agenda_settings = $form_state->get('openagenda_agenda_settings');
    // Store agenda settings to field settings.
    $values[0]['settings'] = json_encode($agenda_settings);

    return $values;
  }

}
