<?php

namespace Drupal\Tests\openagenda\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Tests the OpenAgenda filter blocks with JavaScript.
 *
 * @group openagenda
 */
class OpenagendaFiltersJavascriptTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'node',
    'openagenda',
    'openagenda_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Node for testing.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Setup mock SDK response.
    $this->setupMockSdk();

    // The openagenda module provides the 'openagenda' content type
    // and 'field_openagenda' field via its config/install.
    $this->node = $this->drupalCreateNode([
      'type' => 'openagenda',
      'status' => 1,
      'field_openagenda' => [
        'uid' => '123456',
        'events_per_page' => 10,
        'language' => 'en',
        'settings' => '{}',
      ],
    ]);

    // Place the cities filter block.
    $this->drupalPlaceBlock('openagenda_cities_filter_block', [
      'label' => 'Cities Filter',
      'id' => 'cities_filter',
      'theme' => 'stark',
      'context_mapping' => [
        'node' => '@node.node_route_context:node',
      ],
    ]);

    // Place the search filter block.
    $this->drupalPlaceBlock('openagenda_search_filter_block', [
      'label' => 'Search Filter',
      'id' => 'search_filter',
      'theme' => 'stark',
      'context_mapping' => [
        'node' => '@node.node_route_context:node',
      ],
    ]);
  }

  /**
   * Sets up the mock SDK with default responses.
   *
   * @see OpenagendaFunctionalTestBase::setupMockSdk
   */
  protected function setupMockSdk($response = NULL) {
    if ($response === NULL) {
      $response = [
        'success' => TRUE,
        'total' => 1,
        'events' => [
          [
            'uid' => 123,
            'title' => ['en' => 'Mock Event'],
            'description' => ['en' => 'Mock Description'],
            'dateRange' => ['en' => 'January 1st, 2026'],
            'slug' => 'mock-event',
            'image' => NULL,
            'location' => [
              'city' => 'Mock City',
            ],
          ],
        ],
        'aggregations' => [
          'city' => [
            ['key' => 'Paris', 'count' => 1],
            ['key' => 'Lyon', 'count' => 1],
          ],
        ],
      ];
    }

    $this->container->get('state')->set('openagenda_test.events_response', $response);
  }

  /**
   * Tests that the filter blocks are initialized by the React library.
   */
  public function testFiltersInitialization() {
    $this->drupalGet($this->node->toUrl());

    $session = $this->assertSession();

    // Verify that the filter containers are present in the DOM.
    $this->assertNotNull($session->waitForElement('css', '.oa-cities-filter', 30000));
    $session->elementExists('css', '.oa-cities-filter.oa-filter');

    $this->assertNotNull($session->waitForElement('css', '.oa-search-filter', 30000));
    $session->elementExists('css', '.oa-search-filter.oa-filter');

    // Verify that the React library script tag is added.
    $this->assertNotNull($session->waitForElement('css', 'script[src*="unpkg.com/@openagenda/react-filters"]', 30000));

    // Verify that the mock event is present.
    $session->pageTextContains('Mock Event');

    // Check if the React library is loaded and the filters are initialized.
    // We wait for the "Paris" or "Lyon" text to appear which comes from
    // aggregations.
    $session->waitForText('Paris', 30000);
    $session->pageTextContains('Paris');
    $session->pageTextContains('Lyon');

    // Test search filter initialization.
    // The React library should have rendered an input field.
    $session->waitForElementVisible('css', '.oa-search-filter input', 30000);
    $session->elementExists('css', '.oa-search-filter input');

    // Simulate search.
    $page = $this->getSession()->getPage();
    $search_input = $page->find('css', '.oa-search-filter input');
    $search_input->setValue('Mock');

    // Check for the AJAX throbber which is added in openagenda-filters.js on
    // filter change.
    $session->waitForElementVisible('css', '.ajax-progress', 30000);

    // Generate HTML output for debugging purposes.
    $this->htmlOutput($this->getSession()->getPage()->getContent());
  }

}
