<?php

namespace Drupal\openai_batch\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\Routing\RouteMatch;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\openai_batch\Entity\OpenAiBatchRequest;
use Drupal\openai_batch\OpenAiBatchProcessorPluginManager;
use Drupal\openai_batch\OpenAiBatchService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\Routing\Route;

/**
 * Controller for OpenAI Batch module.
 */
class OpenAiBatchController extends ControllerBase {

  /**
   * Download the result file of a batch.
   */
  public function download(OpenAiBatchRequest $openai_batch_request) {
    if ($openai_batch_request->downloaded->value) {
      $this->messenger->addWarning($this->t('Batch @id is already downloaded.', ['@id' => $openai_batch_request->id()]));
      return $this->doRedirect($openai_batch_request);
    }
    $batch_id = $openai_batch_request->id();
    // Retrieve the batch from OpenAI.
    $batch_response = $this->openAiBatchService->getClient()
      ->batches()
      ->retrieve($batch_id);

    // Check if the status of the batch is "completed".
    if ($batch_response->status !== 'completed') {
      // The status is not "completed". We cannot download the file, but we
      // still can update the local entity with the new values.
      $this->messenger->addWarning($this->t('The status of the batch @id is @status, so the result file cannot be downloaded.', [
        '@id' => $batch_id,
        '@status' => $batch_response->status,
      ]));
      $new_revision = $this->openAiBatchService->updateOpenAiBatchRequestEntity(batch_response: $batch_response, batch_entity: $openai_batch_request, save: FALSE);
      // Do we have a new revision?
      if (!is_null($new_revision)) {
        // Yes we have a new revision, save it and move on.
        $this->messenger->addMessage($this->t('Batch @id refreshed.', ['@id' => $batch_id]));
        $new_revision->save();
        return $this->doRedirect($openai_batch_request);
      }
      else {
        // No we don't have a new revision, so nothing to do here.
        return $this->doRedirect($openai_batch_request);
      }
    }
    // The status is "completed", so we can download the file.
    // First attempt to update the local entity with the new values.
    $new_revision = $this->openAiBatchService->updateOpenAiBatchRequestEntity(batch_response: $batch_response, batch_entity: $openai_batch_request, save: FALSE);
    if (is_null($new_revision)) {
      // No revision got created, so all the other values are the same. But we
      // still must download the result file, so we will create a new revision
      // manually here.
      $new_revision = $this->entityTypeManager
        ->getStorage('openai_batch_request')
        ->createRevision($openai_batch_request);
    }
    // Download the file.
    $file = $this->openAiBatchService->getClient()
      ->files()
      ->download($batch_response->outputFileId);
    $this->openAiBatchService->saveOutputFile($batch_response->outputFileId, $file);
    $new_revision->set('downloaded', TRUE);
    $new_revision->save();
    $this->messenger->addMessage($this->t('Batch @id downloaded.', ['@id' => $batch_id]));
    return $this->doRedirect($openai_batch_request);
  }

  /**
   * Redirect to the destination set in the referrer or to the canonical URL.
   */
  private function doRedirect(OpenAiBatchRequest $openai_batch_request) {
    $redirect_url = $this->redirectDestination->get();
    if ($redirect_url) {
      return new RedirectResponse($redirect_url);
    }
    else {
      $canonical_url = $openai_batch_request->toUrl('canonical')->toString();
      return new RedirectResponse($canonical_url);
    }
  }

  /**
   * Process the result file of a batch.
   */
  public function process(OpenAiBatchRequest $openai_batch_request) {
    if ($openai_batch_request->status->value !== "completed" || !$openai_batch_request->downloaded->value) {
      $this->messenger->addWarning($this->t('Batch @id is not ready to be processed because the result file has not yet been downloaded.', ['@id' => $openai_batch_request->id()]));
      return $this->doRedirect($openai_batch_request);
    }
    $processor_id = $openai_batch_request->processor->value;
    /** @var \Drupal\openai_batch\OpenAiBatchProcessorInterface $processor */
    $processor = $this->processorPluginManager->createInstance($processor_id);
    $process_batch = $processor->getProcessBatch($openai_batch_request);
    $process_batch->addOperation([
      $processor,
      'markAsProcessedOperation',
    ], [$openai_batch_request]);
    batch_set($process_batch->toArray());
    return batch_process(Url::fromRoute('entity.openai_batch_request.collection')
      ->toString());
  }

  /**
   * Refresh a local entity with the remote batch data.
   */
  public function refresh(OpenAiBatchRequest $openai_batch_request) {
    $batch_id = $openai_batch_request->id();
    $batch = $this->openAiBatchService->getClient()
      ->batches()
      ->retrieve($batch_id);
    $new_revision = $this->openAiBatchService->updateOpenAiBatchRequestEntity(batch_response: $batch, batch_entity: $openai_batch_request);
    if (!is_null($new_revision)) {
      $this->messenger->addMessage($this->t('Batch @id refreshed.', ['@id' => $batch_id]));
    }
    else {
      $this->messenger->addMessage($this->t('Batch @id was already up to date.', ['@id' => $batch_id]));
    }
    return $this->doRedirect($openai_batch_request);
  }

  /**
   * Refresh all the local entities with the remote batch data.
   */
  public function refreshAll() {
    $batchBuilder = $this->openAiBatchService->getRefreshBatchesBatchBuilder();
    batch_set($batchBuilder->toArray());
    return batch_process(Url::fromRoute('entity.openai_batch_request.collection')
      ->toString());
  }

  /**
   * Construct the controller.
   */
  public function __construct(
    OpenAiBatchService $open_ai_batch_service,
    EntityTypeManagerInterface $entity_type_manager,
    MessengerInterface $messenger,
    RedirectDestinationInterface $redirect_destination,
    OpenAiBatchProcessorPluginManager $openai_batch_processor_manager,
  ) {
    $this->openAiBatchService = $open_ai_batch_service;
    $this->entityTypeManager = $entity_type_manager;
    $this->messenger = $messenger;
    $this->redirectDestination = $redirect_destination;
    $this->processorPluginManager = $openai_batch_processor_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('openai_batch.batch_service'),
      $container->get('entity_type.manager'),
      $container->get('messenger'),
      $container->get('redirect.destination'),
      $container->get('plugin.manager.openai_batch_processor'),
    );
  }

  /**
   * The OpenAI Batch service.
   *
   * @var \Drupal\openai_batch\OpenAiBatchService
   */
  private OpenAiBatchService $openAiBatchService;

  /**
   * The OpenAI Batch processor plugin manager.
   *
   * @var \Drupal\openai_batch\OpenAiBatchProcessorPluginManager
   */
  private OpenAiBatchProcessorPluginManager $processorPluginManager;

  public function viewJsonlFile(OpenAiBatchRequest $openai_batch_request, string $file_type) {
    if (!in_array($file_type, ['input', 'output'])) {
      $msg = "Invalid file type: " . $file_type. " . Allowed types are 'input' and 'output'.";
      $this->messenger->addError($msg);
      $canonical_url = $openai_batch_request->toUrl('canonical')->toString();
      return new RedirectResponse($canonical_url);

    }
    if ($file_type === 'input') {
      $file_id = $openai_batch_request->input_file_id->value;
      $filepath = $this->openAiBatchService->getInputFilesDirectory() . '/' . $file_id . '.jsonl';
    }
    else {
      $file_id = $openai_batch_request->output_file_id->value;
      $filepath = $this->openAiBatchService->getOutputFilesDirectory() . '/' . $file_id . '.jsonl';
    }

    if (!is_file($filepath)) {
      $msg = "File not found: " . $filepath;
      $this->messenger->addError($msg);
      $canonical_url = $openai_batch_request->toUrl('canonical')->toString();
      return new RedirectResponse($canonical_url);
    }

    $raw_jsonl_content = file_get_contents($filepath);
    $raw_jsonl_array = explode(PHP_EOL, trim($raw_jsonl_content));
    $jsonl_human_readable_array = [];
    foreach ($raw_jsonl_array as $jsonl) {
      $json_object = json_decode($jsonl, TRUE);
      $json_string = json_encode($json_object, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
      $jsonl_human_readable_array[] = $json_string;
    }
    $content = implode(PHP_EOL . PHP_EOL, $jsonl_human_readable_array);

    return [
      '#plain_text' => $content,
      '#prefix' => '<pre><code>',
      '#suffix' => '</code></pre>',
    ];
  }

  public function viewJsonlFileAccess(Route $route, RouteMatch $route_match, AccountInterface $account) {
    /** @var OpenAiBatchRequest $openai_batch_request */
    $openai_batch_request = $route_match->getParameter('openai_batch_request');
    if ($openai_batch_request && $openai_batch_request->access('view', $account)) {
      return AccessResult::allowed();
    }
    return AccessResult::neutral();
  }

}
