<?php

namespace Drupal\openai_batch\Drush\Commands;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\openai_batch\OpenAiBatchService;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Helper\Table;
use Symfony\Component\Console\Output\ConsoleOutput;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for OpenAI Batch module.
 */
final class OpenAiBatchCommands extends DrushCommands {

  /**
   * Lists the existing batches on OpenAI.
   */
  #[CLI\Command(name: 'openai_batch:list-batches', aliases: ['oaib-lb'])]
  #[CLI\Option(name: 'all', description: 'List all the remote batches, not only the ones with a corresponding local entity.')]
  public function listBatches(array $options = ['all' => '0']): void {
    $fetch_all = !empty($options['all']);
    $batches_array = $this->fetchRemoteBatches(fetch_all: $fetch_all);
    if (empty($batches_array)) {
      $this->logger()->warning('No batches found.');
      return;
    }
    $table_structure = [
      'id' => 'ID',
      'endpoint' => 'Endpoint',
      'createdAt' => 'Created at',
      'inputFileId' => 'Input file ID',
      'outputFileId' => 'Output file ID',
      'status' => 'Status',
    ];
    $this->massageItemsList($batches_array, $table_structure);
    $this->printTableWithBorders($batches_array, $table_structure);
  }

  /**
   * Fetches the remote batches from OpenAI.
   */
  private function fetchRemoteBatches($fetch_all = FALSE): array {
    $batchBuilder = new BatchBuilder();
    $batchBuilder->addOperation('\Drupal\openai_batch\Batch\OpenAiBatchOperations::fetchRemoteBatchesOperation', [
      $fetch_all,
    ]);
    $batchBuilder
      ->setTitle($this->t('Fetching batches'))
      ->setFinishCallback('\Drupal\openai_batch\Batch\OpenAiBatchOperations::refreshBatchesOperationFinished')
      ->setErrorMessage($this->t('Batch has encountered an error'));
    batch_set($batchBuilder->toArray());
    $result = drush_backend_batch_process();
    return $result[0];
  }

  /**
   * Retrieves a batch from OpenAI.
   */
  #[CLI\Command(name: 'openai_batch:retrieve-batch', aliases: [
    'oaib-rb',
    'oaib-gb',
    'openai_batch:get-batch',
  ])]
  #[CLI\Argument(name: 'batch_id', description: 'The batch ID.')]
  public function retrieveBatch($batch_id): void {
    try {
      $batch = $this->batchService->getClient()->batches()->retrieve($batch_id);
    }
    catch (\Exception $e) {
      $this->logger()->error($e->getMessage());
      return;
    }
    $batch_array = $batch->toArray();
    foreach ($batch_array as $key => $value) {
      if (str_ends_with($key, '_at') && is_int($value)) {
        $batch_array[$key] = date('d/m/Y H:i:s', $value);
      }
    }
    $this->printAssocArrayAsTable($batch_array);
  }

  /**
   * Cancels a batch from OpenAI.
   */
  #[CLI\Command(name: 'openai_batch:cancel-batch', aliases: ['oaib-cb'])]
  #[CLI\Argument(name: 'batch_id', description: 'The batch ID.')]
  public function cancelBatch($batch_id): void {
    try {
      $response = $this->batchService->getClient()
        ->batches()
        ->cancel($batch_id);
    }
    catch (\Exception $e) {
      $this->logger()->error($e->getMessage());
      return;
    }
    if ($response->status !== 'cancelling' || !is_int($response->cancellingAt)) {
      $this->logger()->error("The batch $batch_id could not be canceled.");
    }
    else {
      $this->logger()->success("The batch $batch_id has been canceled.");
    }
  }

  /**
   * Lists the files uploaded to OpenAI.
   */
  #[CLI\Command(name: 'openai_batch:list-files', aliases: ['oaib-lf'])]
  public function listFiles(): void {
    /*
     * @TODO
     * According to the API
     * (https://platform.openai.com/docs/api-reference/files/list) the files
     * listing should be paginated. But the latest version of the PHP client
     * (v0.10.2) does not support pagination:
     * https://github.com/openai-php/client?tab=readme-ov-file#files-resource
     * When the pagination will get developed on the client, the files fetching
     * must be done with drupal's Batch API, like the batches fetching.
     * Until then, we just call the client's method directly.
     * Update: Open pull request pending:
     * https://github.com/openai-php/client/pull/501
     */
    $files_array = $this->batchService->getFiles();
    if (empty($files_array)) {
      $this->logger()->warning('No files found.');
      return;
    }

    $table_structure = [
      'id' => 'ID',
      'created_at' => 'Created At',
      'filename' => 'Filename',
      'purpose' => 'Purpose',
      'status' => 'Status',
    ];
    $this->massageItemsList($files_array, $table_structure);
    $this->printTableWithBorders($files_array, $table_structure);
  }

  /**
   * Retrieves a file from OpenAI.
   */
  #[CLI\Command(name: 'openai_batch:retrieve-file', aliases: [
    'oaib-rf',
    'oaib-gf',
    'openai_batch:get-file',
  ])]
  #[CLI\Argument(name: 'file_id', description: 'The file ID.')]
  public function retrieveFile($file_id): void {
    try {
      $file = $this->batchService->getClient()->files()->retrieve($file_id);
    }
    catch (\Exception $e) {
      $this->logger()->error($e->getMessage());
      return;
    }
    $file_array = $file->toArray();
    foreach ($file_array as $key => $value) {
      if (str_ends_with($key, '_at') && is_int($value)) {
        $file_array[$key] = date('d/m/Y H:i:s', $value);
      }
    }
    $this->printAssocArrayAsTable($file_array);
  }

  /**
   * Deletes a file from OpenAI.
   */
  #[CLI\Command(name: 'openai_batch:delete-file', aliases: ['oaib-df'])]
  #[CLI\Argument(name: 'file_id', description: 'The file ID.')]
  public function deleteFile($file_id): void {
    try {
      $response = $this->batchService->getClient()->files()->delete($file_id);
    }
    catch (\Exception $e) {
      $this->logger()->error($e->getMessage());
      return;
    }
    if (!$response->deleted) {
      $this->logger()->error("The file $file_id could not be deleted.");
    }
    else {
      $this->logger()->success("The file $file_id has been deleted.");
    }
  }

  /**
   * Massages the items list to remove unnecessary keys and format dates.
   */
  private function massageItemsList(array &$items, array $table_structure): void {
    foreach ($items as $i => $file) {
      foreach ($file as $key => $value) {
        if (!array_key_exists($key, $table_structure)) {
          unset($items[$i][$key]);
          continue;
        }
        if ((str_ends_with($key, 'At') || str_ends_with($key, '_at')) && (is_int($value))) {
          $items[$i][$key] = date('d/m/Y H:i:s', $value);
        }
      }
    }
  }

  /**
   * Print an associative array in a human-readable table format.
   */
  private function printAssocArrayAsTable($array): void {
    $output = new ConsoleOutput();

    $table = new Table($output);

    $table->setHeaders(['Key', 'Value']);

    foreach ($array as $key => $value) {
      if (is_array($value)) {
        $value = json_encode($value, JSON_PRETTY_PRINT);
      }
      $table->addRow([$key, $value]);
    }

    $table->setStyle('box');
    $table->render();
  }

  /**
   * Print an array as a table with borders.
   */
  private function printTableWithBorders(array $data, array $structure): void {
    $output = new ConsoleOutput();

    $table = new Table($output);

    $table->setHeaders(array_values($structure));

    foreach ($data as $row) {
      $table->addRow(array_map(fn($key) => $row[$key], array_keys($structure)));
    }

    $table->setStyle('box');
    $table->render();
  }

  public function __construct(
    private readonly OpenAiBatchService $batchService,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('openai_batch.batch_service'),
    );
  }

  use StringTranslationTrait;

}
