<?php

namespace Drupal\openai_batch\Entity;

use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the OpenAI Batch Request entity class.
 *
 * @ContentEntityType(
 *   id = "openai_batch_request",
 *   label = @Translation("OpenAI Batch Request"),
 *   label_collection = @Translation("OpenAI Batch Requests"),
 *   label_singular = @Translation("OpenAI Batch Request"),
 *   label_plural = @Translation("openai batch requests"),
 *   label_count = @PluralTranslation(
 *     singular = "@count openai batch requests",
 *     plural = "@count openai batch requests",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\openai_batch\OpenAiBatchRequestListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "access" = "Drupal\openai_batch\OpenAiBatchRequestAccessControlHandler",
 *     "form" = {
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *       "revision" = \Drupal\Core\Entity\Routing\RevisionHtmlRouteProvider::class,
 *     },
 *   },
 *   base_table = "openai_batch_request",
 *   revision_table = "openai_batch_request_revision",
 *   show_revision_ui = TRUE,
 *   admin_permission = "administer openai_batch_request",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "revision_id",
 *     "label" = "id",
 *     "uuid" = "uuid",
 *     "owner" = "uid",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_uid",
 *     "revision_created" = "revision_timestamp",
 *     "revision_log_message" = "revision_log",
 *   },
 *   links = {
 *     "collection" = "/admin/structure/openai-batch-request/list",
 *     "canonical" = "/openai-batch-request/{openai_batch_request}",
 *     "delete-form" = "/openai-batch-request/{openai_batch_request}/delete",
 *     "delete-multiple-form" = "/admin/structure/openai-batch-request/delete-multiple",
 *     "revision" = "/openai-batch-request/{openai_batch_request}/revision/{openai_batch_request_revision}/view",
 *     "version-history" = "/openai-batch-request/{openai_batch_request}/revisions",
 *   },
 *   field_ui_base_route = "entity.openai_batch_request.settings",
 * )
 */
final class OpenAiBatchRequest extends RevisionableContentEntityBase {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      $this->setOwnerId(0);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    // The ID must be a string, so here we override the default integer field.
    $fields[$entity_type->getKey('id')] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('ID'))
      ->setReadOnly(TRUE);

    $fields['downloaded'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Result downloaded'))
      ->setDescription(t('Flags the result file as downloaded or not.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue(FALSE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'boolean',
        'weight' => 1,
        'settings' => [
          'format' => 'yes-no',
        ],
      ]);

    $fields['processed'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Result processed'))
      ->setDescription(t('Flags the result as processed or not.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue(FALSE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'boolean',
        'weight' => 2,
        'settings' => [
          'format' => 'yes-no',
        ],
      ]);

    $fields['autosync'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Sync automatically'))
      ->setDescription(t('Flags whether the entity must be refreshed by cron or not.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue(FALSE)
      ->setSetting('on_label', 'Enabled')
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'boolean',
        'weight' => 4,
        'settings' => [
          'format' => 'yes-no',
        ],
      ]);

    $fields['processor'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Processor'))
      ->setDescription(t('The OpenAI Batch Processor plugin to process the results with.'))
      ->setRequired(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
        'weight' => 6,
      ]);

    $fields['status'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Status'))
      ->setRevisionable(TRUE)
      ->setDescription(t('The batch status.'))
      ->setRequired(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
        'weight' => 7,
      ]);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Owner'))
      ->setDescription(t('The entity owner.'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(self::class . '::getDefaultEntityOwner')
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'entity_reference_label',
        'label' => 'inline',
        'weight' => 9,
        'settings' => [
          'link' => FALSE,
        ],
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the OpenAI Batch Request entity was created.'))
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'timestamp',
        'label' => 'inline',
        'weight' => 10,
        'settings' => [
          'date_format' => 'custom',
          'custom_date_format' => 'd/m/Y H:i:s',
          'tooltip' => [
            'date_format' => 'long',
          ],
        ],
      ]);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the OpenAI Batch Request entity was last edited.'));

    $fields['input_file_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Input file ID'))
      ->setDescription(t('The input file ID.'))
      ->setRequired(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
        'weight' => 11,
      ]);

    $fields['output_file_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Output file ID'))
      ->setDescription(t('The output file ID.'))
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
        'weight' => 12,
      ]);

    $fields['created_at'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Created at (remote)'))
      ->setDescription(t('The time that the OpenAI Batch was created.'))
      ->setRevisionable(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'timestamp',
        'label' => 'inline',
        'weight' => 13,
        'settings' => [
          'date_format' => 'custom',
          'custom_date_format' => 'd/m/Y H:i:s',
          'tooltip' => [
            'date_format' => 'long',
          ],
        ],
      ]);

    $fields['in_progress_at'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('In progress at (remote)'))
      ->setDescription(t('The time that the OpenAI Batch started progress.'))
      ->setRevisionable(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'timestamp',
        'label' => 'inline',
        'weight' => 14,
        'settings' => [
          'date_format' => 'custom',
          'custom_date_format' => 'd/m/Y H:i:s',
          'tooltip' => [
            'date_format' => 'long',
          ],
        ],
      ]);

    $fields['expires_at'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Expires at (remote)'))
      ->setDescription(t('The time that the OpenAI Batch is set to expire.'))
      ->setRevisionable(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'timestamp',
        'label' => 'inline',
        'weight' => 15,
        'settings' => [
          'date_format' => 'custom',
          'custom_date_format' => 'd/m/Y H:i:s',
          'tooltip' => [
            'date_format' => 'long',
          ],
        ],
      ]);

    $fields['finalizing_at'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Finalizing at (remote)'))
      ->setDescription(t('The time that the OpenAI Batch started finalizing.'))
      ->setRevisionable(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'timestamp',
        'label' => 'inline',
        'weight' => 16,
        'settings' => [
          'date_format' => 'custom',
          'custom_date_format' => 'd/m/Y H:i:s',
          'tooltip' => [
            'date_format' => 'long',
          ],
        ],
      ]);

    $fields['completed_at'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Completed at (remote)'))
      ->setDescription(t('The time that the OpenAI Batch completed.'))
      ->setRevisionable(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'timestamp',
        'label' => 'inline',
        'weight' => 17,
        'settings' => [
          'date_format' => 'custom',
          'custom_date_format' => 'd/m/Y H:i:s',
          'tooltip' => [
            'date_format' => 'long',
          ],
        ],
      ]);

    $fields['failed_at'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Failed at (remote)'))
      ->setDescription(t('The time that the OpenAI Batch failed.'))
      ->setRevisionable(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'timestamp',
        'label' => 'inline',
        'weight' => 18,
        'settings' => [
          'date_format' => 'custom',
          'custom_date_format' => 'd/m/Y H:i:s',
          'tooltip' => [
            'date_format' => 'long',
          ],
        ],
      ]);

    $fields['expired_at'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Expired at (remote)'))
      ->setDescription(t('The time that the OpenAI Batch expired.'))
      ->setRevisionable(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'timestamp',
        'label' => 'inline',
        'weight' => 19,
        'settings' => [
          'date_format' => 'custom',
          'custom_date_format' => 'd/m/Y H:i:s',
          'tooltip' => [
            'date_format' => 'long',
          ],
        ],
      ]);

    $fields['cancelling_at'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Cancelling at (remote)'))
      ->setDescription(t('The time that the OpenAI Batch was requested to cancel.'))
      ->setRevisionable(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'timestamp',
        'label' => 'inline',
        'weight' => 20,
        'settings' => [
          'date_format' => 'custom',
          'custom_date_format' => 'd/m/Y H:i:s',
          'tooltip' => [
            'date_format' => 'long',
          ],
        ],
      ]);

    $fields['cancelled_at'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Cancelled at (remote)'))
      ->setDescription(t('The time that the OpenAI Batch was cancelled.'))
      ->setRevisionable(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'timestamp',
        'label' => 'inline',
        'weight' => 21,
        'settings' => [
          'date_format' => 'custom',
          'custom_date_format' => 'd/m/Y H:i:s',
          'tooltip' => [
            'date_format' => 'long',
          ],
        ],
      ]);

    $fields['endpoint'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Endpoint'))
      ->setDescription(t('The endpoint.'))
      ->setRequired(TRUE)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
        'weight' => 23,
      ]);

    $fields['metadata'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Metadata'))
      ->setDescription(t('Any metadata related to the entity.'))
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'basic_string',
        'weight' => 24,
      ]);

    return $fields;
  }

}
