<?php

namespace Drupal\openai_batch;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\openai_batch\Entity\OpenAiBatchRequest;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for openai_batch_processor plugins.
 */
abstract class OpenAiBatchProcessorPluginBase extends PluginBase implements OpenAiBatchProcessorInterface, ContainerFactoryPluginInterface {

  /**
   * Returns the batch builder for processing the batch.
   *
   * @param \Drupal\openai_batch\Entity\OpenAiBatchRequest $openai_batch_request
   *   The OpenAI batch request entity.
   *
   * @return \Drupal\Core\Batch\BatchBuilder
   *   The batch builder.
   */
  public function getProcessBatch(OpenAiBatchRequest $openai_batch_request): BatchBuilder {
    $output_file = $openai_batch_request->output_file_id->value;
    $output_filepath = $this->openAiBatchService->getOutputFilesDirectory() . '/' . $output_file . '.jsonl';
    if (!is_file($output_filepath)) {
      \Drupal::logger('openai_batch')->error('Output file not found: @output_filepath', ['@output_filepath' => $output_filepath]);
      throw new \Exception("Output file not found: $output_filepath");
    }
    $content = file_get_contents($output_filepath);
    if (!$content) {
      \Drupal::logger('openai_batch')->error('Failed to read output file: @output_filepath', ['@output_filepath' => $output_filepath]);
      throw new \Exception("Failed to read output file: $output_filepath");
    }
    $content_array = explode(PHP_EOL, $content);

    $batchBuilder = new BatchBuilder();
    foreach ($content_array as $json_string) {
      if (empty($json_string)) {
        continue;
      }
      $json = json_decode($json_string, TRUE);
      $custom_id = $json["custom_id"];
      [, $entity_type_id, $entity_id] = explode('::', $custom_id);

      $result_string = $json["response"]["body"]["choices"][0]["message"]["content"];
      $batchBuilder->addOperation([static::class, 'processResultItemOperation'], [
        $openai_batch_request,
        $entity_type_id,
        $entity_id,
        $result_string,
      ]);
    }
    $batchBuilder
      ->setTitle($this->t('Processing: @label', ['@label' => $this->label()]))
      ->setFinishCallback([$this, 'finished'])
      ->setErrorMessage($this->t('Batch has encountered an error'));
    return $batchBuilder;
  }

  /**
   * Batch operation to mark an entity as processed.
   *
   * @param \Drupal\openai_batch\Entity\OpenAiBatchRequest $openai_batch_request
   *   The OpenAI batch request entity.
   * @param array $context
   *   The batch context.
   *
   * @return void
   *   Nothing to return.
   */
  public function markAsProcessedOperation(OpenAiBatchRequest $openai_batch_request, &$context): void {
    $new_revision = $this->entityTypeManager->getStorage('openai_batch_request')
      ->createRevision($openai_batch_request);
    $new_revision->processed = TRUE;
    $new_revision->save();
  }

  /**
   * Finish callback for the batch.
   */
  public function finished($success, array $results, array $operations) {
    // @todo what else to do here?
  }

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    // Cast the label to a string since it is a TranslatableMarkup object.
    return (string) $this->pluginDefinition['label'];
  }

  public function __construct($configuration, $plugin_id, $plugin_definition, OpenAiBatchService $openai_batch_service, EntityTypeManagerInterface $entityTypeManager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->openAiBatchService = $openai_batch_service;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration, $plugin_id, $plugin_definition,
      $container->get('openai_batch.batch_service'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * The OpenAI Batch service.
   *
   * @var \Drupal\openai_batch\OpenAiBatchService
   */
  private OpenAiBatchService $openAiBatchService;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The VBO configuration.
   *
   * @var array
   */
  private array $vboConfiguration = [];

  /**
   * Set the VBO configuration.
   */
  final public function setVboConfiguration(array $vbo_configuration): void {
    $this->vboConfiguration = $vbo_configuration;
  }

  /**
   * Get the VBO configuration.
   */
  final public function getVboConfiguration(): array {
    return $this->vboConfiguration;
  }

  use StringTranslationTrait;

  use DependencySerializationTrait;

}
