<?php

namespace Drupal\openai_batch\Plugin\OpenAiBatchProcessor;

use Drupal\ai\Plugin\ProviderProxy;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\NodeType;
use Drupal\openai_batch\Attribute\OpenAiBatchProcessor;
use Drupal\openai_batch\DTO\OpenAiBatchRequestItemConfig;
use Drupal\openai_batch\Entity\OpenAiBatchRequest;
use Drupal\openai_batch\OpenAiBatchProcessorPluginBase;
use Drupal\openai_batch\Plugin\Action\OpenAiBatchVBOActionInterface;

/**
 * Write summary on text fields that support it.
 */
#[OpenAiBatchProcessor(
  id: 'write_summary',
  label: new TranslatableMarkup('Write summary'),
  description: new TranslatableMarkup('Write summary on text fields that support it.'),
)]
class WriteSummary extends OpenAiBatchProcessorPluginBase implements OpenAiBatchVBOActionInterface {

  /**
   * {@inheritdoc}
   */
  public static function processResultItemOperation(OpenAiBatchRequest $openai_batch_request, string $entity_type_id, string $entity_id, string $result_string, &$context): void {
    $metadata = $openai_batch_request->metadata->value;
    $metadata = json_decode($metadata, TRUE);
    $field_name = $metadata['vbo_configuration']['field'];
    $entity = \Drupal::entityTypeManager()->getStorage($entity_type_id)->load($entity_id);
    $field = $entity->get($field_name);
    $field->summary = $result_string;
    $entity->save();
  }

  /**
   * {@inheritdoc}
   */
  public function getOpenAiBatchCommandForEntity(string $entity_type, string $entity_id, OpenAiBatchRequestItemConfig $request_item_config, mixed $context = NULL): array {
    $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
    $field_name = $this->getVboConfiguration()['field'];
    $html = '';
    if (!$entity->hasField($field_name)) {
      throw new \Exception('The field ' . $field_name . ' does not exist on the node ' . $entity_id);
    }
    $field = $entity->get($field_name);
    $label = $field->getFieldDefinition()->getLabel();
    $html .= '<h1>' . $label . '</h1>' . PHP_EOL;
    foreach ($field as $item) {
      $value = $item->value;
      $html .= '<div>' . $value . '</div>' . PHP_EOL;
    }

    $system_prompt = $request_item_config->systemPrompt;
    $user_prompt = $request_item_config->userPrompt;

    $replace_with = PHP_EOL . '"""' . PHP_EOL . $html . '"""' . PHP_EOL;
    $user_prompt = str_replace(static::NODE_FIELD_TOKEN, $replace_with, $user_prompt);

    $batch_command = [
      'custom_id' => 'write_summary::' . $entity_type . '::' . $entity_id,
      'method' => 'POST',
      'url' => '/v1/chat/completions',
      'body' => [
        'model' => $request_item_config->model,
        'messages' => [
          [
            'role' => 'system',
            'content' => $system_prompt,
          ],
          [
            'role' => 'user',
            'content' => $user_prompt,
          ],
        ],
        'max_tokens' => $request_item_config->maxTokens,
        'temperature' => $request_item_config->temperature,
        'frequency_penalty' => $request_item_config->frequencyPenalty,
        'presence_penalty' => $request_item_config->presencePenalty,
        'top_p' => $request_item_config->topP,
      ],
    ];
    return $batch_command;
  }

  /**
   * {@inheritdoc}
   */
  public function buildPreConfigurationForm(array $element, array $values, FormStateInterface $form_state): array {
    $bundles = [];
    $all_node_types = NodeType::loadMultiple();
    foreach ($all_node_types as $node_type) {
      $content_type_id = $node_type->id();
      $fields = $this->entityTypeManager->getStorage('field_config')->loadByProperties([
        'entity_type' => 'node',
        'bundle' => $content_type_id,
      ]);
      foreach ($fields as $field) {
        /** @var \Drupal\field\Entity\FieldConfig $field */
        $field_storage = FieldStorageConfig::loadByName('node', $field->getName());
        if ($field_storage && $field_storage->getType() === 'text_with_summary') {
          if (!isset($bundles[$content_type_id])) {
            $bundles[$content_type_id] = [];
          }
          $bundles[$content_type_id][] = $field->getName();
        }
      }
    }

    $options = [];
    foreach ($all_node_types as $node_type) {
      $node_type_id = $node_type->id();
      if (!array_key_exists($node_type_id, $bundles)) {
        continue;
      }
      $options[$node_type->id()] = $node_type->label();
    }
    $element['bundle'] = [
      '#type' => 'select',
      '#title' => $this->t('Content type'),
      '#description' => $this->t('Select which content type you want to process. You must add a relevant filter in your view, to ensure that only nodes of this type will be in the results.'),
      '#options' => $options,
      '#default_value' => $values["bundle"] ?? '',
      '#required' => TRUE,
    ];
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state, array $vbo_context, ProviderProxy $llm_instance): array {
    $bundle = $vbo_context["preconfiguration"]["bundle"];
    $fields = $this->entityTypeManager->getStorage('field_config')->loadByProperties([
      'entity_type' => 'node',
      'bundle' => $bundle,
    ]);
    $text_with_summary_fields = [];
    foreach ($fields as $field) {
      /** @var \Drupal\field\Entity\FieldConfig $field */
      $field_storage = FieldStorageConfig::loadByName('node', $field->getName());
      if ($field_storage && $field_storage->getType() === 'text_with_summary') {
        $text_with_summary_fields[$field->getName()] = $field->getLabel();
      }
    }
    $form['field'] = [
      '#type' => 'select',
      '#title' => 'Field',
      '#description' => $this->t('Select the field to write the summary on.'),
      '#options' => $text_with_summary_fields,
      '#required' => TRUE,
    ];

    $system_prompt = "You are a helpful assistant that processes text.";
    $user_prompt = 'Create a detailed summary of the following text in less than 130 words using the same language as the following text:' . PHP_EOL . PHP_EOL . static::NODE_FIELD_TOKEN;

    $form['options'] = [
      '#type' => 'details',
      '#open' => FALSE,
      '#title' => 'Options',
    ];
    $form['options']['system_prompt'] = [
      '#type' => 'textarea',
      '#title' => "System prompt",
      '#default_value' => $system_prompt,
    ];
    $form['options']['user_prompt'] = [
      '#type' => 'textarea',
      '#title' => "User prompt",
      '#default_value' => $user_prompt,
      // @todo description, with instructions about the NODE_FIELD_TOKEN.
    ];
    $models = $llm_instance->getConfiguredModels("chat");
    $form['options']['model'] = [
      '#type' => 'select',
      '#title' => 'Model',
      '#options' => $models,
      '#default_value' => 'gpt-4o',
    ];

    $available_conf = $llm_instance->getAvailableConfiguration("chat", "gpt-4o");
    $form['options']['areyousure'] = [
      '#type' => 'details',
      '#title' => 'Advanced options',
      '#open' => FALSE,
    ];
    foreach ($available_conf as $key => $definition) {
      $set_key = 'configuration_' . $key;
      $form['options']['areyousure'][$set_key]['#type'] = $this->mapSchemaTypeToFormType($definition);
      $form['options']['areyousure'][$set_key]['#required'] = $definition['required'] ?? FALSE;
      $form['options']['areyousure'][$set_key]['#title'] = $definition['label'] ?? $key;
      $form['options']['areyousure'][$set_key]['#description'] = $definition['description'] ?? '';
      $form['options']['areyousure'][$set_key]['#default_value'] = $definition['default'] ?? NULL;
      if ($key === 'temperature') {
        $form['options']['areyousure'][$set_key]['#default_value'] = 0;
      }

      if (isset($definition['constraints'])) {
        foreach ($definition['constraints'] as $form_key => $value) {
          if ($form_key == 'options') {
            $options = array_combine($value, $value);
            if (empty($definition['required'])) {
              $options = ['' => 'Select an option'] + $options;
            }
            $form['options']['areyousure'][$set_key]['#options'] = $options;
            continue;
          }
          $form['options']['areyousure'][$set_key]['#' . $form_key] = $value;
        }
      }
    }

    return $form;
  }

  public const NODE_FIELD_TOKEN = "[NODE_FIELD_CONTENT]";

  /**
   * Maps schema data types to form element types.
   *
   * @param array $definition
   *   Data type of a configuration value.
   *
   * @return string
   *   Type of widget.
   *
   * @todo this is directly copied from the AI module. Fix this!
   */
  private function mapSchemaTypeToFormType(array $definition): string {
    // Check first for settings constraints.
    if (isset($definition['constraints']['options'])) {
      return 'select';
    }
    return match ($definition['type']) {
      'boolean' => 'checkbox',
      'string_long' => 'textarea',
      default => 'textfield',
    };
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $input = $form_state->getUserInput();
    if (!str_contains($input['user_prompt'], static::NODE_FIELD_TOKEN)) {
      $form_state->setErrorByName('user_prompt', $this->t('The token %node_fields_token must be included in the user prompt.', [
        '%node_fields_token' => static::NODE_FIELD_TOKEN,
      ]));
    }
  }

}
