<?php

namespace Drupal\openai_batch\Plugin\QueueWorker;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\openai_batch\OpenAiBatchService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Queue worker for downloading an OpenAI Batch result.
 *
 * @QueueWorker(
 *   id = "openai_batch_download",
 *   title = @Translation("OpenAI Batch Download Queue Worker"),
 * )
 */
class OpenAiBatchDownloadWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    $storage = $this->entityTypeManager->getStorage('openai_batch_request');
    $batch_id = $data['batch_id'];
    $openai_batch_request = $storage->load($batch_id);
    $client = $this->batchService->getClient();
    $batch_response = $client->batches()->retrieve($batch_id);
    $batch_entity_revision = $storage->createRevision($openai_batch_request);
    $file = $client->files()->download($batch_response->outputFileId);
    $this->batchService->saveOutputFile($batch_response->outputFileId, $file);
    $batch_entity_revision->set('downloaded', TRUE);
    $batch_entity_revision->save();
  }

  public function __construct(array $configuration, $plugin_id, array $plugin_definition, OpenAiBatchService $openai_batch_service, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->batchService = $openai_batch_service;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('openai_batch.batch_service'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * The OpenAI Batch service.
   *
   * @var \Drupal\openai_batch\OpenAiBatchService
   */
  private OpenAiBatchService $batchService;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private EntityTypeManagerInterface $entityTypeManager;

}
