<?php

namespace Drupal\openai_batch\Plugin\QueueWorker;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\openai_batch\OpenAiBatchService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Queue worker for refreshing an OpenAI Batch result entity.
 *
 * @QueueWorker(
 *   id = "openai_batch_refresh",
 *   title = @Translation("OpenAI Batch Refresh Queue Worker"),
 * )
 */
class OpenAiBatchRefreshWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    $batch_id = $data['batch_id'];
    $openai_batch_request = $this->entityTypeManager->getStorage('openai_batch_request')->load($batch_id);
    $batch = $this->batchService->getClient()->batches()->retrieve($batch_id);
    $new_revision = $this->batchService->updateOpenAiBatchRequestEntity(batch_response: $batch, batch_entity: $openai_batch_request, save: TRUE);
  }

  public function __construct(array $configuration, $plugin_id, array $plugin_definition, OpenAiBatchService $openai_batch_service, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->batchService = $openai_batch_service;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('openai_batch.batch_service'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * The OpenAI Batch service.
   *
   * @var \Drupal\openai_batch\OpenAiBatchService
   */
  private OpenAiBatchService $batchService;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private EntityTypeManagerInterface $entityTypeManager;

}
