<?php

namespace Drupal\openai_batch;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Defines the access control handler for the openai batch request entity type.
 *
 * phpcs:disable Drupal.Arrays.Array.LongLineDeclaration
 *
 * @see https://www.drupal.org/project/coder/issues/3185082
 */
final class OpenAiBatchRequestAccessControlHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    // Get the entity owner.
    $owner = $entity->get('uid')->entity;

    // Check permissions based on the operation.
    switch ($operation) {
      case 'view':
        if ($account->hasPermission('administer openai_batch_request')) {
          return AccessResult::allowed();
        }
        if ($account->hasPermission('view all openai_batch_request')) {
          return AccessResult::allowed();
        }
        if ($account->hasPermission('view own openai_batch_request') && $owner && $owner->id() == $account->id()) {
          return AccessResult::allowed();
        }
        break;

      case 'delete':
        if ($account->hasPermission('administer openai_batch_request')) {
          return AccessResult::allowed();
        }
        if ($account->hasPermission('delete all openai_batch_request')) {
          return AccessResult::allowed();
        }
        if ($account->hasPermission('delete own openai_batch_request') && $owner && $owner->id() == $account->id()) {
          return AccessResult::allowed();
        }
        break;

      case 'view revision':
        if ($account->hasPermission('administer openai_batch_request')) {
          return AccessResult::allowed();
        }
        if ($account->hasPermission('view all openai_batch_request revisions')) {
          return AccessResult::allowed();
        }
        if ($account->hasPermission('view own openai_batch_request revisions') && $owner && $owner->id() == $account->id()) {
          return AccessResult::allowed();
        }
        break;

      case 'refresh':
        if ($account->hasPermission('administer openai_batch_request')) {
          return AccessResult::allowed();
        }
        if ($account->hasPermission('refresh all openai_batch_request')) {
          return AccessResult::allowed();
        }
        if ($account->hasPermission('refresh own openai_batch_request') && $owner && $owner->id() == $account->id()) {
          return AccessResult::allowed();
        }
        break;

      case 'download':
        if ($account->hasPermission('administer openai_batch_request')) {
          return AccessResult::allowed();
        }
        if ($account->hasPermission('download all openai_batch_request')) {
          return AccessResult::allowed();
        }
        if ($account->hasPermission('download own openai_batch_request') && $owner && $owner->id() == $account->id()) {
          return AccessResult::allowed();
        }
        break;

      case 'process':
        if ($account->hasPermission('administer openai_batch_request')) {
          return AccessResult::allowed();
        }
        if ($account->hasPermission('process all openai_batch_request')) {
          return AccessResult::allowed();
        }
        if ($account->hasPermission('process own openai_batch_request') && $owner && $owner->id() == $account->id()) {
          return AccessResult::allowed();
        }
        break;
    }

    // By default, deny access.
    return AccessResult::neutral();
  }

}
