<?php

namespace Drupal\openai_batch;

use Drupal\ai\AiProviderPluginManager;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\Exception\FileException;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\file\FileRepositoryInterface;
use Drupal\openai_batch\Entity\OpenAiBatchRequest;
use OpenAI\Client;
use OpenAI\Responses\Batches\BatchResponse;

/**
 * Service class for OpenAI Batch.
 */
final class OpenAiBatchService {

  /**
   * Ges all files from the OpenAI API.
   */
  public function getFiles(): array {
    $files_array = [];
    $response = $this->client->files()->list();
    foreach ($response->data as $result) {
      $file = $result->toArray();
      $files_array[] = $file;
    }
    return $files_array;
  }

  /**
   * Get the directory to store the input files.
   */
  public function getInputFilesDirectory(): string {
    $main_dir = $this->config->get('system.file')->get('default_scheme') . "://openai_batch";
    $dir = $this->config->get('system.file')->get('default_scheme') . "://openai_batch/input_files";
    $this->prepareDirectory($main_dir);
    $this->prepareDirectory($dir);
    return $dir;
  }

  /**
   * Get the directory to store the output (result) files.
   */
  public function getOutputFilesDirectory(): string {
    $main_dir = $this->config->get('system.file')->get('default_scheme') . "://openai_batch";
    $dir = $this->config->get('system.file')->get('default_scheme') . "://openai_batch/output_files";
    $this->prepareDirectory($main_dir);
    $this->prepareDirectory($dir);
    return $dir;
  }

  /**
   * Save a jsonl file.
   *
   * @param string $type
   *   The type of file to save.
   * @param string $file_id
   *   The ID of the file.
   * @param string $content
   *   The content of the file.
   */
  public function saveJsonlFile(string $type, string $file_id, string $content) {
    $sub_dir = match ($type) {
      'input' => $this->getInputFilesDirectory(),
      'output' => $this->getOutputFilesDirectory(),
      default => throw new \Exception("Invalid batch file type: $type"),
    };
    $filepath = $sub_dir . "/$file_id.jsonl";
    $this->fileRepository->writeData($content, $filepath);
  }

  /**
   * Save an input file.
   *
   * @param string $file_id
   *   The ID of the file.
   * @param string $content
   *   The content of the file.
   */
  public function saveInputFile($file_id, $content) {
    $main_dir = $this->config->get('system.file')->get('default_scheme') . "://openai_batch";
    $input_dir = $this->config->get('system.file')->get('default_scheme') . "://openai_batch/input_files";
    $this->prepareDirectory($main_dir);
    $this->prepareDirectory($input_dir);
    $filepath = $input_dir . "/$file_id.jsonl";
    $this->fileRepository->writeData($content, $filepath);
  }

  /**
   * Save an output file.
   *
   * @param string $file_id
   *   The ID of the file.
   * @param string $content
   *   The content of the file.
   */
  public function saveOutputFile($file_id, $content) {
    $main_dir = $this->config->get('system.file')->get('default_scheme') . "://openai_batch";
    $output_dir = $this->config->get('system.file')->get('default_scheme') . "://openai_batch/output_files";
    $this->prepareDirectory($main_dir);
    $this->prepareDirectory($output_dir);
    $filepath = $output_dir . "/$file_id.jsonl";
    $this->fileRepository->writeData($content, $filepath, FileExists::Replace);
  }

  /**
   * Wrapper function to refresh the batches status using Drupal's Batch API.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function refreshBatchesStatusFormSubmitHandler(array &$form, FormStateInterface $form_state) {
    $batchBuilder = $this->getRefreshBatchesBatchBuilder();
    batch_set($batchBuilder->toArray());
  }

  /**
   * Get the batch builder for refreshing the batches.
   *
   * @return \Drupal\Core\Batch\BatchBuilder
   *   The batch builder.
   */
  public function getRefreshBatchesBatchBuilder(): BatchBuilder {
    $batchBuilder = new BatchBuilder();
    $batchBuilder->addOperation('\Drupal\openai_batch\Batch\OpenAiBatchOperations::fetchRemoteBatchesOperation', [
      $fetch_all = FALSE,
    ]);
    $batchBuilder->addOperation('\Drupal\openai_batch\Batch\OpenAiBatchOperations::updateLocalEntitiesOperation', [
      $create_missing = FALSE,
    ]);
    $batchBuilder
      ->setTitle($this->t('Fetching batches'))
      ->setFinishCallback('\Drupal\openai_batch\Batch\OpenAiBatchOperations::refreshBatchesOperationFinished')
      ->setErrorMessage($this->t('Batch has encountered an error'));
    return $batchBuilder;
  }

  /**
   * Update the OpenAI Batch Request entity with the remote data, if different.
   *
   * @param \OpenAI\Responses\Batches\BatchResponse $batch_response
   *   The response from the OpenAI API.
   * @param \Drupal\openai_batch\Entity\OpenAiBatchRequest $batch_entity
   *   The entity to update.
   * @param bool $save
   *   Whether to save the entity.
   *
   * @return \Drupal\openai_batch\Entity\OpenAiBatchRequest|null
   *   The updated entity, or NULL if no changes were made.
   */
  public function updateOpenAiBatchRequestEntity(BatchResponse $batch_response, OpenAiBatchRequest $batch_entity, bool $save = TRUE): ? OpenAiBatchRequest {
    $remote_values = [
      'status' => (string) $batch_response->status,
      'output_file_id' => (string) $batch_response->outputFileId,
      'in_progress_at' => (string) $batch_response->inProgressAt,
      'expires_at' => (string) $batch_response->expiresAt,
      'finalizing_at' => (string) $batch_response->finalizingAt,
      'completed_at' => (string) $batch_response->completedAt,
      'failed_at' => (string) $batch_response->failedAt,
      'expired_at' => (string) $batch_response->expiredAt,
      'cancelling_at' => (string) $batch_response->cancellingAt,
      'cancelled_at' => (string) $batch_response->cancelledAt,
    ];
    $local_values = [
      'status' => (string) $batch_entity->get('status')->value,
      'output_file_id' => (string) $batch_entity->get('output_file_id')->value,
      'in_progress_at' => (string) $batch_entity->get('in_progress_at')->value,
      'expires_at' => (string) $batch_entity->get('expires_at')->value,
      'finalizing_at' => (string) $batch_entity->get('finalizing_at')->value,
      'completed_at' => (string) $batch_entity->get('completed_at')->value,
      'failed_at' => (string) $batch_entity->get('failed_at')->value,
      'expired_at' => (string) $batch_entity->get('expired_at')->value,
      'cancelling_at' => (string) $batch_entity->get('cancelling_at')->value,
      'cancelled_at' => (string) $batch_entity->get('cancelled_at')->value,
    ];
    if (($local_values !== $remote_values)) {
      $storage = $this->entityTypeManager->getStorage('openai_batch_request');
      $batch_entity = $storage->createRevision($batch_entity);
      $batch_entity->set('status', $batch_response->status);
      $batch_entity->set('output_file_id', $batch_response->outputFileId);
      $batch_entity->set('in_progress_at', $batch_response->inProgressAt);
      $batch_entity->set('expires_at', $batch_response->expiresAt);
      $batch_entity->set('finalizing_at', $batch_response->finalizingAt);
      $batch_entity->set('completed_at', $batch_response->completedAt);
      $batch_entity->set('failed_at', $batch_response->failedAt);
      $batch_entity->set('expired_at', $batch_response->expiredAt);
      $batch_entity->set('cancelling_at', $batch_response->cancellingAt);
      $batch_entity->set('cancelled_at', $batch_response->cancelledAt);
      if ($save) {
        $batch_entity->save();
      }
      return $batch_entity;
    }
    return NULL;
  }

  /**
   * Create a jsonl string from an array of data.
   *
   * @param array $data_array
   *   The array of data to convert to a jsonl string.
   *
   * @return string
   *   The jsonl string.
   */
  public function createJsonlContent(array $data_array): string {
    // Initialize an empty string to hold all JSON lines.
    $jsonl_content = '';

    // Loop through each item in the data array.
    foreach ($data_array as $data) {
      // Convert each data array to a JSON string and add a newline.
      $jsonl_content .= json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . PHP_EOL;
    }
    return $jsonl_content;
  }

  private function prepareDirectory($dir) {
    try {
      $this->fileSystem->prepareDirectory($dir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
    }
    catch (FileException $e) {
      $this->messenger->addError($this->t('Failed to create directory: @dir', ['@dir' => $dir]));
      $this->logger->error('Failed to create directory: @dir', ['@dir' => $dir]);
      throw $e;
    }
  }

  /**
   * @TODO write documentation.
   *
   * @param array $batch
   * @param array $local_entity_metadata
   *
   * @return void
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function sendBatch(array $batch, string $processor_id, array $local_entity_metadata = []): void {
    $tmp_filename = uniqid('openai_batch_', TRUE);
    $tmp_filepath = sys_get_temp_dir() . '/' . $tmp_filename . '.jsonl';
    $jsonl_content = $this->createJsonlContent($batch);
    file_put_contents($tmp_filepath, $jsonl_content);

    $upload_response = $this->client->files()->upload([
      'purpose' => 'batch',
      'file' => fopen($tmp_filepath, 'r'),
    ]);
    $file_id = $upload_response->id;

    $batch_response = $this->client->batches()->create(parameters: [
      'input_file_id' => $file_id,
      'endpoint' => '/v1/chat/completions',
      'completion_window' => '24h',
      'metadata' => [
        'processor_id' => $processor_id,
        'uid' => \Drupal::currentUser()->id(),
      ],
    ]
    );
    $this->saveJsonlFile('input', $file_id, $jsonl_content);
    unlink($tmp_filepath);

    $batch_response_id = $batch_response->id;
    $batch_request_entity = OpenAiBatchRequest::create([
      'id' => $batch_response_id,
      'items' => count($batch),
      'items_processed' => 0,
      'processor' => $processor_id,
      // @todo make this configurable
      'autosync' => TRUE,
      'status' => $batch_response->status,
      'input_file_id' => $batch_response->inputFileId,
      'output_file_id' => $batch_response->outputFileId,
      'created_at' => $batch_response->createdAt,
      'expires_at' => $batch_response->expiresAt,
      'expired_at' => $batch_response->expiredAt,
      'endpoint' => $batch_response->endpoint,
      'metadata' => !empty($local_entity_metadata) ? json_encode($local_entity_metadata) : NULL,
    ]);
    $batch_request_entity->save();

    $this->messenger->addMessage($this->t('Request sent to ChatGPT, please wait for the batch processing to finish...'));
  }

  /**
   * The OpenAI API client.
   *
   * @var \OpenAI\Client
   */
  private Client $client;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private ConfigFactoryInterface $config;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  private FileSystemInterface $fileSystem;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepositoryInterface
   */
  private FileRepositoryInterface $fileRepository;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  private StateInterface $state;

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  private LoggerChannelInterface $logger;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  private MessengerInterface $messenger;

  public function __construct(
    AiProviderPluginManager $aiProvider,
    ConfigFactoryInterface $config,
    FileSystemInterface $file_system,
    FileRepositoryInterface $file_repository,
    EntityTypeManagerInterface $entity_type_manager,
    StateInterface $state,
    LoggerChannelFactoryInterface $logger_factory,
    MessengerInterface $messenger,
  ) {
    /** @var \Drupal\ai\Plugin\ProviderProxy $provider */
    $provider = $aiProvider->createInstance("openai");
    $this->client = $provider->getClient();
    $this->config = $config;
    $this->fileSystem = $file_system;
    $this->fileRepository = $file_repository;
    $this->entityTypeManager = $entity_type_manager;
    $this->state = $state;
    $this->logger = $logger_factory->get('openai_batch');
    $this->messenger = $messenger;
  }

  /**
   * Get the OpenAI API client.
   */
  public function getClient(): Client {
    return $this->client;
  }

  use StringTranslationTrait;

}
