<?php

namespace Drupal\openai_batch\Plugin\Action;

use Drupal\ai\AiProviderPluginManager;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TempStore\PrivateTempStore;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\openai_batch\DTO\OpenAiBatchRequestItemConfig;
use Drupal\openai_batch\OpenAiBatchProcessorPluginManager;
use Drupal\openai_batch\OpenAiBatchService;
use Drupal\openai_batch\Plugin\Derivative\OpenAiBatchVBOActionDerivative;
use Drupal\views_bulk_operations\Action\ViewsBulkOperationsActionBase;
use Drupal\views_bulk_operations\Action\ViewsBulkOperationsPreconfigurationInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the base VBO action for the OpenAI Batch processors.
 */
#[Action(
  id: 'openai_batch_vbo_action',
  action_label: new TranslatableMarkup('OpenAI Batch VBO Action'),
  deriver: OpenAiBatchVBOActionDerivative::class
)]
class OpenAiBatchVBOAction extends ViewsBulkOperationsActionBase implements ContainerFactoryPluginInterface, PluginFormInterface, ViewsBulkOperationsPreconfigurationInterface {

  /**
   *
   */
  private function getCurrentBatchRows(array $objects): array {
    $rows = [];
    $request_item_config = new OpenAiBatchRequestItemConfig(
      model: $this->configuration['model'],
      maxTokens: $this->configuration['configuration_max_tokens'],
      temperature: $this->configuration['configuration_temperature'],
      frequencyPenalty: $this->configuration['configuration_frequency_penalty'],
      presencePenalty: $this->configuration['configuration_presence_penalty'],
      topP: $this->configuration['configuration_top_p'],
      systemPrompt: $this->configuration['system_prompt'],
      userPrompt: $this->configuration['user_prompt'],
    );
    foreach ($objects as $entity) {
      $batch_command = $this->openAiBatchProcessor->getOpenAiBatchCommandForEntity(
        entity_type: $entity->getEntityTypeId(),
        entity_id: $entity->id(),
        request_item_config: $request_item_config,
      );
      $rows[]['batch_command'] = $batch_command;
    }
    return $rows;
  }

  /**
   * {@inheritdoc}
   */
  public function executeMultiple(array $objects) {
    if (empty($objects)) {
      return;
    }

    $rows = $this->getCurrentBatchRows($objects);
    $processed = $this->context['sandbox']['processed'] + count($rows);
    $this->saveCurrentBatchRowsToStore($rows);

    // Send the batch, if the last row has been processed.
    if (!isset($this->context['sandbox']['total']) || $processed >= $this->context['sandbox']['total']) {
      $rows = $this->getRowsFromStore();
      $batch = [];
      foreach ($rows as $row) {
        $batch_command = $row["batch_command"];
        $batch[] = $batch_command;
      }
      $local_entity_metadata = ['vbo_configuration' => $this->configuration];
      $this->openAiBatchService->sendBatch($batch, $this->openAiBatchProcessor->getPluginId(), $local_entity_metadata);
      $this->cleanupTempStore();
    }
  }

  private function cleanupTempStore() {
    $cid_prefix = $this->getCidPrefix();
    $cid_index = $this->getCidIndex();
    if (is_null($cid_index)) {
      return;
    }
    for ($i = 0; $i <= $cid_index; $i++) {
      $this->tempStore->delete($cid_prefix . $i);
    }
  }

  /**
   *
   */
  private function saveCurrentBatchRowsToStore(array $rows) {
    $cid_prefix = $this->getCidPrefix();
    foreach ($rows as $row) {
      $cid_index = $this->getCidIndex();
      $cid_current_index = is_null($cid_index) ? 0 : $cid_index + 1;
      $this->setCidIndex($cid_current_index);
      $this->tempStore->set($cid_prefix . $cid_current_index, $row);
    }
  }

  private function getRowsFromStore(): array {
    $cid_prefix = $this->getCidPrefix();
    $cid_index = $this->getCidIndex();
    if (is_null($cid_index)) {
      return [];
    }
    $rows = [];
    for ($i = 0; $i <= $cid_index; $i++) {
      $row = $this->tempStore->get($cid_prefix . $i);
      $rows[] = $row;
    }
    return $rows;
  }

  /**
   *
   */
  private function getCidPrefix() {
    if (!isset($this->context['sandbox']['cid_prefix'])) {
      $cid_prefix = $this->createCidPrefix();
      $this->context['sandbox']['cid_prefix'] = $cid_prefix;
    }
    return $this->context['sandbox']['cid_prefix'];
  }

  private function getCidIndex(): ? int {
    if (isset($this->context['sandbox']['cid_index']) && is_int($this->context['sandbox']['cid_index'])) {
      return $this->context['sandbox']['cid_index'];
    }
    return NULL;
  }

  private function setCidIndex(int $index): void {
    $this->context['sandbox']['cid_index'] = $index;
  }

  /**
   *
   */
  private function createCidPrefix(): string {
    $view_id = $this->context['view_id'];
    $display_id = $this->context['display_id'];
    $action_id = $this->context['action_id'];
    $cid = "$view_id:$display_id:$action_id:'" . json_encode(array_keys($this->context['list']));
    $time = microtime(TRUE);
    $cid = md5($cid . ":$time") . ':';
    return $cid;
  }

  /**
   * {@inheritdoc}
   */
  public function buildPreConfigurationForm(array $element, array $values, FormStateInterface $form_state): array {
    return $this->openAiBatchProcessor->buildPreConfigurationForm($element, $values, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $llm_instance = $this->aiProviderManager->createInstance("openai");
    return $this->openAiBatchProcessor->buildConfigurationForm($form, $form_state, $this->context, $llm_instance);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->openAiBatchProcessor->validateConfigurationForm($form, $form_state);
  }

  protected PrivateTempStore $tempStore;

  protected EntityTypeManagerInterface $entityTypeManager;

  protected AiProviderPluginManager $aiProviderManager;

  protected EntityFieldManagerInterface $entityFieldManager;

  protected OpenAiBatchVBOActionInterface $openAiBatchProcessor;

  private OpenAiBatchService $openAiBatchService;

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    PrivateTempStoreFactory $temp_store_factory,
    EntityTypeManagerInterface $entity_type_manager,
    MessengerInterface $messenger,
    EntityFieldManagerInterface $entity_field_manager,
    AiProviderPluginManager $ai_provider_plugin_manager,
    OpenAiBatchProcessorPluginManager $openai_batch_processor_plugin_manager,
    OpenAiBatchService $openai_batch_service,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->tempStore = $temp_store_factory->get('openai_batch_' . $this->getPluginId());
    $this->entityTypeManager = $entity_type_manager;
    $this->messenger = $messenger;
    $this->entityFieldManager = $entity_field_manager;
    $this->aiProviderManager = $ai_provider_plugin_manager;
    $openai_batch_processor_id = $plugin_definition["openai_batch_processor_id"];
    $this->openAiBatchProcessor = $openai_batch_processor_plugin_manager->createInstance($openai_batch_processor_id);
    $this->openAiBatchService = $openai_batch_service;
    $this->openAiBatchProcessor->setVboConfiguration($this->configuration);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('tempstore.private'),
      $container->get('entity_type.manager'),
      $container->get('messenger'),
      $container->get('entity_field.manager'),
      $container->get('ai.provider'),
      $container->get('plugin.manager.openai_batch_processor'),
      $container->get('openai_batch.batch_service'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    return $object->access('update', $account, $return_as_object);
  }

  /**
   * {@inheritdoc}
   */
  public function execute($entity = NULL) {
    $this->executeMultiple([$entity]);
  }

}
