<?php

namespace Drupal\openfed_migrate\Plugin\migrate\source\bean;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\migrate\Plugin\MigrationInterface;
use Drupal\migrate\Row;
use Drupal\migrate_drupal\Plugin\migrate\source\d7\FieldableEntity;
use Drupal\openfed_migrate\Helper;
use Drupal\openfed_migrate\Plugin\migrate\source\FieldEntityTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drupal 7 Bean source from database.
 *
 * @MigrateSource(
 *   id = "d7_bean_item",
 *   source_module = "bean"
 * )
 */
class BeanItem extends FieldableEntity {
  use FieldEntityTrait;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, MigrationInterface $migration, StateInterface $state, EntityTypeManagerInterface $entity_manager, ModuleHandlerInterface $module_handler) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $migration, $state, $entity_manager);
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition, MigrationInterface $migration = NULL) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $migration,
      $container->get('state'),
      $container->get('entity_type.manager'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    // Select bean in its last revision.
    $query = $this->select('bean_revision', 'br')
      ->fields('b', [
        'bid',
        'title',
        'label',
        'type',
        'uid',
        'created',
        'changed',
      ])
      ->fields('br', [
        'vid',
        'title',
        'delta',
        'log',
      ]);
    $query->addField('b', 'uid', 'block_uid');
    $query->addField('br', 'uid', 'revision_uid');
    $query->innerJoin('bean', 'b', 'b.vid = br.vid');

    if (isset($this->configuration['bean_type'])) {
      $query->condition('b.type', $this->configuration['bean_type']);
    }

    // Add extra validation for existing bundles.
    // Left join bean types to later check if they exist. We additionally add a
    // check to see if there is bean content because in some cases bean_type
    // table is empty and entity type info is in features, allowing content to
    // be added.
    $query->leftJoin('bean_type', 'bean_type', "b.type = bean_type.name");
    $query->leftJoin('bean', 'bean', "b.type = bean.type");
    $query->condition('bean.type', NULL, 'IS NOT NULL');
    $query->distinct();
    $query->orderBy('b.bid');

    // Join block table to get the unique block id.
    $default_theme = $this->variableGet('theme_default', 'Garland');
    $query->leftJoin('block', 'block', 'block.delta=b.delta');
    $query->addField('block', 'bid', 'unique_bid');
    $query->condition('block.theme', $default_theme, 'LIKE');
    $query->condition('block.module', 'bean');

    return $query;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row): bool {

    // Add extra validation for existing bundles.
    if (!Helper::sourcePrepareRowBundleValidator($row, 'type')) {
      return FALSE;
    }
    // Get Field API field values.
    foreach (array_keys($this->getFields('bean', $row->getSourceProperty('type'))) as $field) {
      $bid = $row->getSourceProperty('bid');
      $vid = $row->getSourceProperty('vid');
      $row->setSourceProperty($field, $this->getFieldValues('bean', $field, $bid, $vid));
    }
    return parent::prepareRow($row);
  }

  /**
   * {@inheritdoc}
   */
  public function fields() {
    $fields = [
      'bid' => $this->t('Block ID'),
      'type' => $this->t('Type'),
      'title' => $this->t('Title'),
      'uid' => $this->t('Authored by (uid)'),
      'revision_uid' => $this->t('Revision authored by (uid)'),
      'created' => $this->t('Created timestamp'),
      'changed' => $this->t('Modified timestamp'),
      'revision' => $this->t('Create new revision'),
    ];
    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function getIds() {
    $ids['bid']['type'] = 'integer';
    $ids['bid']['alias'] = 'b';
    return $ids;
  }

}
