<?php

namespace Drupal\openfed_migrate\Plugin\migrate\source\menu_block_settings_override;

use Drupal\migrate\Row;
use Drupal\migrate_drupal\Plugin\migrate\source\DrupalSqlBase;


/**
 * Drupal 7 field instances source from database.
 *
 * @MigrateSource(
 *   id = "d7_menu_block_settings_override_field_instances",
 *   source_module = "menu_block_settings_override"
 * )
 */
class MenuBlockSettingsOverrideFieldInstance extends DrupalSqlBase {

  /**
   * {@inheritdoc}
   */
  public function query() {
    // Since this is a entity field_config, we need entity_type, bundle and
    // field name. As such, we need to get a query with these columns. Since
    // this is not a field migration but more a variable migration, and since
    // menu_block_settings_override has its own table in D7, we will process the
    // rows generated by this query on a iterator.
    $query = $this->select('menu_block_settings_override', 'mbso');

    // We do a left join to get all the nodes in the above conditions.
    // We don't really care about the results, we only care about the bundles.
    $query->join('node', 'n', "mbso.path LIKE CONCAT('node/', n.nid)");
    $query->fields('n', ['type']);

    // We can exclude nodes that were never overriden or hidden. In D7, MBSO was
    // implemented on all content types but not all content types were really
    // making use of it. We won't migrate this new field into unnecessary node
    // types.
    $orGroup = $query->orConditionGroup()
      ->condition('mbso.status', 0, '<>')
      ->condition('mbso.hidden', 0, '<>');
    // Add the group to the query.
    $query->condition($orGroup);

    // We group it by type, which is what we are really looking for.
    $query->groupBy('type');

    return $query;
  }

  /**
   * {@inheritdoc}
   */
  public function fields() {
    return [
      'id' => $this->t('The field instance ID.'),
      'entity_type' => $this->t('The entity type.'),
      'bundle' => $this->t('The entity bundle.'),
      'field_name' => $this->t('The field name.'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row) {
    // Set the settings property, which is empty for now.
    $settings = [];
    $row->setSourceProperty('settings', $settings);

    return parent::prepareRow($row);
  }

  /**
   * {@inheritdoc}
   */
  public function getIds() {
    return [
      'entity_type' => [
        'type' => 'string',
      ],
      'bundle' => [
        'type' => 'string',
      ],
      'field_name' => [
        'type' => 'string',
      ],
    ];
  }
  
  /**
   * {@inheritdoc}
   */
  protected function initializeIterator() {
    $results = $this->prepareQuery()->execute()->fetchAll();

    // Group all instances by their node bundle.
    $instances = [];
    $entity_type = 'node';
    $field_name = 'field_menu_block_settings';
    foreach ($results as $result) {
      $instances[] = [
        'id' => $entity_type . '.' . $result['type'] . '.' . $field_name,
        'field_name' => $field_name,
        'entity_type' => $entity_type,
        'bundle' => $result['type'],
      ];
    }

    return new \ArrayIterator($instances);
  }

}
