<?php

namespace Drupal\openfed_migrate\Plugin\migrate\source\webform;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\migrate\Event\MigrateImportEvent;
use Drupal\migrate\MigrateMessage;
use Drupal\migrate\Row;
use Drupal\node\Entity\Node;
use Drupal\webform\Entity\Webform;
use Drupal\webform_migrate\Plugin\migrate\source\d7\D7Webform;
use Drupal\webform\Utility\WebformYaml;
use Drupal\Component\Utility\Bytes;

/**
 * Drupal 7 webform source from database.
 *
 * @MigrateSource(
 *   id = "d7_webform",
 *   core = {7},
 *   source_module = "webform",
 *   destination_module = "webform"
 * )
 */
class OpenfedWebform extends D7Webform {

  /**
   * {@inheritdoc}
   */
  public function query() {
    $query = parent::query();

    // Add language and tnid fields in order to assign the correct webforms to
    // node translations.
    $query->fields('n', ['language', 'tnid']);

    return $query;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row) {
    parent::prepareRow($row);

    // Send an alert if email handler doesn't exist.
    if (empty($row->getSourceProperty('handlers'))) {
      $message = new MigrateMessage();
      $message->display(
        $this->t(
          "The webform @form_id doesn't have an email handler. Please create one after the migration.\r\n",
          [
            '@form_id' => 'webform_' . $row->getSourceProperty('nid'),
          ]
        ),
        'alert'
      );
    }

    // We'll override the elements generated by parent prepareRow() because we
    // need to handle some issues.
    $nid = $row->getSourceProperty('nid');
    $webform = $this->buildFormElements($nid);
    $elements = $webform['elements'];

    $row->setSourceProperty('elements', $elements);

    return $row;
  }

  /**
   * Build form elements from webform component table.
   */
  private function buildFormElements($nid) {
    // Resulting array build that will be converted to YAML.
    $build = [];
    // Array with all elements keyed by form_key for a quick access.
    $references = [];

    $query = $this->select('webform_component', 'wc');
    $query->fields('wc', [
      'nid',
      'cid',
      'pid',
      'form_key',
      'name',
      'type',
      'value',
      'extra',
      'required',
      'weight',
    ]);
    $components = $query->condition('nid', $nid)->orderBy('pid')->orderBy('weight')->execute();
    $children = [];
    $parents = [];
    $elements = [];
    $xref = [];

    // Build an array of elements in the correct order for rendering based on
    // pid and weight and a cross reference array to match cid with form_key
    // used by email handler.
    $multiPage = FALSE;
    foreach ($components as $component) {
      // CUSTOM: string lower element key as this is a requirement.
      $xref[$component['cid']] = strtolower($component['form_key']);
      if ($component['type'] == 'pagebreak') {
        // Pagebreak found so we have a multi-page form.
        $multiPage = TRUE;
      }
      $children[$component['pid']][] = $component['cid'];
      $parents[$component['cid']][] = $component['pid'];
      $elements[$component['cid']] = $component;
    }
    // Keeps track of the parents we have to process, the last entry is used
    // for the next processing step.
    $process_parents = [];
    $process_parents[] = 0;
    $elements_tree = [];
    // Loops over the parent components and adds its children to the tree array.
    // Uses a loop instead of a recursion, because it's more efficient.
    while (count($process_parents)) {
      $parent = array_pop($process_parents);
      // The number of parents determines the current depth.
      $depth = count($process_parents);
      if (!empty($children[$parent])) {
        $has_children = FALSE;
        $child = current($children[$parent]);
        do {
          if (empty($child)) {
            break;
          }
          $element = &$elements[$child];
          $element['depth'] = $depth;
          // We might get element with same form_key
          // d8 doesn't like that so rename it.
          if ($depth > 0) {
            // CUSTOM: string lower element key as this is a requirement.
            $element['form_key'] = strtolower($element['form_key']) . '_' . $element['pid'];
          }

          // Rename fieldsets to it's own unique key.
          if ($element['type'] == 'fieldset' && strpos($element['form_key'], 'fieldset') === FALSE) {
            $element['form_key'] = 'fieldset_' . $element['form_key'];
          }
          $element['form_key'] = strtolower($element['form_key']);

          $elements_tree[] = $element;
          if (!empty($children[$element['cid']])) {
            $has_children = TRUE;
            // We have to continue with this parent later.
            $process_parents[] = $parent;
            // Use the current component as parent for the next iteration.
            $process_parents[] = $element['cid'];
            // Reset pointers for child lists because we step in there more
            // often with multi parents.
            reset($children[$element['cid']]);
            // Move pointer so that we get the correct term the next time.
            next($children[$parent]);
            break;
          }
        } while ($child = next($children[$parent]));

        if (!$has_children) {
          // We processed all components in this hierarchy-level.
          reset($children[$parent]);
        }
      }
    }

    $parent_element = &$build;
    // If form has multiple pages then start first page automatically.
    if ($multiPage) {
      $build['first_page'] = [
        '#type' => 'webform_wizard_page',
        '#title' => 'Start',
      ];
      $parent_element = &$build['first_page'];
    }

    foreach ($elements_tree as $element) {
      // If this is a multi-page form then indent all elements one level
      // to allow for page elements.
      if ($multiPage && $element['type'] != 'pagebreak') {
        $element['depth'] += 1;
      }
      $indent = str_repeat(' ', $element['depth'] * 2);
      // CUSTOM: we need this because serialized data sometimes is broken and
      // we get "unserialize() [function.unserialize]: Error at offset" error.

      $extra = preg_replace_callback ( '!s:(\d+):"((.|\n)*?)";!',
        function($match) {
          return ($match[1] == strlen($match[2])) ? $match[0] : 's:' . strlen($match[2]) . ':"' . $match[2] . '";';
        }, $element['extra'] );
      $extra = unserialize($extra);


      // Create an option list if there are items for this element.
      $options       = [];
      $valid_options = [];
      if (!empty($extra['items'])) {
        $items = explode("\n", trim($extra['items']));
        foreach ($items as $item) {
          $item = trim($item);
          if (!empty($item)) {
            if (preg_match('/^<(.*)>$/', $item, $matches)) {
              if (empty(trim($matches[1]))) {
                continue;
              }
              // Handle option groups.
              $options[$matches[1]] = '';
            }
            else {
              $option = explode('|', $item);
              $valid_options[] = $option[0];
              if (count($option) == 2) {
                $options[$option[0]] = $option[1];
              }
              else {
                $options[$option[0]] = $option[0];
              }
            }
          }
        }
      }

      // Replace any tokens in the value.
      if (!empty($element['value'])) {
        $element['value'] = $this->replaceTokens($element['value']);
      }

      // Let's find out the parent for the given element.
      if (!empty($element['pid']) && !empty($elements[$element['pid']]['form_key'])) {
        $parent_key = $elements[$element['pid']]['form_key'];
        if (!empty($references[$parent_key])) {
          $parent_element = &$references[$parent_key];
        }
      }
      elseif ($multiPage && $element['type'] !== 'pagebreak') {
        // If previous item was a page, use it as parent element.
        // Otherwise, use previous parent.
        if (!empty($new_element['#type']) && $new_element['#type'] === 'webform_wizard_page') {
          $parent_element = &$new_element;
        }
      }
      else {
        $parent_element = &$build;
      }

      $form_key              = $element['form_key'];
      $new_element           = &$parent_element[$form_key];
      $references[$form_key] = &$new_element;
      switch ($element['type']) {
        case 'fieldset':
          $new_element = [
            '#type' => 'fieldset',
            '#open' => TRUE,
          ];
          if ($multiPage && $parent_element['#type'] === 'webform_wizard_page' && empty($parent_element['#title'])) {
            $parent_element['#title'] = $element['name'];
          }
          break;

        case 'textfield':
          $new_element['#type'] = 'textfield';
          if (!empty($extra['width'])) {
            $new_element['#size'] = (int) $extra['width'];
          }
          break;

        case 'textarea':
          $new_element['#type'] = 'textarea';
          break;

        case 'select':
          if (!empty($extra['aslist'])) {
            $select_type = 'select';
          }
          elseif (!empty($extra['multiple']) && count($valid_options) > 1) {
            $select_type = 'checkboxes';
          }
          elseif (!empty($extra['multiple']) && count($valid_options) == 1) {
            // This code was reverted to the original one on 2023-03-27
            $select_type = 'checkbox';
            list($key, $desc) = explode('|', $extra['items']);
            $markup .= "$indent  '#description': \"" . $this->cleanString($desc) . "\"\n";
          }
          else {
            $select_type = 'radios';
          }

          $new_element = [
            '#type' => $select_type,
            '#options' => $options,
          ];

          if (!empty($extra['multiple'])) {
            $new_element['#multiple'] = TRUE;
          }
          break;

        case 'email':
          $new_element = [
            '#type' => 'email',
            '#size' => 20,
          ];
          break;

        case 'number':
          if ($extra['type'] == 'textfield') {
            $new_element = [
              '#type' => 'textfield',
              '#size' => 20,
            ];
          }
          elseif ($extra['type'] == 'select') {
            $new_element = [
              '#type' => 'select',
              '#options' => $options,
            ];

            $min  = $extra['min'];
            $max  = $extra['max'];
            $step = !empty($extra['step']) ? $extra['step'] : 1;
            for ($value = $min; $value <= $max; $value += $step) {
              $new_element[$value] = $value;
            }
          }
          foreach (['min', 'max', 'step'] as $property) {
            if (!empty($extra[$property])) {
              $new_element["#{$property}"] = $extra[$property];
            }
          }
          if (isset($extra['unique'])) {
            $new_element['#unique'] = (bool) $extra['unique'];
          }
          break;

        case 'markup':
          $new_element      = [
            '#type' => 'processed_text',
            '#format' => 'full_html',
            '#text' => trim($element['value']),
          ];
          $element['value'] = '';
          break;

        case 'file':
        case 'multiple_file':
          $exts = '';
          if (!empty($extra['filtering']['types'])) {
            $types = $extra['filtering']['types'];
            if (!empty($extra['filtering']['addextensions'])) {
              $add_types = explode(',', $extra['filtering']['addextensions']);
              $types     = array_unique(array_merge($types, array_map('trim', $add_types)));
            }
            $exts = implode(' ', $types);
          }

          $file_size = '';
          if (!empty($extra['filtering']['size'])) {
            // Get the string for the size. Will be something like "2 MB".
            $size = $extra['filtering']['size'];

            // Convert the string into an integer in bytes.
            $file_size_bytes = Bytes::toInt($size);

            // Convert that to MB.
            $file_size = floor($file_size_bytes / 1024 / 1024);

            // Failsafe as Webform doesn't let you go less than 1MB.
            $file_size = ($file_size < 1) ? 1 : $file_size;
          }

          $new_element = [
            '#type' => 'managed_file',
            '#max_filesize' => $file_size,
            '#file_extensions' => $exts,
          ];

          if (!empty($extra['width'])) {
            $new_element['#size'] = $extra['width'];
          }
          if ($element['type'] == 'multiple_file') {
            $new_element['#multiple'] = TRUE;
          }
          break;

        case 'date':
          $new_element['#type'] = 'date';
          break;

        case 'time':
          $new_element['#type'] = 'webform_time';

          if (!empty($extra['hourformat'])) {
            if ($extra['hourformat'] == '12-hour') {
              $new_element['#time_format'] = 'g:i A';
            }
            elseif ($extra['hourformat'] == '24-hour') {
              $new_element['#time_format'] = 'H:i';
            }
          }
          /*if (!empty($element['value'])) {
          $element['value'] = date('c', strtotime($element['value']));
          }*/
          break;

        case 'hidden':
          $new_element['#type'] = 'hidden';
          break;

        case 'pagebreak':
          $new_element = [
            '#type' => 'webform_wizard_page',
            '#title' => $element['name'],
          ];
          break;

        case 'addressfield':
          $new_element['#type']                 = 'webform_address';
          $new_element['#state_province__type'] = 'textfield';
          break;

        case 'grid':
          $questions                 = $this->getItemsArray($extra['questions']);
          $new_element['#type']      = 'webform_likert';
          $new_element['#questions'] = $questions;
          $new_element['#answers']   = $this->getItemsArray($extra['options']);
          break;

        default:
          // @todo We should make some notice if element type was not found.
          break;
      }

      // Continue passing element markup to make it less breaking change.
      if (!empty($element['type']) && is_string($element['type'])) {
        $element_markup = !empty($new_element) ? WebformYaml::encode($new_element) : '';
        $this->getModuleHandler()
          ->alter('webform_migrate_d7_webform_element_' . $element['type'], $element_markup, $indent, $element);
        $new_element = WebformYaml::decode($element_markup);
      }

      // Add common fields.
      if (!empty(trim($element['value'])) && (empty($valid_options) || in_array($element['value'], $valid_options))) {
        $new_element['#default_value'] = trim($element['value']);
      }
      if (!empty($extra['field_prefix'])) {
        $new_element['#field_prefix'] = $extra['field_prefix'];
      }
      if (!empty($extra['field_suffix'])) {
        $new_element['#field_suffix'] = $extra['field_suffix'];
      }
      if (!empty($extra['title_display']) && $extra['title_display'] != 'before') {
        $title_display = $extra['title_display'];
        if ($title_display == 'none') {
          $title_display = 'invisible';
        }
        $new_element['#title_display'] = $title_display;
      }
      if ($element['type'] != 'pagebreak') {
        $new_element['#title'] = $element['name'];

        // The description key can be missing (since description is optional and
        // it isn't saved by Drupal 7 webform when it is left empty).
        if (!empty($extra['description'])) {
          $new_element['#description'] = $extra['description'];
        }
      }
      if (!empty($element['required'])) {
        $new_element['#required'] = TRUE;
      }

      // Attach conditionals as Drupal #states.
      if ($states = $this->buildConditionals($element, $elements)) {
        $new_element['#states'] = $states;
      }
    }

    $output = WebformYaml::encode($build);
    return ['elements' => $output, 'xref' => $xref];
  }

  /**
   * Translate webform tokens into regular tokens.
   *
   * %uid - The user id (unsafe)
   * %username - The name of the user if logged in.
   *                       Blank for anonymous users. (unsafe)
   * %useremail - The e-mail address of the user if logged in.
   *                       Blank for anonymous users. (unsafe)
   * %ip_address - The IP address of the user. (unsafe)
   * %site - The name of the site
   *             (i.e. Northland Pioneer College, Arizona) (safe)
   * %date - The current date, formatted according
   *              to the site settings.(safe)
   * %nid - The node ID. (safe)
   * %title - The node title. (safe)
   * %sid - The Submission id (unsafe)
   * %submission_url - The Submission url (unsafe)
   * %profile[key] - Any user profile field or value, such as %profile[name]
   *                         or %profile[profile_first_name] (unsafe)
   * %get[key] - Tokens may be populated from the URL by creating URLs of
   *                    the form http://example.com/my-form?foo=bar.
   *                    Using the token %get[foo] would print "bar". (safe)
   * %post[key] - Tokens may also be populated from POST values
   *                      that are submitted by forms. (safe)
   * %email[key] (unsafe)
   * %value[key] (unsafe)
   * %email_values (unsafe)
   * %cookie[key] (unsafe)
   * %session[key] (unsafe)
   * %request[key] (unsafe)
   * %server[key] (unsafe)
   *
   * Safe values are available to all users and unsafe values
   * should only be shown to authenticated users.
   */
  private function replaceTokens($str) {
    return $str;
  }

  /**
   * {@inheritdoc}
   */
  private function cleanString($str) {
    return str_replace(['"', "\n", "\r"], ["'", '\n', ''], $str);
  }

  /**
   * Build conditionals and translate them to states api in D8.
   */
  private function buildConditionals($element, $elements) {
    $nid = $element['nid'];
    $cid = $element['cid'];
    // CUSTOM: we need this because serialized data sometimes is broken and
    // we get "unserialize() [function.unserialize]: Error at offset" error.
    $extra = preg_replace_callback ( '!s:(\d+):"((.|\n)*?)";!',
      function($match) {
        return ($match[1] == strlen($match[2])) ? $match[0] : 's:' . strlen($match[2]) . ':"' . $match[2] . '";';
      }, $element['extra'] );
    $extra = unserialize($extra);
    // Checkboxes : ':input[name="add_more_locations_24[yes]"]':
    $query = $this->select('webform_conditional', 'wc');
    $query->innerJoin('webform_conditional_actions', 'wca', 'wca.nid=wc.nid AND wca.rgid=wc.rgid');
    $query->innerJoin('webform_conditional_rules', 'wcr', 'wcr.nid=wca.nid AND wcr.rgid=wca.rgid');
    $query->fields('wc', [
      'nid',
      'rgid',
      'andor',
      'weight',
    ])
      ->fields('wca', [
        'aid',
        'target_type',
        'target',
        'invert',
        'action',
        'argument',
      ])
      ->fields('wcr', [
        'rid',
        'source_type',
        'source',
        'operator',
        'value',
      ]);
    $conditions = $query->condition('wc.nid', $nid)->condition('wca.target', $cid)->execute();
    $states = [];

    if (!empty($conditions)) {
      foreach ($conditions as $condition) {
        $unsupported_condition = FALSE;
        // Element states.
        switch ($condition['action']) {
          case 'show':
            $element_state = $condition['invert'] ? 'invisible' : 'visible';
            break;

          case 'require':
            $element_state = $condition['invert'] ? 'optional' : 'required';
            break;

          case 'set':
            // Nothing found in D8 :(.
            $unsupported_condition = TRUE;
            break;
        }
        // Condition states.
        $operator_value = $condition['value'];
        $depedent = $elements[$condition['source']];
        $depedent_extra = unserialize($depedent['extra']);
        $depedent_extra['items'] = explode("\n", $depedent_extra['items']);

        switch ($condition['operator']) {
          case 'equal':
            $element_condition = ['value' => $operator_value];
            if ($depedent['type'] == 'select' && !$depedent_extra['aslist'] && $depedent_extra['multiple']) {
              $element_condition = ['checked' => TRUE];
            }
            break;

          case 'not_equal':
            // There is no handler for this in D8 so we do the reverse.
            $element_state = $condition['invert'] ? 'visible' : 'invisible';
            $element_condition = ['value' => $operator_value];
            // Specially handle the checkboxes.
            if ($depedent['type'] == 'select' && !$depedent_extra['aslist'] && $depedent_extra['multiple']) {
              $element_condition = ['checked' => TRUE];
            }

            break;

          case 'less_than':
            $element_condition = ['value' => ['less' => $operator_value]];
            break;

          case 'less_than_equal':
            $element_condition = ['value' => ['less_equal' => $operator_value]];
            break;

          case 'greater_than':
            $element_condition = ['value' => ['greater' => $operator_value]];
            break;

          case 'greater_than_equal':
            $element_condition = ['value' => ['greater_equal' => $operator_value]];
            break;

          case 'empty':
            if ($operator_value == 'checked') {
              $element_condition = ['unchecked' => TRUE];
            }
            else {
              $element_condition = ['empty' => TRUE];
            }
            break;

          case 'not_empty':
            if ($operator_value == 'checked') {
              $element_condition = ['checked' => TRUE];
            }
            else {
              $element_condition = ['filled' => FALSE];
            }
            break;
        }

        if (!$depedent_extra['aslist'] && $depedent_extra['multiple'] && is_array($depedent_extra['items']) && count($depedent_extra['items']) > 1) {
          $depedent['form_key'] = strtolower($depedent['form_key']) . "[$operator_value]";
        }
        elseif (!$depedent_extra['aslist'] && !$depedent_extra['multiple'] && is_array($depedent_extra['items']) && count($depedent_extra['items']) == 1) {
          $depedent['form_key'] = strtolower($depedent['form_key']) . "[$operator_value]";
        }

        if (!$unsupported_condition) {
          $states[$element_state][] = [':input[name="' . strtolower($depedent['form_key']) . '"]' => $element_condition];
        }

      }
      if (empty($states)) {
        return FALSE;
      }
      return $states;
    }
    else {
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postImport(MigrateImportEvent $event) {
    // This is the same code as on D7Webform.php but we need to improve it to
    // make the fields translateable and save the webform on the right node
    // translation.
    // This is a dirty way of creating the webform field but that's what we
    // have...
    // Add the Webform field to the webform content type
    // if it doesn't already exist.
    $field_storage = FieldStorageConfig::loadByName('node', 'webform');
    if (empty($field_storage)) {
      $field_storage = FieldStorageConfig::create(['entity_type' => 'node', 'field_name' => 'webform', 'type' => 'webform', 'settings' => [], 'cardinality' => 1, 'translatable' => 'true']);
      $field_storage->save();
    }
    $field = FieldConfig::loadByName('node', 'webform', 'webform');
    if (empty($field)) {
      $field = \Drupal::service('entity_type.manager')->getStorage('field_config')->create([
        'field_name' => 'webform',
        'bundle' => 'webform',
        'label' => 'Webform',
        'translatable' => 'true',
        'entity_type' => 'node',
      ]);
      $field->save();
      // Assign widget settings for the 'default' form mode.
      $display = \Drupal::service('entity_display.repository')->getFormDisplay('node', 'webform', 'default')->getComponent('webform');
      \Drupal::service('entity_display.repository')->getFormDisplay('node', 'webform', 'default')
        ->setComponent('webform', [
          'type' => $display['type'],
        ])
        ->save();
      // Assign display settings for the 'default' and 'teaser' view modes.
      $display = \Drupal::service('entity_display.repository')->getViewDisplay('node', 'webform', 'default')->getComponent('webform');
      \Drupal::service('entity_display.repository')->getViewDisplay('node', 'webform', 'default')
        ->setComponent('webform', [
          'label' => $display['label'],
          'type' => $display['type'],
        ])
        ->save();
      // The teaser view mode is created by the Standard profile and therefore
      // might not exist.
      $view_modes = \Drupal::service('entity_display.repository')->getViewModes('node');
      if (isset($view_modes['teaser'])) {
        $display = \Drupal::service('entity_display.repository')->getViewDisplay('node', 'webform', 'teaser')->getComponent('webform');
        \Drupal::service('entity_display.repository')->getViewDisplay('node', 'webform', 'teaser')
          ->setComponent('webform', [
            'label' => $display['label'],
            'type' => $display['type'],
          ])
          ->save();
      }
    }

    // Attach any Webform created to the relevant webforms if
    // Webform exists and Webform exists and Webform field is empty.
    $webforms = $this->query()->execute();
    foreach ($webforms as $webform) {
      $webform_nid = $webform['nid'];
      $webform_id = 'webform_' . $webform_nid;
      $webform_tnid = $webform['tnid'];
      $webform = Webform::load($webform_id);
      if (!empty($webform)) {
        // Change the webform.
        // Attach any Webform created to the relevant webforms if
        // Webform exists and Webform exists and Webform field is empty.
        $webformSettings = $webform->get('settings');
        $webformSettings['results_disabled'] = 1;
        $webformSettings['page'] = 0;
        $elements = $webform->getElementsDecoded();
        if (isset($webformSettings['form_submit_label']) && !empty($webformSettings['form_submit_label'])) {
          $elements['actions'] = [
            '#type' => 'webform_actions',
            '#title' => 'webform_actions',
            '#submit__label' => $webformSettings['form_submit_label'],
          ];
        }
        $webform->set('third_party_settings', ['honeypot' => ['honeypot' => true]]);
        $webform->set('settings', $webformSettings);
        $webform->setElements($elements);
        $webform->save();

        // Change the attached node.
        if (!empty($webform_tnid) && $webform_tnid > 0) {
          $langcode = $webform->getLangcode();
          $node = Node::load($webform_tnid);
          $node = $node->hasTranslation($langcode) ? $node->getTranslation($langcode) : $node;
        } else {
          $node = Node::load($webform_nid);
        }
        if (!empty($node) && $node->getType() == 'webform') {
          $node->webform->target_id = $webform_id;
          $node->webform->status = 'open';
          $node->save();
        }
      }
    }
  }

  /**
   * @todo Add documentation.
   *
   * @param string $rawString
   *   @todo Add documentation.
   *
   * @return array
   *   @todo Add documentation.
   */
  protected function getItemsArray($rawString) {
    $items = explode("\n", $rawString);
    $items = array_map('trim', $items);
    return array_map(function($item) {
      return explode('|', $item);
    }, $items);
  }

  /**
   * @todo Add documentation.
   *
   * @param array $itemsArray
   *   @todo Add documentation.
   * @param string $baseIndent
   *   @todo Add documentation.
   *
   * @return string
   *   @todo Add documentation.
   */
  protected function buildItemsString(array $itemsArray, $baseIndent = '') {
    $preparedItems = array_map(function ($item) use ($baseIndent) {
      return $baseIndent . '  ' . $this->encapsulateString($item[0]) . ': ' . $this->encapsulateString($item[1]);
    }, $itemsArray);

    return implode("\n", $preparedItems);
  }

  /**
   * @todo Add documentation.
   *
   * @param string $string
   *   @todo Add documentation.
   *
   * @return string
   *   @todo Add documentation.
   */
  protected function encapsulateString($string) {
    return sprintf("'%s'", addslashes($string));
  }

}