<?php

namespace Drupal\openfed_migrate\Plugin\migrate\source\menu_link;

use Drupal\migrate\Row;
use Drupal\migrate_drupal\Plugin\migrate\source\DrupalSqlBase;


/**
 * Drupal 7 field instances source from database.
 *
 * @MigrateSource(
 *   id = "d7_menu_link_field_instances",
 *   source_module = "node"
 * )
 */
class MenuLinkFieldInstances extends DrupalSqlBase {

  /**
   * {@inheritdoc}
   */
  public function query() {
    // Since this is a entity field_config, we need entity_type, bundle and
    // field name. As such, we need to get a query with these columns. Since
    // this is not a field migration but more a variable migration, and since
    // menu_options is a variable in D7, we will process the rows generated by
    // this query on a iterator.
    $query = $this->select('field_config_instance', 'fci')->fields('fci');

    // We need a subquery to retrieve the variables "menu_options_" and from
    // there get the node types and see if they used to have menus enabled.
    $subquery = $this->select('variable', 'v')
      ->fields('v');
    $subquery->condition('v.name', 'menu_options_%', 'LIKE');
    foreach ($subquery->execute() as $variable) {
      if (!empty(unserialize($variable['value']))) {
        $node_types[] = str_replace('menu_options_', '', $variable['name']);
      }
    }

    // Conditions to get only the fields we need.
    $query->condition('fci.bundle', $node_types, 'IN');
    $query->condition('fci.entity_type', 'node');

    // We do a left join to get all the fields in the above conditions.
    // We don't really care about the results, we only care about the bundles.
    // See the iterator for more info.
    $query->leftJoin('field_config_instance', 'fci2', 'fci.id = fci2.id');
    $query->fields('fci2');
    $query->condition('fci2.entity_type', 'node');

    return $query;
  }

  /**
   * {@inheritdoc}
   */
  public function fields() {
    return [
      'id' => $this->t('The field instance ID.'),
      'entity_type' => $this->t('The entity type.'),
      'bundle' => $this->t('The entity bundle.'),
      'field_name' => $this->t('The field name.'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row) {
    // The node type is a suffix of menu_options_[nodetype] variable.
    $type = $row->getSourceProperty('bundle');

    // We need to get all menus in order to correctly configure the field. All
    // menus start disabled (0).
    $menus = \Drupal::entityTypeManager()->getListBuilder('menu')->load();
    $available_menus = array_map(function ($a) {
      return '0';
    }, $menus);

    // Let's build the source property array.
    $row->setSourceProperty('entity_type', 'node');
    $row->setSourceProperty('bundle', $type);
    $row->setSourceProperty('field_name', 'field_menulink');

    $menu_options = $this->variableGet('menu_options_' . $type, NULL);
    foreach ($menu_options as $menu) {
      // We need a condition for the main menu since in D7 it was called
      // "main-menu" and in D8 is called just "main".
      if ($menu == 'main-menu') {
        $menu = 'main';
      }
      // We enable, on the previously created array, the menus present in the
      // old configuration.
      $available_menus[$menu] = $menu;
    }
    $menu_parent = str_replace(':0', ':', $this->variableGet('menu_parent_' . $type, NULL));
    // We need a condition for the main menu since in D7 it was called
    // "main-menu" and in D8 is called just "main".
    if ($menu_parent == 'main-menu') {
      $menu_parent = 'main';
    }
    $settings = [
      'available_menus' => $available_menus,
      'default_menu_parent' => $menu_parent,
    ];
    $row->setSourceProperty('settings', $settings);

    return parent::prepareRow($row);
  }

  /**
   * {@inheritdoc}
   */
  public function getIds() {
    return [
      'entity_type' => [
        'type' => 'string',
        'alias' => 'fci2',
      ],
      'bundle' => [
        'type' => 'string',
        'alias' => 'fci2',
      ],
      'field_name' => [
        'type' => 'string',
        'alias' => 'fci2',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function count($refresh = FALSE) {
    // We alter the count function to have the iterator into consideration.
    return $this->initializeIterator()->count();
  }

  /**
   * {@inheritdoc}
   */
  protected function initializeIterator() {
    $results = $this->prepareQuery()->execute()->fetchAll();

    // Group all instances by their node bundle.
    $instances = [];
    foreach ($results as $result) {
      $instances[$result['bundle']] = [
        'id' => $result['entity_type'] . '.' . $result['bundle'] . '.field_menulink',
        'field_name' => 'field_menulink',
        'entity_type' => $result['entity_type'],
        'bundle' => $result['bundle'],
      ];
    }

    // Generate an indexed array of rows.
    $rows = [];
    foreach ($instances as $instance) {
      $rows[] = $instance;
    }

    return new \ArrayIterator($rows);
  }

}
