<?php

namespace Drupal\openfed_migrate\Plugin\migrate\source\weight;

use Drupal\migrate\Row;
use Drupal\migrate_drupal\Plugin\migrate\source\DrupalSqlBase;


/**
 * Drupal 7 field instances source from database.
 *
 * @MigrateSource(
 *   id = "d7_weight_field_instance",
 *   source_module = "weight"
 * )
 */
class WeightFieldInstance extends DrupalSqlBase {

  /**
   * {@inheritdoc}
   */
  public function query() {
    // Since this is a entity field_config, we need entity_type, bundle and
    // field name. As such, we need to get a query with these columns. Since
    // this is not a field migration but more a variable migration, and since
    // weight has its own table in D7, we will process the
    // rows generated by this query on a iterator.
    $query = $this->select('weight_settings', 'ws');
    $query->fields('ws', ['type', 'weight_range', 'weight_default', 'sync_translations']);
    $query->condition('ws.weight_enabled', TRUE);
    return $query;
  }

  /**
   * {@inheritdoc}
   */
  public function fields() {
    return [
      'id' => $this->t('The field instance ID.'),
      'entity_type' => $this->t('The entity type.'),
      'bundle' => $this->t('The entity bundle.'),
      'field_name' => $this->t('The field name.'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getIds() {
    return [
      'entity_type' => [
        'type' => 'string',
      ],
      'bundle' => [
        'type' => 'string',
      ],
      'field_name' => [
        'type' => 'string',
      ],
    ];
  }
  
  /**
   * {@inheritdoc}
   */
  protected function initializeIterator() {
    $results = $this->prepareQuery()->execute()->fetchAll();

    // Group all instances by their node bundle.
    $instances = [];
    $entity_type = 'node';
    $field_name = 'field_weight';
    foreach ($results as $result) {
      $instances[] = [
        'id' => $entity_type . '.' . $result['type'] . '.' . $field_name,
        'field_name' => $field_name,
        'entity_type' => $entity_type,
        'bundle' => $result['type'],
        'settings' => [
          'range' => $result['weight_range'],
        ],
        'default_value' => $result['weight_default'],
        'translatable' => $result['sync_translations'],
      ];
    }

    return new \ArrayIterator($instances);
  }

}
