<?php

declare(strict_types=1);

namespace Drupal\Tests\openid_client_advanced\Unit;

use Drupal\openid_client_advanced\Service\JwtSignatureValidator;
use Firebase\JWT\JWT;
use Firebase\JWT\SignatureInvalidException;
use PHPUnit\Framework\TestCase;

/**
 * @coversDefaultClass \Drupal\openid_client_advanced\Service\JwtSignatureValidator
 *
 * @group openid_client_advanced
 */
class JwtSignatureValidatorTest extends TestCase {

  /**
   * Provides a reusable RSA key pair for testing.
   *
   * @var array<string, mixed>
   */
  protected array $keyPair;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->keyPair = $this->generateKeyPair();
  }

  /**
   * @covers ::validate
   */
  public function testValidateWithPemKey(): void {
    $validator = new JwtSignatureValidator();
    $payload = ['sub' => 'example-user', 'nonce' => 'nonce-value'];
    $jwt = JWT::encode($payload, $this->keyPair['private'], 'RS256', $this->keyPair['kid']);

    $decoded = $validator->validate($jwt, [$this->keyPair['public']], ['RS256']);

    $this->assertIsArray($decoded);
    $this->assertSame($payload['sub'], $decoded['sub']);
    $this->assertSame($payload['nonce'], $decoded['nonce']);
  }

  /**
   * @covers ::validate
   */
  public function testValidateWithJwks(): void {
    $validator = new JwtSignatureValidator();
    $payload = ['sub' => 'example-user'];
    $jwt = JWT::encode($payload, $this->keyPair['private'], 'RS256', $this->keyPair['kid']);

    $decoded = $validator->validate($jwt, [$this->keyPair['jwks']], ['RS256']);

    $this->assertIsArray($decoded);
    $this->assertSame($payload['sub'], $decoded['sub']);
  }

  /**
   * @covers ::validate
   */
  public function testValidateDisallowedAlgorithm(): void {
    $validator = new JwtSignatureValidator();
    $payload = ['sub' => 'example-user'];
    $jwt = JWT::encode($payload, $this->keyPair['private'], 'RS256', $this->keyPair['kid']);

    $this->expectException(SignatureInvalidException::class);
    $this->expectExceptionMessage('not allowed');
    $validator->validate($jwt, [$this->keyPair['public']], ['HS256']);
  }

  /**
   * @covers ::validate
   */
  public function testValidateWithJwksWithoutSignatureKeys(): void {
    $validator = new JwtSignatureValidator();
    $payload = ['sub' => 'example-user'];
    $jwt = JWT::encode($payload, $this->keyPair['private'], 'RS256', $this->keyPair['kid']);

    $jwks = json_encode([
      'keys' => [
        [
          'kid' => 'encryption-only',
          'kty' => 'RSA',
          'alg' => 'RSA-OAEP',
          'use' => 'enc',
          'n' => $this->keyPair['n'],
          'e' => $this->keyPair['e'],
        ],
      ],
    ], JSON_UNESCAPED_SLASHES);

    $this->expectException(\UnexpectedValueException::class);
    $this->expectExceptionMessage('No matching signature keys found');
    $validator->validate($jwt, [$jwks], ['RS256']);
  }

  /**
   * Generate an RSA key pair and JWKS representation for testing.
   *
   * @return array<string, mixed>
   *   Key data containing:
   *   private key,
   *   public key,
   *   JWKS JSON,
   *   modulus and exponent.
   */
  protected function generateKeyPair(): array {
    $resource = openssl_pkey_new([
      'private_key_type' => OPENSSL_KEYTYPE_RSA,
      'private_key_bits' => 2048,
    ]);
    if ($resource === FALSE) {
      $this->fail('Unable to generate RSA key pair for tests.');
    }

    $private = '';
    openssl_pkey_export($resource, $private);
    $details = openssl_pkey_get_details($resource);

    $public = $details['key'];
    $kid = 'test-key';

    $n = $this->base64UrlEncode($details['rsa']['n']);
    $e = $this->base64UrlEncode($details['rsa']['e']);

    $jwks = json_encode([
      'keys' => [
        [
          'kid' => $kid,
          'kty' => 'RSA',
          'alg' => 'RS256',
          'use' => 'sig',
          'n' => $n,
          'e' => $e,
        ],
        [
          'kid' => 'enc-key',
          'kty' => 'RSA',
          'alg' => 'RSA-OAEP',
          'use' => 'enc',
          'n' => $n,
          'e' => $e,
        ],
      ],
    ], JSON_UNESCAPED_SLASHES);

    return [
      'private' => $private,
      'public' => $public,
      'jwks' => $jwks,
      'kid' => $kid,
      'n' => $n,
      'e' => $e,
    ];
  }

  /**
   * Base64 URL encodes binary data.
   *
   * @param string $data
   *   Data to encode.
   *
   * @return string
   *   Encoded string.
   */
  protected function base64UrlEncode(string $data): string {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
  }

}
