<?php

declare(strict_types=1);

namespace Drupal\Tests\openid_client_advanced\Unit;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\GeneratedUrl;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\PageCache\ResponsePolicy\KillSwitch;
use Drupal\Core\Url;
use Drupal\Core\Utility\UnroutedUrlAssemblerInterface;
use Drupal\openid_client_advanced\Plugin\OpenIDConnectClient\OpenIDConnectAdvancedClient;
use Drupal\openid_client_advanced\Service\JwtSignatureValidator;
use Drupal\openid_connect\OpenIDConnectAutoDiscover;
use Drupal\openid_connect\OpenIDConnectStateTokenInterface;
use Drupal\Tests\UnitTestCase;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Psr7\Response;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpFoundation\Session\Storage\MockArraySessionStorage;

/**
 * @coversDefaultClass \Drupal\openid_client_advanced\Plugin\OpenIDConnectClient\OpenIDConnectAdvancedClient
 *
 * @group openid_client_advanced
 */
class OpenIDConnectAdvancedClientTest extends UnitTestCase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $container = new ContainerBuilder();
    $container->set('unrouted_url_assembler', new StubUnroutedUrlAssembler());
    \Drupal::setContainer($container);
  }

  /**
   * Tests that a nonce value is generated and stored in the session.
   *
   * @covers ::getUrlOptions
   */
  public function testGetUrlOptionsStoresNonce(): void {
    $session = new Session(new MockArraySessionStorage());
    $request_stack = $this->createRequestStack($session);
    $plugin = $this->createPlugin(['use_nonce' => TRUE], $request_stack);
    $plugin->setParentEntityId('advanced_client');

    $method = new \ReflectionMethod(OpenIDConnectAdvancedClient::class, 'getUrlOptions');
    $method->setAccessible(TRUE);
    $redirect = (new GeneratedUrl())->setGeneratedUrl('https://example.com/callback');
    $options = $method->invoke($plugin, 'openid email', $redirect);

    $this->assertArrayHasKey('nonce', $options['query']);
    $nonce = $options['query']['nonce'];
    $this->assertNotEmpty($nonce);
    $this->assertSame($nonce, $session->get('openid_client_advanced_nonce_advanced_client'));
  }

  /**
   * @covers ::resolveClientSecret
   */
  public function testResolveClientSecretPlainText(): void {
    $session = new Session(new MockArraySessionStorage());
    $plugin = $this->createPlugin(['client_secret' => 'plain-secret'], $this->createRequestStack($session));
    $this->assertSame('plain-secret', $this->invokeResolveClientSecret($plugin));
  }

  /**
   * @covers ::resolveClientSecret
   */
  public function testResolveClientSecretFromFile(): void {
    $session = new Session(new MockArraySessionStorage());
    $file = tempnam(sys_get_temp_dir(), 'oidc_secret_');
    $valid_file = $file . '.txt';
    rename($file, $valid_file);
    $this->assertNotFalse($valid_file);
    file_put_contents($valid_file, "file-secret\n");

    try {
      $plugin = $this->createPlugin(['client_secret' => "file: $valid_file"], $this->createRequestStack($session));
      $this->assertSame('file-secret', $this->invokeResolveClientSecret($plugin));
    }
    finally {
      @unlink($valid_file);
    }
  }

  /**
   * @covers ::resolveClientSecret
   */
  public function testResolveClientSecretDisallowedExtension(): void {
    $session = new Session(new MockArraySessionStorage());
    $file = tempnam(sys_get_temp_dir(), 'oidc_secret_');
    $new_file = $file . '.php';
    rename($file, $new_file);
    $this->assertNotFalse($new_file);
    file_put_contents($new_file, "file-secret\n");

    try {
      $plugin = $this->createPlugin(['client_secret' => "file: $new_file"], $this->createRequestStack($session));
      $this->assertSame('', $this->invokeResolveClientSecret($plugin));
    }
    finally {
      @unlink($new_file);
    }
  }

  /**
   * @covers ::resolveClientSecret
   */
  public function testResolveClientSecretFromEnvironment(): void {
    $session = new Session(new MockArraySessionStorage());
    $env_name = uniqid('OIDC_ADVANCED_SECRET_', FALSE);
    $env_value = 'env-secret';
    putenv($env_name . '=' . $env_value);
    $_ENV[$env_name] = $env_value;
    $_SERVER[$env_name] = $env_value;

    try {
      $plugin = $this->createPlugin(['client_secret' => "env: $env_name"], $this->createRequestStack($session));
      $this->assertSame($env_value, $this->invokeResolveClientSecret($plugin));
    }
    finally {
      putenv($env_name);
      unset($_ENV[$env_name], $_SERVER[$env_name]);
    }
  }

  /**
   * @covers ::resolveClientSecret
   */
  public function testResolveClientSecretIgnoresUnrelatedYaml(): void {
    $session = new Session(new MockArraySessionStorage());
    $raw_secret = "plain: value\nother: item";
    $plugin = $this->createPlugin(['client_secret' => $raw_secret], $this->createRequestStack($session));
    $this->assertSame($raw_secret, $this->invokeResolveClientSecret($plugin));
  }

  /**
   * Ensures request options use the resolved client secret.
   *
   * @covers ::getRequestOptions
   */
  public function testRequestOptionsUseResolvedClientSecret(): void {
    $session = new Session(new MockArraySessionStorage());
    $env_name = uniqid('OIDC_ADVANCED_SECRET_', FALSE);
    $env_value = 'resolved-secret';
    putenv($env_name . '=' . $env_value);
    $_ENV[$env_name] = $env_value;
    $_SERVER[$env_name] = $env_value;

    try {
      $plugin = $this->createPlugin(['client_secret' => "env: $env_name"], $this->createRequestStack($session));
      $method = new \ReflectionMethod(OpenIDConnectAdvancedClient::class, 'getRequestOptions');
      $method->setAccessible(TRUE);
      $options = $method->invoke($plugin, 'auth-code', 'https://example.com/callback');
      $this->assertSame($env_value, $options['form_params']['client_secret']);
    }
    finally {
      putenv($env_name);
      unset($_ENV[$env_name], $_SERVER[$env_name]);
    }
  }

  /**
   * Tests that token retrieval.
   *
   * @covers ::retrieveTokens
   */
  public function testRetrieveTokensWithMatchingNonce(): void {
    $session = new Session(new MockArraySessionStorage());
    $nonce = 'matching-nonce';
    $session->set('openid_client_advanced_nonce_advanced_client', $nonce);

    $http_client = new StubHttpClient([
      new Response(200, [], Json::encode([
        'id_token' => $this->createUnsignedToken([
          'sub' => 'user-1',
          'nonce' => $nonce,
        ]),
        'access_token' => 'access-token',
        'expires_in' => 3600,
      ])),
    ]);

    $plugin = $this->createPlugin(['use_nonce' => TRUE], $this->createRequestStack($session), $http_client);
    $plugin->setParentEntityId('advanced_client');

    $tokens = $plugin->retrieveTokens('code-123');

    $this->assertIsArray($tokens);
    $this->assertSame('access-token', $tokens['access_token']);
    $this->assertSame(4600, $tokens['expire']);
    $payload = Json::decode($this->extractPayload($tokens['id_token']));
    $this->assertSame($nonce, $payload['nonce']);
    $this->assertNull($session->get('openid_client_advanced_nonce_advanced_client'));
  }

  /**
   * Tests that token retrieval fails.
   *
   * @covers ::retrieveTokens
   */
  public function testRetrieveTokensWithMismatchedNonce(): void {
    $session = new Session(new MockArraySessionStorage());
    $session->set('openid_client_advanced_nonce_advanced_client', 'expected-nonce');

    $http_client = new StubHttpClient([
      new Response(200, [], Json::encode([
        'id_token' => $this->createUnsignedToken([
          'sub' => 'user-1',
          'nonce' => 'other-nonce',
        ]),
        'access_token' => 'access-token',
        'expires_in' => 3600,
      ])),
    ]);

    $plugin = $this->createPlugin(['use_nonce' => TRUE], $this->createRequestStack($session), $http_client);
    $plugin->setParentEntityId('advanced_client');

    $this->assertNull($plugin->retrieveTokens('code-123'));
    $this->assertNull($session->get('openid_client_advanced_nonce_advanced_client'));
  }

  /**
   * Tests the URL validation logic.
   *
   * @covers ::validateUrl
   */
  public function testValidateUrl(): void {
    $session = new Session(new MockArraySessionStorage());
    $plugin = $this->createPlugin([], $this->createRequestStack($session));
    $method = new \ReflectionMethod(OpenIDConnectAdvancedClient::class, 'validateUrl');
    $method->setAccessible(TRUE);

    // Public IP (Google DNS) - Should pass.
    $this->assertTrue($method->invoke($plugin, 'https://8.8.8.8'));

    // Localhost - Should fail.
    $this->assertFalse($method->invoke($plugin, 'http://localhost'));

    // Loopback IP - Should fail.
    $this->assertFalse($method->invoke($plugin, 'http://127.0.0.1'));

    // Private IP (10.x.x.x) - Should fail.
    $this->assertFalse($method->invoke($plugin, 'http://10.0.0.1'));

    // Private IP (192.168.x.x) - Should fail.
    $this->assertFalse($method->invoke($plugin, 'http://192.168.1.1'));
  }

  /**
   * Ensures private hosts can be explicitly permitted.
   *
   * @covers ::validateUrl
   */
  public function testValidateUrlAllowsPrivateHostsWhenEnabled(): void {
    $session = new Session(new MockArraySessionStorage());
    $plugin = $this->createPlugin(['allow_private_issuer' => TRUE], $this->createRequestStack($session));
    $plugin->setDnsRecords('localhost', [['ip' => '127.0.0.1', 'type' => 'A']]);

    $method = new \ReflectionMethod(OpenIDConnectAdvancedClient::class, 'validateUrl');
    $method->setAccessible(TRUE);

    $this->assertTrue($method->invoke($plugin, 'http://localhost'));
  }

  /**
   * Tests the URL validation logic with IPv6.
   *
   * @covers ::validateUrl
   */
  public function testValidateUrlIpv6(): void {
    $session = new Session(new MockArraySessionStorage());
    $plugin = $this->createPlugin([], $this->createRequestStack($session));
    $method = new \ReflectionMethod(OpenIDConnectAdvancedClient::class, 'validateUrl');
    $method->setAccessible(TRUE);

    // Public IPv6 (Google DNS) - Should pass.
    $plugin->setDnsRecords('2001:4860:4860::8888', [['ipv6' => '2001:4860:4860::8888', 'type' => 'AAAA']]);
    $this->assertTrue($method->invoke($plugin, 'https://[2001:4860:4860::8888]'));

    // Localhost IPv6 - Should fail.
    $plugin->setDnsRecords('::1', [['ipv6' => '::1', 'type' => 'AAAA']]);
    $this->assertFalse($method->invoke($plugin, 'http://[::1]'));

    // Private IPv6 (fc00::/7) - Should fail.
    $plugin->setDnsRecords('fc00::1', [['ipv6' => 'fc00::1', 'type' => 'AAAA']]);
    $this->assertFalse($method->invoke($plugin, 'http://[fc00::1]'));

    // IPv4-mapped IPv6 localhost (::ffff:127.0.0.1) - Should fail.
    // If this passes, it's a bypass.
    $plugin->setDnsRecords('::ffff:127.0.0.1', [['ipv6' => '::ffff:127.0.0.1', 'type' => 'AAAA']]);
    $this->assertFalse($method->invoke($plugin, 'http://[::ffff:127.0.0.1]'));
  }

  /**
   * Tests that validateUrl enforces HTTP/HTTPS schemes.
   */
  public function testValidateUrlScheme(): void {
    $session = new Session(new MockArraySessionStorage());
    $plugin = $this->createPlugin([], $this->createRequestStack($session));
    $method = new \ReflectionMethod(OpenIDConnectAdvancedClient::class, 'validateUrl');
    $method->setAccessible(TRUE);

    // FTP scheme - Should fail (but currently might pass if host is valid).
    $plugin->setDnsRecords('example.com', [['ip' => '93.184.216.34', 'type' => 'A']]);
    $this->assertFalse($method->invoke($plugin, 'ftp://example.com/file'), 'FTP scheme should be rejected.');

    // No scheme - Should fail.
    $this->assertFalse($method->invoke($plugin, 'example.com/file'), 'Missing scheme should be rejected.');
  }

  /**
   * Tests auto-discovery with DNS rebinding protection.
   *
   * @covers ::autoDiscoverEndpoints
   */
  public function testAutoDiscoverEndpoints(): void {
    $session = new Session(new MockArraySessionStorage());
    $http_client = new StubHttpClient([
      new Response(200, [], Json::encode([
        'authorization_endpoint' => 'https://example.com/auth',
        'token_endpoint' => 'https://example.com/token',
        'userinfo_endpoint' => 'https://example.com/userinfo',
      ])),
    ]);

    $plugin = $this->createPlugin([], $this->createRequestStack($session), $http_client);
    $plugin->setParentEntityId('advanced_client');

    // Mock DNS for example.com.
    $plugin->setDnsRecords('example.com', [['ip' => '93.184.216.34', 'type' => 'A']]);

    $method = new \ReflectionMethod(OpenIDConnectAdvancedClient::class, 'autoDiscoverEndpoints');
    $method->setAccessible(TRUE);

    $result = $method->invoke($plugin, 'https://example.com');

    $this->assertIsArray($result);
    $this->assertSame('https://example.com/auth', $result['authorization_endpoint']);

    // Verify that the request used the resolved IP to prevent rebinding.
    $this->assertCount(1, $http_client->requests);
    $request = $http_client->requests[0];
    $this->assertSame('GET', $request[0]);
    $this->assertSame('https://example.com/.well-known/openid-configuration', $request[1]);
    // Check for the resolve option we intend to add.
    $this->assertArrayHasKey('resolve', $request[2]);
    $this->assertSame('93.184.216.34', $request[2]['resolve']['example.com']);
  }

  /**
   * Tests auto-discovery fails with private IP.
   *
   * @covers ::autoDiscoverEndpoints
   */
  public function testAutoDiscoverEndpointsPrivateIp(): void {
    $session = new Session(new MockArraySessionStorage());
    $http_client = new StubHttpClient([]);
    $plugin = $this->createPlugin([], $this->createRequestStack($session), $http_client);

    // Mock DNS for private.example.com.
    $plugin->setDnsRecords('private.example.com', [['ip' => '192.168.1.1', 'type' => 'A']]);

    $method = new \ReflectionMethod(OpenIDConnectAdvancedClient::class, 'autoDiscoverEndpoints');
    $method->setAccessible(TRUE);

    $result = $method->invoke($plugin, 'https://private.example.com');
    $this->assertFalse($result);
  }

  /**
   * Ensures private hosts can be used when explicitly allowed.
   *
   * @covers ::autoDiscoverEndpoints
   */
  public function testAutoDiscoverEndpointsAllowsPrivateHosts(): void {
    $session = new Session(new MockArraySessionStorage());
    $http_client = new StubHttpClient([
      new Response(200, [], Json::encode([
        'authorization_endpoint' => 'https://example.com/auth',
        'token_endpoint' => 'https://example.com/token',
        'userinfo_endpoint' => 'https://example.com/userinfo',
      ])),
    ]);

    $plugin = $this->createPlugin(['allow_private_issuer' => TRUE], $this->createRequestStack($session), $http_client);
    $plugin->setParentEntityId('advanced_client_private');
    $plugin->setDnsRecords('localhost', [['ip' => '127.0.0.1', 'type' => 'A']]);

    $method = new \ReflectionMethod(OpenIDConnectAdvancedClient::class, 'autoDiscoverEndpoints');
    $method->setAccessible(TRUE);

    $result = $method->invoke($plugin, 'http://localhost:8888');

    $this->assertIsArray($result);
    $this->assertSame('https://example.com/auth', $result['authorization_endpoint']);

    $this->assertCount(1, $http_client->requests);
    $request = $http_client->requests[0];
    $this->assertSame('GET', $request[0]);
    $this->assertSame('http://localhost:8888/.well-known/openid-configuration', $request[1]);
    $this->assertSame(['localhost' => '127.0.0.1'], $request[2]['resolve']);
  }

  /**
   * Tests that auto-discovery does not follow redirects.
   *
   * @covers ::autoDiscoverEndpoints
   */
  public function testAutoDiscoverEndpointsRedirect(): void {
    $session = new Session(new MockArraySessionStorage());
    $http_client = new StubHttpClient([
      new Response(302, ['Location' => 'http://localhost/metadata']),
    ]);
    $plugin = $this->createPlugin([], $this->createRequestStack($session), $http_client);

    $plugin = $this->createPlugin([], $this->createRequestStack($session), $http_client);
    $plugin->setParentEntityId('advanced_client_redirect');

    // Mock DNS for redirect.example.com.
    $plugin->setDnsRecords('redirect.example.com', [['ip' => '93.184.216.34', 'type' => 'A']]);

    $method = new \ReflectionMethod(OpenIDConnectAdvancedClient::class, 'autoDiscoverEndpoints');
    $method->setAccessible(TRUE);

    // Use a unique URL to bypass static cache from previous tests.
    $method->invoke($plugin, 'https://redirect.example.com');

    $this->assertCount(1, $http_client->requests);
    $request = $http_client->requests[0];
    // Check that allow_redirects is set to FALSE.
    $this->assertArrayHasKey('allow_redirects', $request[2]);
    $this->assertFalse($request[2]['allow_redirects']);
  }

  /**
   * Builds a plugin instance with common dependencies replaced by mocks.
   *
   * @param array $configuration
   *   Configuration overrides for the plugin.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   Request stack providing a session.
   * @param \GuzzleHttp\ClientInterface|null $http_client
   *   Optional HTTP client mock.
   *
   * @return \Drupal\Tests\openid_client_advanced\Unit\TestableOpenIDConnectAdvancedClient
   *   Configured plugin instance.
   */
  protected function createPlugin(array $configuration, RequestStack $request_stack, ?ClientInterface $http_client = NULL): TestableOpenIDConnectAdvancedClient {
    $http_client = $http_client ?: new StubHttpClient([]);

    $logger = $this->createMock(LoggerInterface::class);
    $logger_factory = $this->createMock(LoggerChannelFactoryInterface::class);
    $logger_factory->method('get')->willReturn($logger);

    $time = $this->createMock(TimeInterface::class);
    $time->method('getRequestTime')->willReturn(1000);

    $language = $this->createMock(LanguageInterface::class);
    $language->method('getId')->willReturn(LanguageInterface::LANGCODE_NOT_APPLICABLE);
    $language_manager = $this->createMock(LanguageManagerInterface::class);
    $language_manager->method('getLanguage')->willReturn($language);

    $state_token = $this->createMock(OpenIDConnectStateTokenInterface::class);
    $state_token->method('generateToken')->willReturn('state-token');

    $auto_discover = $this->createMock(OpenIDConnectAutoDiscover::class);

    $defaults = [
      'client_id' => 'client-id',
      'client_secret' => 'client-secret',
      'iss_allowed_domains' => '',
      'issuer_url' => 'https://issuer.example.com',
      'allow_private_issuer' => FALSE,
      'authorization_endpoint' => 'https://example.com/auth',
      'token_endpoint' => 'https://example.com/token',
      'userinfo_endpoint' => 'https://example.com/userinfo',
      'end_session_endpoint' => 'https://example.com/logout',
      'scopes' => ['openid', 'email'],
      'use_nonce' => FALSE,
      'validate_signature' => FALSE,
      'allowed_algorithms' => ['RS256'],
      'idp_public_keys' => [],
    ];

    $config = $configuration + $defaults;

    return new TestableOpenIDConnectAdvancedClient(
      $config,
      'advanced',
      [],
      $request_stack,
      $http_client,
      $logger_factory,
      $time,
      new KillSwitch(),
      $language_manager,
      $state_token,
      $auto_discover,
      new JwtSignatureValidator()
    );
  }

  /**
   * Creates a request stack with a session-enabled request.
   *
   * @param \Symfony\Component\HttpFoundation\Session\Session $session
   *   Session instance.
   *
   * @return \Symfony\Component\HttpFoundation\RequestStack
   *   Populated request stack.
   */
  protected function createRequestStack(Session $session): RequestStack {
    $request = new Request();
    $request->setSession($session);
    $stack = new RequestStack();
    $stack->push($request);
    return $stack;
  }

  /**
   * Invokes the protected resolveClientSecret() method on the plugin.
   *
   * @param \Drupal\openid_client_advanced\Plugin\OpenIDConnectClient\OpenIDConnectAdvancedClient $plugin
   *   The plugin instance.
   *
   * @return string
   *   The resolved client secret.
   *
   * @throws \ReflectionException
   */
  protected function invokeResolveClientSecret(OpenIDConnectAdvancedClient $plugin): string {
    $method = new \ReflectionMethod(OpenIDConnectAdvancedClient::class, 'resolveClientSecret');
    $method->setAccessible(TRUE);
    return (string) $method->invoke($plugin);
  }

  /**
   * Creates an unsigned JWT-like string for nonce validation tests.
   *
   * @param array $payload
   *   Payload data.
   *
   * @return string
   *   Encoded token string.
   */
  protected function createUnsignedToken(array $payload): string {
    $header = $this->base64UrlEncode(Json::encode(['alg' => 'none']));
    $body = $this->base64UrlEncode(Json::encode($payload));
    return $header . '.' . $body . '.signature';
  }

  /**
   * Extracts the payload segment from a JWT.
   *
   * @param string $token
   *   Token string.
   *
   * @return string
   *   JSON payload.
   */
  protected function extractPayload(string $token): string {
    $parts = explode('.', $token);
    return base64_decode(strtr($parts[1], '-_', '+/'), TRUE) ?: '';
  }

  /**
   * Encodes data using base64-url alphabet.
   *
   * @param string $value
   *   Value to encode.
   *
   * @return string
   *   Encoded value.
   */
  protected function base64UrlEncode(string $value): string {
    return rtrim(strtr(base64_encode($value), '+/', '-_'), '=');
  }

}

/**
 * Lightweight HTTP client stub for token retrieval tests.
 */
class StubHttpClient implements ClientInterface {

  /**
   * Queue of responses to return for POST/REQUEST operations.
   *
   * @var \Psr\Http\Message\ResponseInterface[]
   */
  protected array $responses;

  /**
   * Captured requests performed through the stub.
   *
   * @var array<int, array>
   */
  public array $requests = [];

  /**
   * Constructs a new stub HTTP client.
   *
   * @param array $responses
   *   Responses to return for subsequent requests.
   */
  public function __construct(array $responses) {
    $this->responses = $responses;
  }

  /**
   * Convenience wrapper matching the real client's magic method.
   */
  public function post($uri, array $options = []): ResponseInterface {
    return $this->request('POST', $uri, $options);
  }

  /**
   * {@inheritdoc}
   */
  public function request(string $method, $uri, array $options = []): ResponseInterface {
    $this->requests[] = [$method, $uri, $options];
    if (empty($this->responses)) {
      throw new \RuntimeException('No queued responses available.');
    }
    return array_shift($this->responses);
  }

  /**
   * {@inheritdoc}
   */
  public function send(RequestInterface $request, array $options = []): ResponseInterface {
    throw new \BadMethodCallException('Not implemented in stub.');
  }

  /**
   * {@inheritdoc}
   */
  public function sendAsync(RequestInterface $request, array $options = []): PromiseInterface {
    throw new \BadMethodCallException('Not implemented in stub.');
  }

  /**
   * {@inheritdoc}
   */
  public function requestAsync(string $method, $uri, array $options = []): PromiseInterface {
    throw new \BadMethodCallException('Not implemented in stub.');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(?string $option = NULL) {
    return NULL;
  }

}

/**
 * Simple unrouted URL assembler stub returning raw URLs.
 */
class StubUnroutedUrlAssembler implements UnroutedUrlAssemblerInterface {

  /**
   * {@inheritdoc}
   */
  public function assemble($uri, array $options = [], $collect_bubbleable_metadata = FALSE) {
    $query = '';
    if (!empty($options['query'])) {
      $query = '?' . http_build_query($options['query']);
    }
    $fragment = '';
    if (!empty($options['fragment'])) {
      $fragment = '#' . $options['fragment'];
    }
    $assembled = $uri . $query . $fragment;
    if ($collect_bubbleable_metadata) {
      $generated = new GeneratedUrl();
      $generated->setGeneratedUrl($assembled);
      return $generated;
    }
    return $assembled;
  }

}

/**
 * Simple subclass to override redirect URL generation for tests.
 */
class TestableOpenIDConnectAdvancedClient extends OpenIDConnectAdvancedClient {

  /**
   * Redirect URI used during token retrieval.
   *
   * @var string
   */
  protected string $redirectUri = 'https://example.com/callback';

  /**
   * Mocked DNS records.
   *
   * @var array
   */
  protected array $dnsRecords = [];

  /**
   * Sets mocked DNS records for a host.
   *
   * @param string $host
   *   The hostname.
   * @param array $records
   *   Array of records (e.g. [['ip' => '1.2.3.4', 'type' => 'A']]).
   */
  public function setDnsRecords(string $host, array $records): void {
    $this->dnsRecords[$host] = $records;
  }

  /**
   * {@inheritdoc}
   */
  protected function getDnsRecords(string $host): array {
    return $this->dnsRecords[$host] ?? [];
  }

  /**
   * Allows tests to override the redirect URI.
   *
   * @param string $uri
   *   The redirect URI to use.
   */
  public function setRedirectUri(string $uri): void {
    $this->redirectUri = $uri;
  }

  /**
   * {@inheritdoc}
   */
  protected function getRedirectUrl(array $route_parameters = [], array $options = []): Url {
    return Url::fromUri($this->redirectUri);
  }

}
