<?php

namespace Drupal\openid_client_advanced\Plugin\OpenIDConnectClient;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Component\Serialization\Yaml;
use Drupal\Component\Utility\Crypt;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\GeneratedUrl;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\PageCache\ResponsePolicy\KillSwitch;
use Drupal\openid_client_advanced\Service\JwtSignatureValidator;
use Drupal\openid_connect\OpenIDConnectAutoDiscover;
use Drupal\openid_connect\OpenIDConnectStateTokenInterface;
use Drupal\openid_connect\Plugin\OpenIDConnectClientBase;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * OAuth 2.0 client with signature validation and nonce support.
 *
 * @OpenIDConnectClient(
 *   id = "advanced",
 *   label = @Translation("OAuth 2.0 Advanced")
 * )
 */
class OpenIDConnectAdvancedClient extends OpenIDConnectClientBase {

  /**
   * JWT signature validator service.
   *
   * @var \Drupal\openid_client_advanced\Service\JwtSignatureValidator
   */
  protected JwtSignatureValidator $jwtValidator;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messengerService;

  /**
   * Constructs a new advanced client instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin identifier.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Component\Datetime\TimeInterface $datetime_time
   *   The datetime.time service.
   * @param \Drupal\Core\PageCache\ResponsePolicy\KillSwitch $page_cache_kill_switch
   *   The page cache kill switch.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\openid_connect\OpenIDConnectStateTokenInterface $state_token
   *   The OpenID Connect state token service.
   * @param \Drupal\openid_connect\OpenIDConnectAutoDiscover $auto_discover
   *   The OpenID Connect auto-discovery service.
   * @param \Drupal\openid_client_advanced\Service\JwtSignatureValidator $jwt_validator
   *   The JWT signature validator service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(array $configuration, string $plugin_id, $plugin_definition, RequestStack $request_stack, ClientInterface $http_client, LoggerChannelFactoryInterface $logger_factory, TimeInterface $datetime_time, KillSwitch $page_cache_kill_switch, LanguageManagerInterface $language_manager, OpenIDConnectStateTokenInterface $state_token, OpenIDConnectAutoDiscover $auto_discover, JwtSignatureValidator $jwt_validator, MessengerInterface $messenger) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $request_stack, $http_client, $logger_factory, $datetime_time, $page_cache_kill_switch, $language_manager, $state_token, $auto_discover);
    $this->jwtValidator = $jwt_validator;
    $this->messengerService = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('request_stack'),
      $container->get('http_client'),
      $container->get('logger.factory'),
      $container->get('datetime.time'),
      $container->get('page_cache_kill_switch'),
      $container->get('language_manager'),
      $container->get('openid_connect.state_token'),
      $container->get('openid_connect.autodiscover'),
      $container->get('openid_client_advanced.jwt_signature_validator'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'issuer_url' => '',
      'allow_private_issuer' => FALSE,
      'authorization_endpoint' => '',
      'token_endpoint' => '',
      'userinfo_endpoint' => '',
      'end_session_endpoint' => '',
      'scopes' => ['openid', 'email'],
      'use_nonce' => FALSE,
      'validate_signature' => FALSE,
      'allowed_algorithms' => ['RS256'],
      'idp_public_keys' => [],
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    if (isset($form['client_secret'])) {
      $form['client_secret']['#description'] = $this->t('Enter the client secret directly or provide YAML such as "file: path/to/secret" or "env: VARIABLE_NAME" to load it from a file or environment variable.');
    }

    $form['use_well_known'] = [
      '#title' => $this->t('Auto discover endpoints'),
      '#type' => 'checkbox',
      '#description' => $this->t(
        'Requires IDP support for "<a href="@url" target="_blank">OpenID Connect Discovery</a>".',
        ['@url' => 'https://openid.net/specs/openid-connect-discovery-1_0.html']
      ),
      '#default_value' => !empty($this->configuration['issuer_url']),
    ];

    $form['issuer_url'] = [
      '#title' => $this->t('Issuer URL'),
      '#type' => 'url',
      '#default_value' => $this->configuration['issuer_url'],
      '#states' => [
        'visible' => [':input[name="settings[use_well_known]"]' => ['checked' => TRUE]],
      ],
    ];

    $form['allow_private_issuer'] = [
      '#title' => $this->t('Allow private issuer hosts'),
      '#type' => 'checkbox',
      '#default_value' => !empty($this->configuration['allow_private_issuer']),
      '#description' => $this->t('Permit auto-discovery against issuers that resolve to private or loopback IP addresses (for example, localhost). Enable only for trusted development providers and never on production sites.'),
      '#states' => [
        'visible' => [':input[name="settings[use_well_known]"]' => ['checked' => TRUE]],
      ],
    ];

    $form['authorization_endpoint'] = [
      '#title' => $this->t('Authorization endpoint'),
      '#type' => 'url',
      '#default_value' => $this->configuration['authorization_endpoint'],
      '#states' => [
        'visible' => [':input[name="settings[use_well_known]"]' => ['checked' => FALSE]],
      ],
    ];

    $form['token_endpoint'] = [
      '#title' => $this->t('Token endpoint'),
      '#type' => 'url',
      '#default_value' => $this->configuration['token_endpoint'],
      '#states' => [
        'visible' => [':input[name="settings[use_well_known]"]' => ['checked' => FALSE]],
      ],
    ];

    $form['userinfo_endpoint'] = [
      '#title' => $this->t('UserInfo endpoint'),
      '#type' => 'url',
      '#default_value' => $this->configuration['userinfo_endpoint'],
      '#states' => [
        'visible' => [':input[name="settings[use_well_known]"]' => ['checked' => FALSE]],
      ],
    ];

    $form['end_session_endpoint'] = [
      '#title' => $this->t('End Session endpoint'),
      '#type' => 'url',
      '#default_value' => $this->configuration['end_session_endpoint'],
      '#states' => [
        'visible' => [':input[name="settings[use_well_known]"]' => ['checked' => FALSE]],
      ],
    ];

    $form['scopes'] = [
      '#title' => $this->t('Scopes'),
      '#type' => 'textfield',
      '#description' => $this->t('Custom scopes, separated by spaces, for example: openid email'),
      '#default_value' => implode(' ', $this->configuration['scopes']),
    ];

    $form['validate_signature'] = [
      '#title' => $this->t('Validate ID token signature'),
      '#type' => 'checkbox',
      '#default_value' => !empty($this->configuration['validate_signature']),
      '#description' => $this->t('Verify ID token signatures using the configured public keys.'),
    ];

    $form['allowed_algorithms'] = [
      '#title' => $this->t('Allowed signature algorithms'),
      '#type' => 'textfield',
      '#default_value' => implode(' ', $this->configuration['allowed_algorithms']),
      '#description' => $this->t('Space separated list of allowed algorithms, e.g. RS256 RS512.'),
      '#states' => [
        'visible' => [':input[name="settings[validate_signature]"]' => ['checked' => TRUE]],
      ],
    ];

    $form['idp_public_keys'] = [
      '#title' => $this->t('Identity Provider public keys'),
      '#type' => 'textarea',
      '#default_value' => implode("\n\n", $this->configuration['idp_public_keys']),
      '#description' => $this->t('Provide PEM encoded public keys separated by a blank line, or paste the JWKS JSON returned by the Identity Provider jwks_uri endpoint.'),
      '#states' => [
        'visible' => [':input[name="settings[validate_signature]"]' => ['checked' => TRUE]],
      ],
    ];

    $form['use_nonce'] = [
      '#title' => $this->t('Send nonce parameter'),
      '#type' => 'checkbox',
      '#default_value' => !empty($this->configuration['use_nonce']),
      '#description' => $this->t('Attach a nonce to authorization requests and validate it in the ID token response.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);

    $configuration = $form_state->getValues();
    if (!empty($configuration['use_well_known'])) {
      $allow_private = !empty($configuration['allow_private_issuer']);
      $endpoints = $this->autoDiscoverEndpoints($configuration['issuer_url'], $allow_private);
      if ($endpoints === FALSE) {
        $form_state->setErrorByName('issuer_url', $this->t('The issuer URL @url appears to be invalid.', ['@url' => $configuration['issuer_url']]));
      }
    }

    $raw_public_keys = $configuration['idp_public_keys'] ?? '';
    $normalized_public_keys = [];
    if (trim((string) $raw_public_keys) !== '') {
      try {
        $normalized_public_keys = $this->normalizePublicKeys($raw_public_keys);
      }
      catch (\InvalidArgumentException $exception) {
        $form_state->setErrorByName('idp_public_keys', $this->t('Invalid public keys: @message', ['@message' => $exception->getMessage()]));
      }
    }
    $form_state->set('normalized_public_keys', $normalized_public_keys);

    $algorithms = $this->normalizeAlgorithms($configuration['allowed_algorithms'] ?? '');
    $form_state->set('normalized_allowed_algorithms', $algorithms);

    if (!empty($configuration['validate_signature'])) {
      if (empty($normalized_public_keys)) {
        $form_state->setErrorByName('idp_public_keys', $this->t('At least one public key is required for signature validation.'));
      }
      if (empty($algorithms)) {
        $form_state->setErrorByName('allowed_algorithms', $this->t('Specify at least one allowed signature algorithm.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $configuration = $form_state->getValues();

    $endpoints = [
      'authorization_endpoint' => $configuration['authorization_endpoint'],
      'token_endpoint' => $configuration['token_endpoint'],
      'userinfo_endpoint' => $configuration['userinfo_endpoint'],
    ];

    if (!empty($configuration['use_well_known'])) {
      $discovered = $this->autoDiscoverEndpoints($configuration['issuer_url'], !empty($configuration['allow_private_issuer']));
      if (is_array($discovered)) {
        $endpoints = [
          'authorization_endpoint' => $discovered['authorization_endpoint'],
          'token_endpoint' => $discovered['token_endpoint'],
          'userinfo_endpoint' => $discovered['userinfo_endpoint'],
        ];
      }
    }

    $this->unsetConfigurationKeys(['use_well_known']);

    if (!empty($configuration['scopes'])) {
      $this->setConfiguration(['scopes' => preg_split('/\s+/', trim($configuration['scopes']))]);
    }

    $public_keys = $form_state->get('normalized_public_keys');
    if (!is_array($public_keys)) {
      $public_keys = $this->normalizePublicKeys($configuration['idp_public_keys'] ?? '');
    }

    $algorithms = $form_state->get('normalized_allowed_algorithms');
    if (!is_array($algorithms)) {
      $algorithms = $this->normalizeAlgorithms($configuration['allowed_algorithms'] ?? '');
    }

    $this->setConfiguration([
      'validate_signature' => !empty($configuration['validate_signature']),
      'use_nonce' => !empty($configuration['use_nonce']),
      'allow_private_issuer' => !empty($configuration['allow_private_issuer']),
      'idp_public_keys' => $public_keys,
      'allowed_algorithms' => $algorithms,
      'issuer_url' => $configuration['issuer_url'],
      'authorization_endpoint' => $endpoints['authorization_endpoint'],
      'token_endpoint' => $endpoints['token_endpoint'],
      'userinfo_endpoint' => $endpoints['userinfo_endpoint'],
      'end_session_endpoint' => $configuration['end_session_endpoint'],
    ]);

    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function getClientScopes(): ?array {
    return $this->configuration['scopes'];
  }

  /**
   * {@inheritdoc}
   */
  protected function getUrlOptions(string $scope, GeneratedUrl $redirect_uri): array {
    $options = parent::getUrlOptions($scope, $redirect_uri);

    if (!empty($this->configuration['use_nonce'])) {
      $nonce = Crypt::randomBytesBase64(32);
      $this->storeNonce($nonce);
      $options['query']['nonce'] = $nonce;
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  protected function getRequestOptions(string $authorization_code, string $redirect_uri): array {
    $request_options = parent::getRequestOptions($authorization_code, $redirect_uri);
    $request_options['form_params']['client_secret'] = $this->resolveClientSecret();
    return $request_options;
  }

  /**
   * Resolves the configured client secret.
   *
   * Supports plain text, file, or environment-based secrets.
   *
   * @return string
   *   The resolved client secret or an empty string on failure.
   */
  protected function resolveClientSecret(): string {
    $raw_secret = $this->configuration['client_secret'] ?? '';

    if (!is_string($raw_secret) || $raw_secret === '') {
      return '';
    }

    try {
      $decoded_secret = Yaml::decode($raw_secret);
    }
    catch (\Exception $exception) {
      if ($this->startsWithSecretDirective($raw_secret)) {
        $this->loggerFactory->get('openid_connect_' . $this->pluginId)
          ->error('Unable to parse client secret directive: @message', ['@message' => $exception->getMessage()]);
        return '';
      }
      return $raw_secret;
    }

    if (!is_array($decoded_secret)) {
      return $raw_secret;
    }

    $directive_keys = array_intersect(['file', 'env'], array_keys($decoded_secret));
    if (count($directive_keys) !== 1 || count($decoded_secret) !== 1) {
      return $raw_secret;
    }

    $directive = reset($directive_keys);
    $source = $decoded_secret[$directive];

    if (!is_string($source) || trim($source) === '') {
      $this->loggerFactory->get('openid_connect_' . $this->pluginId)
        ->error('Client secret directive "@directive" must provide a non-empty string value.', ['@directive' => $directive]);
      return '';
    }

    if ($directive === 'file') {
      return $this->readSecretFromFile($source);
    }

    if ($directive === 'env') {
      return $this->readSecretFromEnvironment($source);
    }

    return $raw_secret;
  }

  /**
   * Reads a secret from a file on disk.
   *
   * @param mixed $source
   *   The decoded YAML value for the file key.
   *
   * @return string
   *   The file contents or an empty string on failure.
   */
  protected function readSecretFromFile(mixed $source): string {
    $path = is_string($source) ? trim($source) : '';
    if ($path === '') {
      $this->loggerFactory->get('openid_connect_' . $this->pluginId)
        ->error('Client secret configuration specifies a file path, but no path was provided.');
      return '';
    }

    $candidate_paths = [$path];
    if (!$this->isAbsolutePath($path) && defined('DRUPAL_ROOT')) {
      $candidate_paths[] = DRUPAL_ROOT . DIRECTORY_SEPARATOR . ltrim($path, DIRECTORY_SEPARATOR);
    }

    $resolved_path = NULL;
    foreach ($candidate_paths as $candidate) {
      if (@is_readable($candidate) && @is_file($candidate)) {
        $resolved_path = $candidate;
        break;
      }
    }

    if ($resolved_path === NULL) {
      $this->loggerFactory->get('openid_connect_' . $this->pluginId)
        ->error('Client secret file "@path" is not readable.', ['@path' => $path]);
      return '';
    }

    // Security check: prevent reading arbitrary system files or PHP files.
    $extension = pathinfo($resolved_path, PATHINFO_EXTENSION);
    $allowed_extensions = ['txt', 'secret', 'key', 'json'];
    if (!in_array(strtolower($extension), $allowed_extensions, TRUE) || strtolower($extension) === 'php') {
      $this->loggerFactory->get('openid_connect_' . $this->pluginId)
        ->error('Client secret file "@path" has a disallowed extension. Allowed extensions: @allowed', [
          '@path' => $resolved_path,
          '@allowed' => implode(', ', $allowed_extensions),
        ]);
      return '';
    }

    // Limit the read to 4KB to prevent memory exhaustion (DoS).
    $contents = @file_get_contents($resolved_path, FALSE, NULL, 0, 4096);
    if ($contents === FALSE) {
      $this->loggerFactory->get('openid_connect_' . $this->pluginId)
        ->error('Failed to read client secret file "@path".', ['@path' => $resolved_path]);
      return '';
    }

    return rtrim($contents, "\r\n");
  }

  /**
   * Checks if the raw secret looks like a supported directive.
   *
   * @param string $raw_secret
   *   The raw client secret value.
   *
   * @return bool
   *   TRUE when the value starts with a known directive, FALSE otherwise.
   */
  protected function startsWithSecretDirective(string $raw_secret): bool {
    return (bool) preg_match('/^\s*(file|env)\s*:/i', $raw_secret);
  }

  /**
   * Reads the secret value from an environment variable.
   *
   * @param mixed $source
   *   The decoded YAML value for the env key.
   *
   * @return string
   *   The environment variable value or an empty string.
   */
  protected function readSecretFromEnvironment(mixed $source): string {
    $variable = is_string($source) ? trim($source) : '';
    if ($variable === '') {
      $this->loggerFactory->get('openid_connect_' . $this->pluginId)
        ->error('Client secret configuration specifies an environment variable, but no name was provided.');
      return '';
    }

    $value = getenv($variable);
    if ($value === FALSE) {
      $value = $_ENV[$variable] ?? $_SERVER[$variable] ?? FALSE;
    }

    if ($value === FALSE) {
      $this->loggerFactory->get('openid_connect_' . $this->pluginId)
        ->error('Client secret environment variable "@name" is not set.', ['@name' => $variable]);
      return '';
    }

    return (string) $value;
  }

  /**
   * Checks if the given path is absolute.
   *
   * @param string $path
   *   The path to evaluate.
   *
   * @return bool
   *   TRUE if the path is absolute, FALSE otherwise.
   */
  protected function isAbsolutePath(string $path): bool {
    if ($path === '') {
      return FALSE;
    }

    if (str_starts_with($path, DIRECTORY_SEPARATOR)) {
      return TRUE;
    }

    return (bool) preg_match('/^[A-Z]:[\\\\\\/]/i', $path);
  }

  /**
   * {@inheritdoc}
   */
  public function retrieveTokens(string $authorization_code): ?array {
    $tokens = parent::retrieveTokens($authorization_code);
    if (!$tokens) {
      if (!empty($this->configuration['use_nonce'])) {
        $this->clearNonce();
      }
      return NULL;
    }

    $id_token = $tokens['id_token'] ?? NULL;
    $expected_nonce = NULL;
    $payload = NULL;

    if (!empty($this->configuration['use_nonce'])) {
      $expected_nonce = $this->consumeNonce();
    }

    if (!empty($this->configuration['validate_signature']) && !is_string($id_token)) {
      $trace_id = uniqid();
      $this->loggerFactory->get('openid_connect_' . $this->pluginId)
        ->error('ID token signature validation requested, but the provider response did not include an ID token. Trace ID: @trace_id', ['@trace_id' => $trace_id]);
      $this->messengerService->addError($this->t('Authentication failed. Please provide this Trace ID to support: @trace_id', ['@trace_id' => $trace_id]));
      return NULL;
    }

    if (!empty($this->configuration['use_nonce']) && !is_string($id_token)) {
      $trace_id = uniqid();
      $this->loggerFactory->get('openid_connect_' . $this->pluginId)
        ->error('Nonce validation requested, but the provider response did not include an ID token. Trace ID: @trace_id', ['@trace_id' => $trace_id]);
      $this->messengerService->addError($this->t('Authentication failed. Please provide this Trace ID to support: @trace_id', ['@trace_id' => $trace_id]));
      return NULL;
    }

    if (!empty($this->configuration['validate_signature']) && is_string($id_token)) {
      try {
        $payload = $this->jwtValidator->validate(
          $id_token,
          $this->configuration['idp_public_keys'],
          $this->configuration['allowed_algorithms']
        );
      }
      catch (\Throwable $exception) {
        $trace_id = uniqid();
        $this->loggerFactory->get('openid_connect_' . $this->pluginId)
          ->error('ID token signature validation failed: @message. Trace ID: @trace_id', [
            '@message' => $exception->getMessage(),
            '@trace_id' => $trace_id,
          ]);
        $this->messengerService->addError($this->t('Authentication failed. Please provide this Trace ID to support: @trace_id', ['@trace_id' => $trace_id]));
        return NULL;
      }
    }

    if (!empty($this->configuration['use_nonce']) && is_string($id_token)) {
      $payload = $payload ?? $this->decodePayload($id_token);

      if (empty($expected_nonce)) {
        $trace_id = uniqid();
        $this->loggerFactory->get('openid_connect_' . $this->pluginId)
          ->error('Nonce validation failed: no stored nonce for comparison. Trace ID: @trace_id', ['@trace_id' => $trace_id]);
        $this->messengerService->addError($this->t('Authentication failed. Please provide this Trace ID to support: @trace_id', ['@trace_id' => $trace_id]));
        return NULL;
      }

      if (empty($payload) || !is_array($payload)) {
        $trace_id = uniqid();
        $this->loggerFactory->get('openid_connect_' . $this->pluginId)
          ->error('Nonce validation failed: unable to decode ID token payload. Trace ID: @trace_id', ['@trace_id' => $trace_id]);
        $this->messengerService->addError($this->t('Authentication failed. Please provide this Trace ID to support: @trace_id', ['@trace_id' => $trace_id]));
        return NULL;
      }

      $token_nonce = $payload['nonce'] ?? NULL;
      if ($token_nonce === NULL || !hash_equals($expected_nonce, $token_nonce)) {
        $trace_id = uniqid();
        $this->loggerFactory->get('openid_connect_' . $this->pluginId)
          ->error('Nonce validation failed: expected @expected but received @actual. Trace ID: @trace_id', [
            '@expected' => $expected_nonce,
            '@actual' => $token_nonce ?? 'NULL',
            '@trace_id' => $trace_id,
          ]);
        $this->messengerService->addError($this->t('Authentication failed. Please provide this Trace ID to support: @trace_id', ['@trace_id' => $trace_id]));
        return NULL;
      }
    }

    return $tokens;
  }

  /**
   * {@inheritdoc}
   */
  public function getEndpoints(): array {
    return [
      'authorization' => $this->configuration['authorization_endpoint'],
      'token' => $this->configuration['token_endpoint'],
      'userinfo' => $this->configuration['userinfo_endpoint'],
      'end_session' => $this->configuration['end_session_endpoint'],
    ];
  }

  /**
   * Performs endpoint discovery.
   *
   * @param string $issuer_url
   *   The issuer URL.
   * @param bool|null $allow_private
   *   TRUE to allow private/loopback hosts; NULL to defer to configuration.
   *
   * @return array|false
   *   Array with discovered endpoints; FALSE on failure.
   */
  protected function autoDiscoverEndpoints(string $issuer_url = '', ?bool $allow_private = NULL) {
    static $results = [];

    if ($allow_private === NULL) {
      $allow_private = !empty($this->configuration['allow_private_issuer']);
    }

    if (empty($issuer_url)) {
      $issuer_url = $this->configuration['issuer_url'];
    }

    $cache_id = $issuer_url . '::' . ($allow_private ? 'allow_private' : 'public_only');

    if (!isset($results[$cache_id])) {
      // Resolve the host to an IP to prevent DNS rebinding.
      $safe_ip = $this->getSafeIpAddress($issuer_url, $allow_private);
      if (!$safe_ip) {
        $this->loggerFactory->get('openid_connect_' . $this->pluginId)
          ->error('Auto-discovery failed: Invalid or unsafe issuer URL "@url".', ['@url' => $issuer_url]);
        return FALSE;
      }

      $parsed = parse_url($issuer_url);
      $host = $parsed['host'];

      // Perform the request using the resolved IP.
      $configuration_url = rtrim($issuer_url, '/') . '/.well-known/openid-configuration';
      try {
        $response = $this->httpClient->request('GET', $configuration_url, [
          'headers' => ['Accept' => 'application/json'],
          'resolve' => [$host => $safe_ip],
          'allow_redirects' => FALSE,
        ]);
        $data = (string) $response->getBody();
      }
      catch (\Exception $e) {
        $this->loggerFactory->get('openid_connect_' . $this->pluginId)
          ->error('Auto-discovery failed: @message', ['@message' => $e->getMessage()]);
        return FALSE;
      }

      $results[$cache_id] = Json::decode($data);
    }

    $result = $results[$cache_id];
    if ($result && isset($result['authorization_endpoint']) && isset($result['token_endpoint']) && isset($result['userinfo_endpoint'])) {
      return $result;
    }
    return FALSE;
  }

  /**
   * Convert a text area input into a normalized collection of keys.
   *
   * @param string $raw_keys
   *   Raw text containing one or more keys.
   *
   * @return array
   *   Normalized list of keys.
   */
  protected function normalizePublicKeys(string $raw_keys): array {
    $raw_keys = trim($raw_keys);
    if ($raw_keys === '') {
      return [];
    }

    $first_character = $raw_keys[0];
    if ($first_character === '{' || $first_character === '[') {
      try {
        $decoded = Json::decode($raw_keys);
      }
      catch (\Throwable $exception) {
        throw new \InvalidArgumentException(sprintf('Unable to parse the JWKS document: %s', $exception->getMessage()));
      }

      if (isset($decoded['keys']) && is_array($decoded['keys'])) {
        $keys = $decoded['keys'];
      }
      elseif (is_array($decoded) && array_is_list($decoded)) {
        $keys = $decoded;
      }
      elseif (is_array($decoded) && isset($decoded['kty'])) {
        $keys = [$decoded];
      }
      else {
        throw new \InvalidArgumentException('The JWKS document must contain either a "keys" array or a single key definition.');
      }

      $keys = array_values(array_filter($keys, function ($key) {
        if (!is_array($key)) {
          return FALSE;
        }
        if (isset($key['use']) && $key['use'] !== 'sig') {
          return FALSE;
        }
        return TRUE;
      }));

      if (empty($keys)) {
        throw new \InvalidArgumentException('The JWKS document does not contain any signature keys.');
      }

      try {
        $normalized = Json::encode(['keys' => $keys]);
      }
      catch (\Throwable $exception) {
        throw new \InvalidArgumentException(sprintf('Unable to normalize the JWKS document: %s', $exception->getMessage()));
      }

      return [$normalized];
    }

    $parts = preg_split('/\R{2,}/', $raw_keys);
    $keys = [];
    foreach ($parts as $part) {
      $part = trim($part);
      if ($part !== '') {
        $keys[] = $part;
      }
    }

    return $keys;
  }

  /**
   * Normalize the allowed algorithm string into an array.
   *
   * @param string $algorithms
   *   Raw algorithm string from the configuration form.
   *
   * @return array
   *   Clean list of algorithm identifiers.
   */
  protected function normalizeAlgorithms(string $algorithms): array {
    $algorithms = trim($algorithms);
    if ($algorithms === '') {
      return [];
    }

    $list = preg_split('/[\s,]+/', $algorithms);
    $list = array_filter(array_map('strtoupper', $list));

    return array_values(array_unique($list));
  }

  /**
   * Store the generated nonce in the user session.
   *
   * @param string $nonce
   *   The nonce value to store.
   */
  protected function storeNonce(string $nonce): void {
    $session = $this->requestStack->getCurrentRequest()->getSession();
    if ($session) {
      $session->set($this->getNonceSessionKey(), $nonce);
    }
  }

  /**
   * Consume and remove the stored nonce.
   *
   * @return string|null
   *   The stored nonce value.
   */
  protected function consumeNonce(): ?string {
    $session = $this->requestStack->getCurrentRequest()->getSession();
    if (!$session) {
      return NULL;
    }

    $key = $this->getNonceSessionKey();
    $nonce = $session->get($key);
    $session->remove($key);
    return $nonce;
  }

  /**
   * Clear the stored nonce without returning it.
   */
  protected function clearNonce(): void {
    $session = $this->requestStack->getCurrentRequest()->getSession();
    if ($session) {
      $session->remove($this->getNonceSessionKey());
    }
  }

  /**
   * Build the session key used to persist the nonce.
   *
   * @return string
   *   The nonce key.
   */
  protected function getNonceSessionKey(): string {
    return 'openid_client_advanced_nonce_' . $this->parentEntityId;
  }

  /**
   * Decode the payload of a JWT.
   *
   * @param string $jwt
   *   The encoded JWT.
   *
   * @return array|null
   *   The decoded payload or NULL on failure.
   */
  protected function decodePayload(string $jwt): ?array {
    // JWT structure is header.payload.signature;.
    $parts = explode('.', $jwt, 3);
    if (count($parts) !== 3) {
      return NULL;
    }
    // The payload lives in the second segment.
    $payload_segment = $parts[1];
    $remainder = strlen($payload_segment) % 4;
    if ($remainder) {
      $payload_segment .= str_repeat('=', 4 - $remainder);
    }

    $decoded = base64_decode(strtr($payload_segment, '-_', '+/'));
    if ($decoded === FALSE) {
      return NULL;
    }

    $payload = Json::decode($decoded);
    return is_array($payload) ? $payload : NULL;
  }

  /**
   * Validates a URL to prevent SSRF attacks.
   *
   * @param string $url
   *   The URL to validate.
   *
   * @return bool
   *   TRUE if the URL is safe, FALSE otherwise.
   */

  /**
   * Validates a URL to prevent SSRF attacks.
   *
   * @param string $url
   *   The URL to validate.
   *
   * @return bool
   *   TRUE if the URL is safe, FALSE otherwise.
   */
  protected function validateUrl(string $url): bool {
    return (bool) $this->getSafeIpAddress($url, !empty($this->configuration['allow_private_issuer']));
  }

  /**
   * Resolves a URL to a safe IP address.
   *
   * @param string $url
   *   The URL to resolve.
   * @param bool $allow_private
   *   TRUE to allow private or loopback IP addresses.
   *
   * @return string|null
   *   The safe IP address or NULL if unsafe/invalid.
   */
  protected function getSafeIpAddress(string $url, bool $allow_private = FALSE): ?string {
    $parsed = parse_url($url);
    if (!isset($parsed['host']) || !isset($parsed['scheme'])) {
      return NULL;
    }

    if (!in_array(strtolower($parsed['scheme']), ['http', 'https'], TRUE)) {
      return NULL;
    }

    $host = $parsed['host'];
    // Handle IPv6 literals in parse_url (wrapped in brackets).
    $host = trim($host, '[]');

    // Resolve the host to IP addresses.
    $records = $this->getDnsRecords($host);
    if (empty($records)) {
      // If no records, it might be an IP address already.
      if (filter_var($host, FILTER_VALIDATE_IP)) {
        $type = filter_var($host, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6) ? 'AAAA' : 'A';
        $key = ($type === 'AAAA') ? 'ipv6' : 'ip';
        $records = [[$key => $host, 'type' => $type]];
      }
      elseif ($allow_private) {
        $resolved_ip = gethostbyname($host);
        if ($resolved_ip !== $host && filter_var($resolved_ip, FILTER_VALIDATE_IP)) {
          $records = [['ip' => $resolved_ip, 'type' => 'A']];
        }
        else {
          return NULL;
        }
      }
      else {
        return NULL;
      }
    }

    foreach ($records as $record) {
      if ($record['type'] === 'A') {
        $ip = $record['ip'];
      }
      elseif ($record['type'] === 'AAAA') {
        $ip = $record['ipv6'];
      }
      else {
        continue;
      }

      $mapped_ipv4 = $this->extractMappedIpv4($ip);
      if ($mapped_ipv4 !== NULL) {
        $ip = $mapped_ipv4;
      }

      // Check for private IP ranges and loopback addresses.
      if ($allow_private) {
        if (filter_var($ip, FILTER_VALIDATE_IP)) {
          return $ip;
        }
      }
      elseif (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
        return $ip;
      }
    }

    return NULL;
  }

  /**
   * Extracts the IPv4 portion of an IPv4-mapped IPv6 address.
   *
   * @param string $ip
   *   The IPv6 address to inspect.
   *
   * @return string|null
   *   IPv4 string if mapped; otherwise NULL.
   */
  protected function extractMappedIpv4(string $ip): ?string {
    $packed = @inet_pton($ip);
    if ($packed === FALSE || strlen($packed) !== 16) {
      return NULL;
    }

    $prefix = str_repeat("\x00", 10) . "\xff\xff";
    if (strncmp($packed, $prefix, 12) !== 0) {
      return NULL;
    }

    $ipv4 = @inet_ntop(substr($packed, 12));
    return is_string($ipv4) ? $ipv4 : NULL;
  }

  /**
   * Retrieves DNS records for a hostname.
   *
   * @param string $host
   *   The hostname.
   *
   * @return array
   *   Array of DNS records.
   */
  protected function getDnsRecords(string $host): array {
    return dns_get_record($host, DNS_A + DNS_AAAA) ?: [];
  }

}
