<?php

namespace Drupal\openid_line_client\Plugin\OpenIDConnectClient;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Form\FormStateInterface;
use Drupal\openid_connect\Plugin\OpenIDConnectClientBase;
use GuzzleHttp\Exception\RequestException;
use Symfony\Component\HttpFoundation\Response;

/**
 * LINE OpenID Connect client.
 *
 * @OpenIDConnectClient(
 *   id = "line",
 *   label = @Translation("LINE")
 * )
 */
class Line extends OpenIDConnectClientBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $url = 'https://developers.line.biz/';
    $form['description'] = [
      '#markup' => '<div class="description">' . $this->t('Set up your app in <a href="@url" target="_blank">Providers</a> on Link Console.', ['@url' => $url]) . '</div>',
    ];

    return $form;
  }

  /**
   * The id_token.
   *
   * @var string|null
   */
  protected $idToken;

  protected $fields = [
    'id', 'name', 'email', 'first_name', 'last_name', 'gender', 'locale', 'timezone', 'picture.height(500)',
  ];

  /**
   * {@inheritdoc}
   */
  public function getEndpoints(): array {
    return [
      'authorization' => 'https://access.line.me/oauth2/v2.1/authorize',
      'token' => 'https://api.line.me/oauth2/v2.1/token',
      'userinfo' => 'https://api.line.me/oauth2/v2.1/userinfo',
      'verify' => 'https://api.line.me/oauth2/v2.1/verify'
    ];
  }

  public function authorize(string $scope = 'openid email', array $additional_params = []): Response {
    return parent::authorize('profile openid email');
  }

  /**
   * {@inheritdoc}
   */
  public function retrieveTokens(string $authorization_code): ?array {
    $tokens = parent::retrieveTokens($authorization_code);
    if ($tokens && isset($tokens['id_token'])) {
      $this->idToken = $tokens['id_token'];
    }
    return $tokens;
  }

  /**
   * {@inheritdoc}
   */
  public function retrieveUserInfo(string $access_token): ?array {

    $userinfo=$this->verifyIdToken();

    return [
      'sub' => $userinfo['sub']?? NULL,
      'name' => $userinfo['name']?? NULL,
      'email' => $userinfo['email']?? NULL,
      'picture' => $userinfo['picture']?? NULL,
    ];
  }

  public function verifyIdToken() {
    if($this->idToken) {
      $request_options = [
        'headers' => [
          'Accept' => 'application/json',
        ],
        'form_params' => [
          'id_token' => $this->idToken,
          'client_id' => $this->configuration['client_id'],
          'client_secret' => $this->configuration['client_secret'],
        ],
      ];
      $endpoints = $this->getEndpoints();
      $client = $this->httpClient;
      try {
        $response = $client->post($endpoints['verify'], $request_options);
        $response_data = Json::decode((string) $response->getBody());

        return (is_array($response_data)) ? $response_data : NULL;
      }
      catch (\Exception $e) {
        $error_message = $e->getMessage();
        if ($e instanceof RequestException && $e->hasResponse()) {
          $error_message .= ' Response: ' . $e->getResponse()->getBody()->getContents();
        }

        $this->loggerFactory->get('openid_connect_' . $this->pluginId)
          ->error('Could not verify idToken. Details: @error_message', ['@error_message' => $error_message]);
      }
    }
  }

}
