<?php

declare(strict_types=1);

namespace Drupal\opensearch_nlp\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Url;
use Drupal\search_api\Entity\Index;
use Drupal\opensearch_nlp\Service\NLPIngestionService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for managing NLP resources (models, pipelines, connectors).
 */
class NlpManagementController extends ControllerBase {

  /**
   * Constructor.
   *
   * @param \Drupal\opensearch_nlp\Service\NLPIngestionService $ingestionService
   *   The NLP ingestion service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    /**
     * The NLP ingestion service.
     */
    protected NLPIngestionService $ingestionService,
    /**
     * The state service.
     */
    protected StateInterface $state,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): NlpManagementController {
    return new self(
      $container->get('opensearch_nlp.nlp_ingestion'),
      $container->get('state')
    );
  }

  /**
   * Display pipelines and models.
   *
   * @return array
   *   The render array.
   */
  public function displayPipelinesAndModel(): array {
    // Add the clear state button.
    $build['clear_state'] = [
      '#type' => 'link',
      '#title' => $this->t('Clean up'),
      '#url' => Url::fromRoute('opensearch_nlp.clear_state'),
      '#attributes' => [
        'class' => ['button', 'button--danger'],
        'style' => 'margin-bottom: 20px;',
      ],
    ];
    // Add a description underneath the button.
    $build['clear_state_description'] = [
      '#markup' => '<div class="description" style="margin-bottom:20px;">' .
      $this->t('This will clear all stale or unused Model, Model group and ingestion pipelines. Use it only if re-connecting with Opensearch server or getting issues while adding/deploying new Model or Model group or connector.') .
      '</div>',
    ];
    $ingestion_pipelines = $this->ingestionService->getIngestionPipelines();
    $model_groups = $this->ingestionService->getModelGroups();
    $search_pipelines = $this->ingestionService->getSearchPipelines();
    $ml_connectors = $this->ingestionService->getAllConnectors();
    if (isset($ingestion_pipelines['error'])) {
      return ['#markup' => $this->t('Failed to fetch ingestion pipelines: @message', ['@message' => $ingestion_pipelines['error']])];
    }
    $build['tables'] = [
      $this->buildTable('Ingestion Pipelines', $this->getPipelineRows($ingestion_pipelines)),
      $this->buildTable('Search Pipelines', $this->getSearchPipelineRows($search_pipelines)),
      $this->buildTable('ML Connectors', $this->getMlConnectors($ml_connectors)),
      $this->buildTable('Model Groups', $this->getModelGroupRows($model_groups)),
      $this->buildTable('Deployed Models', $this->getModelRows($model_groups)),
    ];

    return $build;
  }

  /**
   * Builds a table.
   *
   * @param string $title
   *   The table title.
   * @param array $tableData
   *   The table data.
   *
   * @return array
   *   The render array.
   */
  private function buildTable(string $title, array $tableData): array {
    return [
      ['#markup' => '<h2>' . $title . '</h2>'],
      [
        '#theme' => 'table',
        '#header' => $tableData['header'],
        '#rows' => $tableData['rows'],
        '#attributes' => ['class' => [$tableData['class']]],
      ],
    ];
  }

  /**
   * Clears Old Unused Data.
   */
  public function clearDrupalStateData() {
    // Delete state keys.
    $this->state->delete('nlp_ingestion.model_group_id');
    $this->state->delete('nlp_ingestion.model_id');
    $model_groups = $this->ingestionService->getModelGroups();
    // Delete all models.
    $deleted_models = 0;
    foreach ($model_groups['hits']['hits'] ?? [] as $group) {
      $models = $this->ingestionService->getModelByGroupId($group['_id']);
      foreach ($models['hits']['hits'] ?? [] as $model) {
        $model_id = $model['_id'];
        $this->ingestionService->undeployModel($model_id);
        $this->ingestionService->deleteModel($model_id);
        $deleted_models++;
      }
    }

    // Delete all pipelines.
    $pipelines = $this->ingestionService->getIngestionPipelines();
    $deleted_pipelines = 0;
    foreach (array_keys($pipelines) as $pipeline_name) {
      $this->ingestionService->deletePipeline($pipeline_name);
      $deleted_pipelines++;
    }

    $this->messenger()->addStatus($this->t('Drupal state data, all models (@models), and pipelines (@pipelines) cleared successfully.', [
      '@models' => $deleted_models,
      '@pipelines' => $deleted_pipelines,
    ]));

    return $this->redirect('opensearch_nlp.ingestion_pipelines');
  }

  /**
   * Gets the pipeline rows.
   *
   * @param array $pipelines
   *   The pipelines.
   *
   * @return array
   *   The pipeline rows.
   */
  private function getPipelineRows(array $pipelines): array {
    $pipeline_rows = [];
    $pipeline_header = [
      $this->t('Pipeline Name'),
      $this->t('Description'),
      $this->t('Model ID'),
      $this->t('Field Map (MAP:EMBEDDING)'),
      $this->t('Actions'),
      $this->t('Test'),
    ];
    $deployed_models = $this->ingestionService->getDeployedModels();
    foreach ($pipelines as $name => $details) {
      $modelId = $details['processors'][0]['text_embedding']['model_id'] ?? 'N/A';
      $fieldMap = json_encode($details['processors'][0]['text_embedding']['field_map'], JSON_PRETTY_PRINT);
      $mapping_field = array_keys($details['processors'][0]['text_embedding']['field_map']);
      // Extract index name from description.
      $descriptionParts = explode(':', (string) $details['description'], 2);
      $indexId = isset($descriptionParts[1]) ? trim($descriptionParts[1]) : '';
      // Load search index and check if the field exists.
      $fieldExists = FALSE;
      if ($indexId !== '' && $indexId !== '0') {
        $searchIndex = Index::load($indexId);
        if ($searchIndex) {
          $indexFields = $searchIndex->getFields();
          $fieldMapArray = json_decode($fieldMap, TRUE) ?? [];
          foreach ($fieldMapArray as $sourceField => $targetField) {
            if (isset($indexFields[$targetField]) && $sourceField) {
              $fieldExists = TRUE;
              break;
            }
          }
        }
      }
      // Create delete button.
      $delete_link = $this->buildDeleteLink('opensearch_nlp.delete_pipeline', ['pipeline_name' => $name], $fieldExists);
      $test_link = $this->buildTestLink(
        'opensearch_nlp.test_nlp_ingestion_pipeline_form',
        [
          'pipeline_name' => $name,
          'index' => $indexId,
          'mapping_field' => $mapping_field[0],
        ],
        TRUE
      );
      // Disable delete button if field exists.
      if ($fieldExists && isset($deployed_models['hits']['hits'][0]['_id'])) {
        $delete_link['#attributes']['class'][] = 'disabled';
        $delete_link['#attributes']['onclick'] = 'return false;';
        $delete_link['#attributes']['style'] = 'pointer-events: none; opacity: 0.5;';
      }

      $pipeline_rows[] = [
        ['data' => $name],
        ['data' => $details['description']],
        ['data' => $modelId],
        ['data' => $fieldMap, 'class' => ['field-map']],
        ['data' => $delete_link],
        ['data' => $test_link],
      ];
    }
    return ['header' => $pipeline_header, 'rows' => $pipeline_rows, 'class' => 'ingestion-pipelines-table'];
  }

  /**
   * Gets the search pipeline rows.
   *
   * @param array $pipelines
   *   The pipelines.
   *
   * @return array
   *   The search pipeline rows.
   */
  private function getSearchPipelineRows(array $pipelines): array {
    $header = [
      $this->t('Pipeline Name'),
      $this->t('Description'),
      $this->t('Phase Results Processors'),
      $this->t('Actions'),
    ];
    $rows = [];
    foreach ($pipelines as $name => $details) {
      $processors = isset($details['phase_results_processors']) ? json_encode($details['phase_results_processors'], JSON_PRETTY_PRINT) : 'N/A';
      $delete_link = $this->buildDeleteLink('opensearch_nlp.delete_search_pipeline', ['pipeline_id' => $name], FALSE);
      $edit_link = $this->buildEditLink('opensearch_nlp.edit_search_pipeline', ['pipeline_id' => $name], TRUE);
      $rows[] = [
        ['data' => $name],
        ['data' => $details['description'] ?? 'N/A'],
        ['data' => $processors, 'class' => ['json-data']],
        ['data' => $delete_link],
        ['data' => $edit_link],
      ];
    }
    return ['header' => $header, 'rows' => $rows, 'class' => 'search-pipelines-table'];
  }

  /**
   * Gets the ml connectors rows.
   *
   * @param array $connectors
   *   The connectors.
   *
   * @return array
   *   The ml connector rows.
   */
  private function getMlConnectors(array $connectors): array {
    $header = [
      $this->t('Connector ID'),
      $this->t('Connector Name'),
      $this->t('Description'),
      $this->t('Protocol'),
      $this->t('Model'),
      $this->t('Actions'),
    ];
    // Initialize rows array.
    $rows = [];
    foreach ($connectors['hits']['hits'] as $connector) {
      $source = $connector['_source'];
      $id = $connector['_id'] ?? 'N/A';
      $name = $source['name'] ?? 'N/A';
      $description = $source['description'] ?? 'N/A';
      $protocol = $source['protocol'] ?? 'N/A';
      $model = $source['parameters']['model'] ?? 'N/A';
      // Build the delete link.
      $delete_link = $this->buildDeleteLink('opensearch_nlp.delete_ml_connector', ['id' => $id], FALSE);
      $rows[] = [
        ['data' => $id],
        ['data' => $name],
        ['data' => $description],
        ['data' => $protocol],
        ['data' => $model],
        ['data' => $delete_link],
      ];
    }

    return ['header' => $header, 'rows' => $rows, 'class' => 'ml-connectors-table'];
  }

  /**
   * Gets the model group rows.
   *
   * @param array $groups
   *   The model groups.
   *
   * @return array
   *   The model group rows.
   */
  private function getModelGroupRows(array $groups): array {
    $header = [
      $this->t('Group ID'),
      $this->t('Group Name'),
      $this->t('Description'),
      $this->t('Latest Version'),
      $this->t('Access Type'),
      $this->t('Created Time'),
      $this->t('Last Updated Time'),
      $this->t('Actions'),
    ];
    $rows = [];

    foreach ($groups['hits']['hits'] ?? [] as $group) {
      $source = $group['_source'];
      // Build the delete link.
      $delete_link = $this->buildDeleteLink('opensearch_nlp.delete_ml_group', ['id' => $group['_id']], FALSE);
      $rows[] = [
        ['data' => $group['_id']],
        ['data' => $source['name']],
        ['data' => $source['description']],
        ['data' => $source['latest_version']],
        ['data' => $source['access']],
        ['data' => date('Y-m-d H:i:s', (int) round($source['created_time'] / 1000))],
        ['data' => date('Y-m-d H:i:s', (int) round($source['last_updated_time'] / 1000))],
        ['data' => $delete_link],
      ];
    }
    return ['header' => $header, 'rows' => $rows, 'class' => 'model-groups-table'];
  }

  /**
   * Deletes a model group via API.
   *
   * @param string $id
   *   The model group ID.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   The redirect response.
   */
  public function deleteMlGroup($id) {
    if (!$id) {
      $this->messenger()->addError($this->t('Model group ID is required.'));
      return $this->redirect('opensearch_nlp.ingestion_pipelines');
    }
    // Delete the model group.
    $result = $this->ingestionService->deleteModelGroup($id);

    if (isset($result['error'])) {
      $this->messenger()->addError($this->t('Failed to delete model group: @message', ['@message' => $result['error']]));
    }
    else {
      $this->messenger()->addStatus($this->t('Model group "@id" deleted successfully.', ['@id' => $id]));
    }

    return $this->redirect('opensearch_nlp.ingestion_pipelines');
  }

  /**
   * Gets the model rows.
   *
   * @param array $groups
   *   The model groups.
   *
   * @return array
   *   The model rows.
   */
  private function getModelRows(array $groups): array {
    $header = [
      $this->t('Model ID'),
      $this->t('Model Name'),
      $this->t('Model Group ID'),
      $this->t('Version'),
      $this->t('Format'),
      $this->t('Algorithm'),
      $this->t('State'),
      $this->t('Created Time'),
      $this->t('Last Deployed Time'),
      $this->t('Worker Node Count'),
      $this->t('Size (Bytes)'),
      $this->t('Actions'),
      $this->t('Predict'),
      $this->t('Redeploy Model'),
      $this->t('Notice'),
    ];
    $rows = [];
    $registered_models = [];

    // Fetch all ingestion pipelines to check for model usage.
    $ingestion_pipelines = $this->ingestionService->getIngestionPipelines();
    $used_model_ids = [];

    // Extract all model IDs used in ingestion pipelines.
    foreach ($ingestion_pipelines as $pipeline) {
      $modelId = $pipeline['processors'][0]['text_embedding']['model_id'] ?? NULL;
      if ($modelId) {
        $used_model_ids[] = $modelId;
      }
    }
    if (!isset($groups['hits']['hits'])) {
      return [
        'header' => $header,
        'rows' => $rows,
        'class' => 'models-table',
      ];
    }

    foreach ($groups['hits']['hits'] as $model_group) {
      $registered_models[] = $this->ingestionService->getModelByGroupId($model_group['_id']);
    }

    if (empty($registered_models)) {
      return [
        'header' => $header,
        'rows' => $rows,
        'class' => 'models-table',
      ];
    }

    foreach ($registered_models as $modeldata) {
      $model = $modeldata['hits']['hits'][0] ?? [];
      if (!empty($model)) {
        $source = $model['_source'];
        $type = $source['algorithm'] === 'REMOTE' ? 'external' : 'internal';
        // Check if the model ID is used in any ingestion pipeline.
        $is_used_in_pipeline = in_array($model['_id'], $used_model_ids);
        // Build the delete link.
        $delete_link = $this->buildDeleteLink(
          'opensearch_nlp.delete_model',
          ['model_id' => $model['_id'], 'type' => $type],
          $source['model_state'] == 'DEPLOYED'
        );
        // Build the redeploy link if model deplyment fails.
        $redeploy_link = $this->buildRedployLink(
          'opensearch_nlp.redeploy_model',
          ['model_id' => $model['_id'], 'type' => $type],
          $source['model_state'] != 'DEPLOYED'
        );
        if ($is_used_in_pipeline && $source['model_state'] == 'DEPLOYED') {
          $delete_link['#attributes']['class'][] = 'disabled';
          $delete_link['#attributes']['onclick'] = 'return false;';
          $delete_link['#attributes']['style'] = 'pointer-events: none; opacity: 0.5;';
          $redeploy_link['#attributes']['class'][] = 'disabled';
          $redeploy_link['#attributes']['onclick'] = 'return false;';
          $redeploy_link['#attributes']['style'] = 'pointer-events: none; opacity: 0.5;';
        }
        // Add a description if the model is used in a pipeline.
        $description = $is_used_in_pipeline
          ? $this->t('This model is deployed and currently used in an ingestion pipeline and cannot be deleted.')
          : ($source['model_state'] == 'DEPLOYED'
            ? $this->t('This model is not used anywhere.')
            : '');

        // Predict button.
        $predict_link = $this->buildPredictLink('opensearch_nlp.predict_form', ['model_id' => $model['_id']], TRUE);
        $rows[] = [
          ['data' => $model['_id']],
          ['data' => $source['name']],
          ['data' => $source['model_group_id']],
          ['data' => $source['model_version']],
          ['data' => $source['model_format'] ?? 'NA'],
          ['data' => $source['algorithm']],
          ['data' => $source['model_state']],
          ['data' => date('Y-m-d H:i:s', (int) round($source['created_time'] / 1000))],
          ['data' => $source['model_state'] == 'DEPLOYED' ? date('Y-m-d H:i:s', (int) round($source['last_deployed_time'] / 1000)) : 'NA'],
          ['data' => $source['model_state'] == 'DEPLOYED' ? $source['current_worker_node_count'] : 'NA'],
          ['data' => isset($source['model_content_size_in_bytes']) ? number_format($source['model_content_size_in_bytes']) : 'NA'],
          ['data' => $delete_link],
          ['data' => $predict_link],
          ['data' => $redeploy_link],
          ['data' => $description],
        ];
      }
    }
    return ['header' => $header, 'rows' => $rows, 'class' => 'models-table'];
  }

  /**
   * Builds a delete link.
   *
   * @param string $route
   *   The route name.
   * @param array $parameters
   *   The route parameters.
   * @param bool $use_button
   *   Whether to use a button element.
   *
   * @return array|string
   *   A render array or a string (link).
   */
  protected function buildDeleteLink($route, array $parameters = [], $use_button = TRUE) {
    $url = Url::fromRoute($route, $parameters);

    if ($use_button) {
      return [
        '#type' => 'link',
        '#title' => $this->t('Delete'),
        '#url' => $url,
        '#attributes' => [
          'class' => ['button', 'button--danger'],
        ],
      ];
    }

    return Link::fromTextAndUrl($this->t('Delete'), $url)->toString();
  }

  /**
   * Builds an edit link.
   *
   * @param string $route_name
   *   The route name.
   * @param array $route_parameters
   *   The route parameters.
   * @param bool $use_ajax
   *   Whether to use AJAX for the link.
   *
   * @return array|string
   *   A render array or a string (link).
   */
  private function buildEditLink(string $route_name, array $route_parameters, bool $use_ajax = TRUE) {
    $url = Url::fromRoute($route_name, $route_parameters);

    $link_options = [
      'attributes' => [
        'class' => ['use-ajax', 'button', 'button--primary'],
        'data-dialog-type' => 'modal',
        'data-dialog-options' => json_encode([
          'width' => 800,
        ]),
      ],
    ];

    if ($use_ajax) {
      $url->setOptions($link_options);
    }

    return Link::fromTextAndUrl($this->t('Edit Pipeline'), $url)->toString();
  }

  /**
   * Builds a redeploy link.
   *
   * @param string $route_name
   *   The route name.
   * @param array $route_parameters
   *   The route parameters.
   * @param bool $failed
   *   Whether the model deployment failed.
   */
  private function buildRedployLink(string $route_name, array $route_parameters, bool $failed = TRUE) {
    $url = Url::fromRoute($route_name, $route_parameters);
    if (!$failed) {
      return [
        '#type' => 'link',
        '#title' => $this->t('ReDeploy'),
        '#url' => $url,
        '#attributes' => [
          'class' => ['button', 'button--danger'],
        ],
      ];
    }
    return Link::fromTextAndUrl($this->t('ReDeploy Model'), $url)->toString();

  }

  /**
   * Builds a predict link.
   *
   * @param string $route_name
   *   The route name.
   * @param array $route_parameters
   *   The route parameters.
   * @param bool $use_ajax
   *   Whether to use AJAX for the link.
   *
   * @return string
   *   The rendered link.
   */
  private function buildPredictLink(string $route_name, array $route_parameters, bool $use_ajax = TRUE) {
    $url = Url::fromRoute($route_name, $route_parameters);

    $link_options = [
      'attributes' => [
        'class' => ['use-ajax', 'button', 'button--primary'],
        'data-dialog-type' => 'modal',
        'data-dialog-options' => json_encode([
          'width' => 800,
        ]),
      ],
    ];

    if ($use_ajax) {
      $url->setOptions($link_options);
    }

    return Link::fromTextAndUrl($this->t('Test/Predict Model'), $url)->toString();
  }

  /**
   * Builds a TEST link.
   *
   * @param string $route_name
   *   The route name.
   * @param array $route_parameters
   *   The route parameters.
   * @param bool $use_ajax
   *   Whether to use AJAX for the link.
   *
   * @return string
   *   The rendered link.
   */
  private function buildTestLink(string $route_name, array $route_parameters, bool $use_ajax = TRUE) {
    $url = Url::fromRoute($route_name, $route_parameters);

    $link_options = [
      'attributes' => [
        'class' => ['use-ajax', 'button', 'button--primary'],
        'data-dialog-type' => 'modal',
        'data-dialog-options' => json_encode([
          'width' => 800,
        ]),
      ],
    ];

    if ($use_ajax) {
      $url->setOptions($link_options);
    }

    return Link::fromTextAndUrl($this->t('Test Pipeline'), $url)->toString();
  }

  /**
   * Deletes a pipeline via API.
   */
  public function deletePipeline($pipeline_name) {
    if ($pipeline_name === NULL || $pipeline_name === '') {
      $this->messenger()->addError($this->t('Pipeline name is required.'));
      return $this->redirect('opensearch_nlp.ingestion_pipelines');
    }
    $this->state->delete($pipeline_name);
    $result = $this->ingestionService->deletePipeline($pipeline_name);

    if (isset($result['error'])) {
      $this->messenger()->addError($this->t('Failed to delete pipeline: @message', ['@message' => $result['error']]));
    }
    else {
      $this->messenger()->addStatus($this->t('Pipeline "@name" deleted successfully.', ['@name' => $pipeline_name]));
    }

    return $this->redirect('opensearch_nlp.ingestion_pipelines');
  }

  /**
   * Redeploys a model via API.
   *
   * @param string $model_id
   *   The model ID.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   The redirect response.
   */
  public function redeployModel($model_id) {
    if (!$model_id || $model_id === NULL) {
      $this->messenger()->addError($this->t('Model ID is required.'));
      return $this->redirect('opensearch_nlp.ingestion_pipelines');
    }
    // Redeploy the model.
    $result = $this->ingestionService->deployModel($model_id);

    if (isset($result['error'])) {
      $this->messenger()->addError($this->t('Failed to redeploy model: @message', ['@message' => $result['error']]));
    }
    else {
      $this->messenger()->addStatus($this->t('Model "@id" redeployed successfully.', ['@id' => $model_id]));
    }

    return $this->redirect('opensearch_nlp.ingestion_pipelines');
  }

  /**
   * Deletes a model via API.
   */
  public function deleteModel(?string $model_id, string $type) {
    if (!$model_id || $model_id === NULL) {
      $this->messenger()->addError($this->t('Model ID is required.'));
      return $this->redirect('opensearch_nlp.ingestion_pipelines');
    }
    // Delete the model from the state.
    $this->state->delete($model_id . '-' . $type);
    // Undeploy the model.
    $this->ingestionService->undeployModel($model_id);
    // Delete the model.
    $result = $this->ingestionService->deleteModel($model_id);

    if (isset($result['error'])) {
      $this->messenger()->addError($this->t('Failed to delete model: @message', ['@message' => $result['error']]));
    }
    else {
      $this->messenger()->addStatus($this->t('Model "@id" deleted successfully.', ['@id' => $model_id]));
    }

    return $this->redirect('opensearch_nlp.ingestion_pipelines');
  }

  /**
   * Deletes ml connector via API.
   */
  public function deleteConnector($id) {
    if (!$id || $id === NULL) {
      $this->messenger()->addError($this->t('Connector ID is required.'));
      return $this->redirect('opensearch_nlp.ingestion_pipelines');
    }
    // Delete the model.
    $result = $this->ingestionService->deleteConnector($id);

    if (isset($result['error'])) {
      $this->messenger()->addError($this->t('Failed to delete connector: @message', ['@message' => $result['error']]));
    }
    else {
      $this->messenger()->addStatus($this->t('Connector "@id" deleted successfully.', ['@id' => $id]));
    }

    return $this->redirect('opensearch_nlp.ingestion_pipelines');
  }

  /**
   * Deletes a search pipeline via API.
   */
  public function deleteSearchPipeline($pipeline_id) {
    if (!$pipeline_id || $pipeline_id === NULL) {
      $this->messenger()->addError($this->t('Pipeline ID is required.'));
      return $this->redirect('opensearch_nlp.ingestion_pipelines');
    }
    // Delete the pipeline from the state.
    $this->state->delete($pipeline_id);
    // Delete the pipeline.
    $result = $this->ingestionService->deleteSearchPipeline($pipeline_id);

    if (isset($result['error'])) {
      $this->messenger()->addError($this->t('Failed to delete pipeline: @message', ['@message' => $result['error']]));
    }
    else {
      $this->messenger()->addStatus($this->t('Pipeline "@id" deleted successfully.', ['@id' => $pipeline_id]));
    }

    return $this->redirect('opensearch_nlp.ingestion_pipelines');
  }

}
