<?php

declare(strict_types=1);

namespace Drupal\opensearch_nlp\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\opensearch_nlp\Cache\SemanticCache;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\Pager\PagerManagerInterface;

/**
 * Controller for querying semantic cache data.
 */
class SemanticCacheController extends ControllerBase {

  /**
   * Constructs the SemanticCacheController.
   *
   * @param \Drupal\opensearch_nlp\Cache\SemanticCache $semanticCache
   *   The SemanticCache service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack service.
   * @param \Drupal\Core\Pager\PagerManagerInterface $pagerManager
   *   The pager manager service.
   */
  public function __construct(protected SemanticCache $semanticCache, protected RequestStack $requestStack, protected PagerManagerInterface $pagerManager) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): SemanticCacheController {
    return new self(
      $container->get('opensearch_nlp.semantic_cache'),
      $container->get('request_stack'),
      $container->get('pager.manager')
    );
  }

  /**
   * Displays all cached data.
   *
   * @return array
   *   A render array containing the cached data.
   */
  public function displayAllCachedData(): array {
    // Fetch all cached data.
    $cached_data = $this->semanticCache->queryCachedData();

    $latest_rows = [];
    $hits_count = [];

    foreach ($cached_data as $row) {
      $query = strtolower((string) $row['query']);
      // Sum hits for each query.
      if (!isset($hits_count[$query])) {
        $hits_count[$query] = 0;
      }
      $hits_count[$query] += $row['hits'];

      // If this query is not set yet or this row is newer, keep it.
      if (
        !isset($latest_rows[$query]) ||
        strtotime((string) $row['date']) > strtotime((string) $latest_rows[$query]['date'])
      ) {
        $latest_rows[$query] = $row;
        $latest_rows[$query]['query'] = ucfirst($query);
      }
    }

    $header = [
      $this->t('Query'),
      $this->t('Search Type'),
      $this->t('Hits'),
      $this->t('Similar Queries and Similarity'),
      $this->t('Date'),
    ];

    $rows = [];
    foreach ($latest_rows as $query => $row) {
      // Format similar queries as a comma-separated string.
      $similar_queries = empty($row['similar_queries'])
        ? $this->t('None')
        : implode(', ', array_unique(array_map(fn($q): string => ucfirst(strtolower((string) $q['query'])) . ' (' . round($q['similarity'] * 100, 2) . '%)', $row['similar_queries']), SORT_STRING));
      $rows[] = [
        'query' => $row['query'],
        'search_type' => $row['search_type'],
        'hits' => intval($hits_count[$query] / 2),
        'similar_queries' => $similar_queries,
        'date' => $row['date'],
      ];
    }

    // Add pagination (20 items per page).
    $items_per_page = 10;
    $total_items = count($rows);
    $current_page = (int) $this->requestStack->getCurrentRequest()->query->get('page', 0);
    $offset = $current_page * $items_per_page;
    // Register the pager.
    $this->pagerManager->createPager($total_items, $items_per_page);

    $paged_rows = array_slice($rows, $offset, $items_per_page);
    // Add a link to clear the cache.
    $clear_cache_link = [
      '#type' => 'link',
      '#title' => $this->t('Clear Semantic Cache'),
      '#url' => Url::fromRoute('opensearch_nlp.semantic_cache_clear'),
      '#attributes' => [
        'class' => ['button', 'button--danger'],
      ],
    ];

    $settings_link = [
      '#type' => 'link',
      '#title' => $this->t('Enable/Disable Semantic Cache'),
      '#url' => Url::fromRoute('opensearch_nlp.semantic_cache_settings'),
      '#attributes' => ['class' => ['button']],
    ];

    return [
      $settings_link,
      $clear_cache_link,
      [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => array_map(function (array $row): array {
          return [
            'query' => $row['query'],
            'search_type' => $row['search_type'],
            'hits' => $row['hits'] == 0 ? 1 : $row['hits'],
            'similar_queries' => $row['similar_queries'],
            'date' => $row['date'],
          ];
        }, $paged_rows),
        '#empty' => $this->t('No cached data found.'),
      ],
      ['#type' => 'pager'],
    ];
  }

  /**
   * Clears the semantic cache.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the cached data display page.
   */
  public function clearCache() {
    // Call the cleanupCacheKeys method from the SemanticCache service.
    $this->semanticCache->clearAllCachedData();
    // Display a confirmation message.
    $this->messenger()->addMessage($this->t('Semantic cache has been cleared.'));
    // Redirect back to the cached data display page.
    return $this->redirect('opensearch_nlp.semantic_cache_query');
  }

}
