<?php

declare(strict_types=1);

namespace Drupal\opensearch_nlp\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\IndexInterface;
use Drupal\search_api_opensearch\Event\AlterSettingsEvent;
use Drupal\search_api_opensearch\Event\FieldMappingEvent;
use Drupal\search_api_opensearch\Event\IndexParamsEvent;
use Drupal\search_api_opensearch\SearchAPI\FieldMapper;
use Drupal\opensearch_nlp\Plugin\search_api\data_type\opensearchTextEmbeddingVector as EmbeddingVectorField;
use Drupal\opensearch_nlp\Service\NLPIngestionService;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\Core\Routing\RouteMatchInterface;

/**
 * Event subscriber for managing an NLP-based OpenSearch pipeline.
 */
class OpensearchBuildIndexParams implements EventSubscriberInterface {

  /**
   * The current index being processed.
   */
  protected ?IndexInterface $currentIndex = NULL;

  /**
   * Constructs a new EventSubscriber.
   *
   * @param \Drupal\search_api_opensearch\SearchAPI\FieldMapper $fieldParamsBuilder
   *   The field mapper.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory for retrieving settings.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The route match service to get the current route.
   * @param \Drupal\opensearch_nlp\Service\NLPIngestionService $ingestionService
   *   The NLP ingestion service.
   */
  public function __construct(
    protected FieldMapper $fieldParamsBuilder,
    protected ConfigFactoryInterface $configFactory,
    /**
     * The route match service.
     */
    protected RouteMatchInterface $routeMatch,
    protected NLPIngestionService $ingestionService,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      IndexParamsEvent::class => ['indexParams'],
      FieldMappingEvent::class => ['onFieldMapping'],
      AlterSettingsEvent::class => ['onAlterSettings', -10],
    ];
  }

  /**
   * Returns the current index object.
   */
  public function getCurrentIndex(): ?IndexInterface {
    return $this->currentIndex;
  }

  /**
   * Event called when field mappings are determined.
   */
  public function onFieldMapping(FieldMappingEvent $event): void {
    $this->currentIndex = $event->getField()->getIndex();
    $config = $this->configFactory->get('opensearch_nlp.nlp_settings');
    // Check if NLP is enabled.
    if (!$config->get('enable_nlp')) {
      return;
    }

    if ($event->getField()->getType() === EmbeddingVectorField::PLUGIN_ID) {
      $vector_config = $this->configFactory->get('opensearch_nlp.vector_settings');
      $event->setParam([
        'type' => 'knn_vector',
        'dimension' => (int) ($vector_config->get('dimension') ?? 768),
        'method' => [
          'name' => $vector_config->get('method') ?? 'hnsw',
          'space_type' => $vector_config->get('space_type') ?? 'l2',
          'engine' => $vector_config->get('engine') ?? 'lucene',
        ],
      ]);
    }

  }

  /**
   * Event called when index settings are altered.
   */
  public function onAlterSettings(AlterSettingsEvent $event): void {
    // Ensure ML Commons cluster settings are applied.
    $this->ingestionService->ensureMlCommonsSettings();

    $index_id = NULL;
    if ($this->currentIndex instanceof IndexInterface) {
      $index_id = $this->currentIndex->id();
    }
    if (!$index_id) {
      $route_match = $this->routeMatch;
      $route_name = $route_match->getRouteName();
      if (in_array($route_name, ['entity.search_api_index.fields', 'entity.search_api.index_fields_edit'], TRUE)) {
        $index_param = $route_match->getParameter('search_api_index');
        if ($index_param && method_exists($index_param, 'get')) {
          $index_id = $index_param->get('id');
        }
      }
    }
    if (!$index_id) {
      return;
    }
    $config = $this->configFactory->get('opensearch_nlp.nlp_settings');
    $vector_config = $this->configFactory->get('opensearch_nlp.vector_settings');
    // Get the indexes config from settings.
    $indexes_config = $config->get('indexes') ?? [];
    // Check if the NLP settings for the given index ID exist.
    if (isset($indexes_config[$index_id])) {
      $nlp_settings = $indexes_config[$index_id];
      // Check if the necessary fields are set in the NLP settings.
      if (!empty($nlp_settings['mapping_embedding_pairs'])) {
        $index = Index::load($index_id);
        if ($index instanceof IndexInterface) {
          // @phpstan-ignore-next-line
          $backend = $index->getServerInstance()->getBackend()->getBackendClient();
          $params = $this->fieldParamsBuilder->mapFieldParams($backend->getIndexId($index), $index);
          // Ensure the embedding vector field exists and is of type knn_vector.
          $embedding_field = explode('|', (string) $nlp_settings['mapping_embedding_pairs'])[1];
          $is_valid = array_filter($params['body']['properties'], function (array $type, $field_name) use ($embedding_field): bool {
            return $field_name === $embedding_field && $type['type'] === 'knn_vector';
          }, ARRAY_FILTER_USE_BOTH);
          if ($is_valid !== []) {
            $settings = $event->getSettings();
            $settings['index']['knn.space_type'] = $vector_config->get('space_type') ?? 'l2';
            $settings['index']['search']['default_pipeline'] = $nlp_settings['search_pipeline_id'];
            $settings['index']['default_pipeline'] = $nlp_settings['ingestion_pipeline_id'];
            $event->setSettings($settings);
          }
        }
      }
    }
  }

  /**
   * Modify index parameters before indexing.
   */
  public function indexParams(IndexParamsEvent $event): void {
    $config = $this->configFactory->get('opensearch_nlp.nlp_settings');
    // Check if NLP is enabled.
    if (!$config->get('enable_nlp')) {
      return;
    }
    $params = $event->getParams();

    if (!isset($params['body']) || !is_array($params['body'])) {
      return;
    }

    $indexes_config = $config->get('indexes');
    foreach ($indexes_config as $nlp_settings) {
      // Check if a single mapping-embedding pair exists for the index.
      if (!empty($nlp_settings['mapping_embedding_pairs'])) {
        // Split the combined pair into mapping and embedding fields.
        $mapping_field = explode('|', (string) $nlp_settings['mapping_embedding_pairs'])[0];
        // Process the mapping field in the body.
        foreach ($params['body'] as &$bodyItem) {
          if (isset($bodyItem[$mapping_field]) && is_array($bodyItem[$mapping_field]) && !empty($bodyItem[$mapping_field])) {
            // Convert the array of values into a single string.
            $bodyItem[$mapping_field] = implode(' ', $bodyItem[$mapping_field]);
          }
        }
      }
    }

    $event->setParams($params);
  }

}
