<?php

declare(strict_types=1);

namespace Drupal\opensearch_nlp\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Psr\Log\LoggerInterface;
use Drupal\opensearch_nlp\Service\NLPIngestionService;

/**
 * Provides a form to create an ML connector.
 */
class MlConnectorForm extends FormBase {

  /**
   * Constructs a new MlConnectorForm.
   *
   * @param \Drupal\opensearch_nlp\Service\NLPIngestionService $nlpIngestionService
   *   The NLP ingestion service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(protected NLPIngestionService $nlpIngestionService, protected LoggerInterface $logger) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): MlConnectorForm {
    return new self(
      $container->get('opensearch_nlp.nlp_ingestion'),
      $container->get('logger.factory')->get('opensearch_nlp')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ml_connector_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Connector Name'),
      '#required' => TRUE,
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#required' => TRUE,
    ];

    $form['version'] = [
      '#type' => 'number',
      '#title' => $this->t('Version'),
      '#default_value' => 1,
      '#required' => TRUE,
    ];

    $form['protocol'] = [
      '#type' => 'select',
      '#title' => $this->t('Protocol'),
      '#options' => [
        'http' => $this->t('HTTP'),
        'https' => $this->t('HTTPS'),
        'aws' => $this->t('AWS'),
        'aws_sigv4' => $this->t('AWS SigV4'),
      ],
      '#required' => TRUE,
    ];

    $form['connector_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Connector Type'),
      '#options' => [
        'openai_self_managed' => $this->t('OpenAI'),
        'open_ai_rag' => $this->t('OpenAI RAG'),
        'aws_openai' => $this->t('AWS with OpenAI Model'),
        'aws_model' => $this->t('AWS Model'),
      ],
      '#default_value' => 'openai_self_managed',
      '#required' => TRUE,
    ];

    $form['model'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Model'),
      '#default_value' => 'text-embedding-ada-002',
      '#required' => TRUE,
    ];

    $form['temperature'] = [
      '#type' => 'number',
      '#title' => $this->t('Temperature'),
      '#default_value' => 0.0,
      '#step' => 0.1,
      '#min' => 0.0,
      '#max' => 1.0,
      '#description' => $this->t('Controls the randomness of the output. A value of 0.0 means deterministic output, while higher values introduce more variability.'),
      '#states' => [
        'visible' => [
          ':input[name="connector_type"]' => ['value' => 'open_ai_rag'],
        ],
      ],
    ];

    $form['url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Model Api endpoint'),
      '#default_value' => 'https://api.openai.com/v1/embeddings',
      '#required' => TRUE,
    ];

    $form['service'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Service Name'),
      '#default_value' => 'es',
      '#description' => $this->t('AWS service name'),
      '#states' => [
        'visible' => [
          ':input[name="connector_type"]' => ['value' => 'aws_model'],
        ],
      ],
    ];

    $form['pre_process_function'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pre-process Function'),
      '#default_value' => 'connector.pre_process.openai.embedding',
      '#description' => $this->t('Function to pre-process the input before sending to OpenAI.'),
      '#required' => TRUE,
    ];

    $form['post_process_function'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Post-process Function'),
      '#default_value' => 'connector.post_process.openai.embedding',
      '#description' => $this->t('Function to post-process the response from OpenAI.'),
      '#required' => TRUE,
    ];

    $form['openai_key'] = [
      '#type' => 'textfield',
      '#maxlength' => 300,
      '#title' => $this->t('OpenAI API Key'),
      '#states' => [
        'visible' => [
          [':input[name="connector_type"]' => ['value' => 'openai_self_managed']],
          [':input[name="connector_type"]' => ['value' => 'aws_openai']],
        ],
      ],
    ];

    $form['aws_region'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AWS Region'),
      '#states' => [
        'visible' => [
          [
            ':input[name="connector_type"]' => ['value' => 'aws_openai'],
          ],
          [
            ':input[name="connector_type"]' => ['value' => 'aws_model'],
          ],
        ],
      ],
    ];

    $form['aws_service_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AWS Service Name'),
      '#default_value' => 'bedrock',
      '#states' => [
        'visible' => [
          [
            ':input[name="connector_type"]' => ['value' => 'aws_openai'],
          ],
          [
            ':input[name="connector_type"]' => ['value' => 'aws_model'],
          ],
        ],
      ],
    ];

    $form['secret_arn'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Secret ARN for key stored in AWS Secrets Manager'),
      '#states' => [
        'visible' => [
          [
            ':input[name="connector_type"]' => ['value' => 'aws_openai'],
          ],
          [
            ':input[name="connector_type"]' => ['value' => 'aws_model'],
          ],
        ],
      ],
    ];

    $form['role_arn'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Role ARN'),
      '#states' => [
        'visible' => [
          [
            ':input[name="connector_type"]' => ['value' => 'aws_openai'],
          ],
          [
            ':input[name="connector_type"]' => ['value' => 'aws_model'],
          ],
        ],
      ],
    ];

    $form['external_account_role_arn'] = [
      '#type' => 'textfield',
      '#title' => $this->t('External Role ARN'),
      '#states' => [
        'visible' => [
          [
            ':input[name="connector_type"]' => ['value' => 'aws_openai'],
          ],
          [
            ':input[name="connector_type"]' => ['value' => 'aws_model'],
          ],
        ],
      ],
    ];

    $form['request_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Request Body'),
      '#default_value' => '{ "inputText": "${parameters.inputText}" }',
      '#description' => $this->t('
          <strong>Request body template based on connector type:</strong><br>
          <ul>
            <li><strong>OpenAI:</strong> <code>{ "input": ${parameters.input}, "model": "${parameters.model}" }</code></li>
            <li><strong>Cohere:</strong> <code>{ "texts": ${parameters.texts}, "truncate": "${parameters.truncate}", "model": "${parameters.model}", "input_type": "${parameters.input_type}" }</code></li>
            <li><strong>AWS Bedrock Titan:</strong> <code>{ "inputText": "${parameters.inputText}" }</code></li>
            <li><strong>Conversational Search (RAG):</strong> <code>{ "model": "${parameters.model}", "messages": ${parameters.messages}, "temperature": ${parameters.temperature} }</code></li>
          </ul>
          Adjust the request body as needed for your connector type.
        '),
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Create Connector'),
    ];

    return $form;
  }

  /**
   * Helper to build connector config based on connector_type.
   */
  protected function buildConnectorConfig(FormStateInterface $form_state): array {
    $connector_type = $form_state->getValue('connector_type');
    $config = [
      'name' => $form_state->getValue('name'),
      'description' => $form_state->getValue('description'),
      'version' => (int) $form_state->getValue('version'),
      'protocol' => $form_state->getValue('protocol'),
      'parameters' => [],
      'credential' => [],
      'actions' => [],
    ];

    switch ($connector_type) {
      case 'openai_self_managed':
        $config['parameters'] = [
          'model' => $form_state->getValue('model'),
        ];
        $config['credential'] = [
          'openAI_key' => $form_state->getValue('openai_key'),
        ];
        $config['actions'][] = [
          'action_type' => 'predict',
          'method' => 'POST',
          'url' => $form_state->getValue('url'),
          'headers' => [
            'Authorization' => 'Bearer ${credential.openAI_key}',
          ],
          'request_body' => $form_state->getValue('request_body'),
          'pre_process_function' => $form_state->getValue('pre_process_function'),
          'post_process_function' => $form_state->getValue('post_process_function'),
        ];
        break;

      case 'open_ai_rag':
        $config['parameters'] = [
          'model' => $form_state->getValue('model'),
          'temperature' => $form_state->getValue('temperature'),
          'messages' => '${parameters.messages}',
        ];
        $config['actions'][] = [
          'action_type' => 'rag',
          'method' => 'POST',
          'url' => $form_state->getValue('url'),
          'headers' => [
            'Authorization' => 'Bearer ${credential.openAI_key}',
          ],
          'request_body' => $form_state->getValue('request_body'),
          'pre_process_function' => $form_state->getValue('pre_process_function'),
          'post_process_function' => $form_state->getValue('post_process_function'),
        ];
        break;

      case 'aws_openai':
        $config['parameters'] = [
          'model' => $form_state->getValue('model'),
          'region' => $form_state->getValue('aws_region'),
          'service_name' => $form_state->getValue('aws_service_name'),
        ];
        $config['credential'] = [
          'secretArn' => $form_state->getValue('secret_arn'),
          'roleArn' => $form_state->getValue('role_arn'),
        ];
        $config['actions'][] = [
          'action_type' => 'predict',
          'method' => 'POST',
          'url' => $form_state->getValue('url'),
          'headers' => [
            'Authorization' => 'Bearer ${credential.secretArn.' . $form_state->getValue('openai_key') . '}',
          ],
          'request_body' => $form_state->getValue('request_body'),
          'pre_process_function' => $form_state->getValue('pre_process_function'),
          'post_process_function' => $form_state->getValue('post_process_function'),
        ];
        break;

      case 'aws_model':
        $config['parameters'] = [
          'region' => $form_state->getValue('aws_region'),
          'service_name' => $form_state->getValue('aws_service_name'),
        ];
        $config['credential'] = [
          'secretArn' => $form_state->getValue('secret_arn'),
          'roleArn' => $form_state->getValue('role_arn'),
          'externalAccountRoleArn' => $form_state->getValue('external_account_role_arn'),
        ];
        $config['actions'][] = [
          'action_type' => 'predict',
          'method' => 'POST',
          'url' => $form_state->getValue('url'),
          'headers' => [
            'content-type' => 'application/json',
            'x-amz-content-sha256' => 'required',
          ],
          'request_body' => $form_state->getValue('request_body'),
          'pre_process_function' => $form_state->getValue('pre_process_function'),
          'post_process_function' => $form_state->getValue('post_process_function'),
        ];
        break;
    }
    return $config;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $existing_connectors = $this->getConnectors();
    if ($existing_connectors && isset($existing_connectors['hits']['hits'])) {
      foreach ($existing_connectors['hits']['hits'] as $connector) {
        if ($connector['_source']['name'] === $form_state->getValue('name')) {
          $this->messenger()->addWarning($this->t('Connector with this name already exists.'));
          return;
        }
      }
    }

    $config = $this->buildConnectorConfig($form_state);
    $response = $this->nlpIngestionService->createConnector($config);
    if ($response) {
      $this->messenger()->addMessage($this->t('Connector created successfully.'));
    }
    else {
      $this->messenger()->addError($this->t('Failed to create connector.'));
    }
  }

  /**
   * Get ML connectors.
   */
  public function getConnectors() {
    try {
      $connectors = $this->nlpIngestionService->getAllConnectors();
      return isset($connectors['hits']['hits']) ? $connectors : NULL;
    }
    catch (\Exception $exception) {
      $this->logger->error('Error fetching ML connectors: @message', ['@message' => $exception->getMessage()]);
      return NULL;
    }
  }

}
