<?php

declare(strict_types=1);

namespace Drupal\opensearch_nlp\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Site\Settings;
use Drupal\search_api\Entity\Index;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for NLP settings.
 */
class NLPConfigForm extends ConfigFormBase {

  /**
   * Constructs a new NLPConfigForm.
   *
   * @param \Drupal\opensearch_nlp\Service\NLPIngestionService $nlpIngestionService
   *   The NLP ingestion service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service for displaying messages.
   */
  public function __construct(
    /**
     * The NLP ingestion service.
     */
    protected $nlpIngestionService,
    /**
     * The state service.
     */
    protected $state,
    /**
     * The messenger service for displaying messages.
     */
    protected $messenger,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): NLPConfigForm {
    return new self(
      $container->get('opensearch_nlp.nlp_ingestion'),
      $container->get('state'),
      $container->get('messenger'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['opensearch_nlp.nlp_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'nlp_config_form';
  }

  /**
   * Get a configuration value with optional settings.php override.
   *
   * @param string $key
   *   The configuration key.
   * @param mixed $default
   *   The default value.
   *
   * @return mixed
   *   The value from settings.php override or config.
   */
  protected function getConfigValue(string $key, mixed $default = NULL): mixed {
    // Check for settings.php override (for environment-specific values).
    $settings_key = "opensearch_nlp.{$key}";
    $settings_override = Settings::get($settings_key);

    if ($settings_override !== NULL) {
      return $settings_override;
    }

    // Fall back to configuration.
    $config = $this->config('opensearch_nlp.nlp_settings');
    return $config->get($key) ?? $default;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('opensearch_nlp.nlp_settings');
    $connector_id = $config->get('connector_id');
    // Get common mapping and embedding fields.
    $common_fields = $this->getCommonMappingAndEmbeddingFields();
    // Enable NLP Checkbox.
    $form['enable_nlp'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable/Disable NLP processing for all indexes.'),
      '#default_value' => $this->getConfigValue('enable_nlp'),
    ];

    // Use an externally hosted model.
    $form['is_externally_hosted_model'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use externally hosted model.'),
      '#default_value' => $this->getConfigValue('is_externally_hosted_model'),
      '#description' => empty($this->nlpIngestionService->getDeployedModels())
        ? $this->t(
        'Enable this option to use an externally hosted model.'
      )
        : $this->t(
        'A deployed model already exists. You cannot create a new model. Please delete the existing model to enable this option.'
      ),
    ];

    // NLP Connector ID.
    $form['connector_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('NLP Connector ID'),
      '#default_value' => $this->getConfigValue('connector_id'),
      '#description' => $this->t('Enter the connector id for externally hosted model.'),
      '#states' => [
        'visible' => [
          ':input[name="is_externally_hosted_model"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['model_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Model Type'),
      '#options' => [
        'local' => $this->t('Local'),
        'openai' => $this->t('OpenAI'),
        'rag' => $this->t('RAG'),
        'aws' => $this->t('AWS'),
      ],
      '#default_value' => $this->getConfigValue('model_type', 'local'),
      '#required' => TRUE,
    ];
    // NLP Model Info Field.
    $form['model_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('NLP Model Path/Name'),
      '#default_value' => $this->getConfigValue('model_path'),
      '#description' => $this->t('Enter the model path or name(for externally hosted models) for NLP processing.'),
      '#required' => TRUE,
    ];

    // NLP model version.
    $form['model_version'] = [
      '#type' => 'textfield',
      '#title' => $this->t('NLP Model Version'),
      '#default_value' => $this->getConfigValue('model_version'),
      '#description' => $this->t('Enter the model version for NLP processing eg. 1.0.1, 1.0.2, 1.0.3 etc.
          <br><strong>Note:</strong> If you are using an externally hosted model, this field is optional and can be left blank.
          <br>For locally hosted models, it is recommended to specify a version to avoid conflicts with future updates.'),
    ];

    // NLP Model format.
    $form['model_format'] = [
      '#type' => 'select',
      '#title' => $this->t('NLP Model Format'),
      '#options' => [
        'TORCH_SCRIPT' => $this->t('TORCH SCRIPT'),
        'ONNX' => $this->t('ONNX'),
      ],
      '#default_value' => $this->getConfigValue('model_format', 'TORCH_SCRIPT'),
      '#description' => $this->t('Select the format of the NLP model. If you are using an externally hosted model, this field is optional.
          <br>For locally hosted models, it is recommended to specify the format to ensure compatibility with the processing pipeline.'),
    ];

    // NLP Model Description.
    $form['model_description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('NLP Model Description'),
      '#default_value' => $this->getConfigValue('model_description'),
      '#description' => $this->t('Enter the model description for NLP processing.'),
      '#required' => TRUE,
    ];

    // NLP Model Group Name.
    $form['model_group'] = [
      '#type' => 'textfield',
      '#title' => $this->t('NLP Model Group Name'),
      '#default_value' => $this->getConfigValue('model_group'),
      '#description' => $this->t('Enter the model group name for NLP processing.'),
      '#required' => TRUE,
    ];

    // NLP Model group description.
    $form['model_group_description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('NLP Model Group Description'),
      '#default_value' => $this->getConfigValue('model_group_description'),
      '#description' => $this->t('Enter the model group description for NLP processing.'),
      '#required' => TRUE,
    ];

    // Multi-index Search Fields.
    $form['multi_index_search_fields'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Multi-index Search Fields(Only for multi-index search).'),
      '#description' => $this->t('These fields are used for configuring multi-index search.'),
    ];

    // Add the search_type field to the fieldset.
    $form['multi_index_search_fields']['search_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Search Type'),
      '#options' => [
        'keyword' => $this->t('Keyword Search'),
        'semantic' => $this->t('Semantic Search'),
        'hybrid' => $this->t('Hybrid Search'),
        'script_score' => $this->t('Knn Score'),
      ],
      '#default_value' => $config->get('search_type') ?? 'keyword',
      '#description' => $this->t('This option is for multi-index search. Select the type of search to perform. 
          <br><strong>Note:</strong> Semantic and Hybrid searches require 
          <ul>
              <li>Properly configured NLP models</li>
              <li>Ingestion pipelines</li>
              <li>Search pipelines</li>
          </ul> 
          If these are not set up, fallback to <strong>Keyword Search</strong>.'),
      '#allowed_tags' => ['br', 'strong', 'ul', 'li'],
    ];

    // Show pagination_depth only if search_type is 'hybrid'.
    $form['multi_index_search_fields']['pagination_depth'] = [
      '#type' => 'select',
      '#title' => $this->t('Pagination Depth'),
      '#options' => [
        10 => $this->t('10'),
        20 => $this->t('20'),
        30 => $this->t('30'),
        40 => $this->t('40'),
        50 => $this->t('50'),
      ],
      '#default_value' => $config->get('pagination_depth') ?? 10,
      '#description' => $this->t('Select the pagination depth for search results.'),
      '#states' => [
        'visible' => [
          ':input[name="search_type"]' => ['value' => 'hybrid'],
        ],
      ],
    ];
    // Add the nearest_neighbors field to the fieldset.
    $form['multi_index_search_fields']['nearest_neighbors'] = [
      '#type' => 'select',
      '#title' => $this->t('Number of Nearest Neighbors(k) for neural or sematic search only.'),
      '#options' => [
        5 => $this->t('5'),
        10 => $this->t('10'),
        20 => $this->t('20'),
        30 => $this->t('30'),
        50 => $this->t('50'),
        60 => $this->t('60'),
        100 => $this->t('100'),
        200 => $this->t('200'),
        300 => $this->t('300'),
        500 => $this->t('500'),
        800 => $this->t('800'),
        1000 => $this->t('1000'),
      ],
      '#default_value' => $config->get('nearest_neighbors') ?? 10,
      '#description' => $this->t('Select the number of nearest neighbors or similar results to retrieve when using Semantic/Neural Search.'),
    ];

    // Add the universal_mapping_field to the fieldset.
    $form['multi_index_search_fields']['universal_mapping_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Universal Mapping Field Name'),
      '#options' => $common_fields['mapping_fields'],
      '#default_value' => $config->get('universal_mapping_field') ?? '',
      '#description' => $this->t('Select a common mapping field that is available across all indexes.'),
      '#access' => !empty($common_fields['mapping_fields']),
    ];

    // Add the universal_embedd_vector_field to the fieldset.
    $form['multi_index_search_fields']['universal_embedd_vector_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Universal Embedding Vector Field Name'),
      '#options' => $common_fields['embedding_fields'],
      '#default_value' => $config->get('universal_embedd_vector_field') ?? '',
      '#description' => $this->t('Select a common embedding vector field that is available across all indexes and mapped to above mapping field.'),
      '#access' => !empty($common_fields['embedding_fields']),
    ];

    // Get available indexes with embedding_vector field.
    $index_options = $this->getIndexesWithEmbeddingVector();

    if (empty($index_options)) {
      $form['search_api_index_message'] = [
        '#markup' => '<p><strong>' . $this->t('No Search API indexes found with an embedding vector field.') . '</strong></p>',
      ];
    }
    else {
      $stored_indexes = $config->get('indexes') ?? [];
      // Initialize an array to track indexes.
      $changed_indexes = [];
      // Find new embedding vector indexes.
      $new_indexes = array_diff_key($index_options, $stored_indexes);
      if (!empty($new_indexes)) {
        $this->cleanNewIndexes($new_indexes);
      }
      $form['indexes'] = [
        '#type' => 'table',
        '#header' => [
          $this->t('Search API Index'),
          $this->t('Enable NLP'),
          $this->t('Search Type'),
          $this->t('Pagination Depth'),
          $this->t('Mapping-Embedding Field Pairs (These are the available mapping-embedding field pairs for this index.)'),
          $this->t('Number of Nearest Neighbors (k)'),
          $this->t('Ingestion Pipeline ID'),
          $this->t('Search Pipeline ID'),
          $this->t('Exclude fields from Search Results'),
        ],
      ];
      foreach ($index_options as $index_id => $index_label) {
        $selected_index = $stored_indexes[$index_id]['enable_nlp'] ?? 0;
        $ingestion_pipeline_id = $stored_indexes[$index_id]['ingestion_pipeline_id'] ?? '';
        $search_pipeline_id = $stored_indexes[$index_id]['search_pipeline_id'] ?? '';
        // Detect changes in mapping-embedding pairs.
        $current_pair = $stored_indexes[$index_id]['mapping_embedding_pairs'] ?? '';
        $submitted_pair = $form_state->getValue(['indexes', $index_id, 'mapping_embedding_pairs']) ?? '';
        if ($current_pair !== $submitted_pair) {
          $changed_indexes[$index_id] = $index_label;
        }
        $form['indexes'][$index_id]['index_name'] = [
          '#markup' => $index_label,
        ];

        $form['indexes'][$index_id]['enable_nlp'] = [
          '#type' => 'checkbox',
          '#default_value' => $selected_index,
        ];

        $form['indexes'][$index_id]['search_type'] = [
          '#type' => 'select',
          '#description' => $this->t('Select the type of search to perform on this index. If NLP not enabled, default to keyword search.'),
          '#options' => [
            'script_score' => $this->t('Knn Score'),
            'semantic' => $this->t('Semantic Search'),
            'hybrid' => $this->t('Hybrid Search'),
          ],
          '#default_value' => $stored_indexes[$index_id]['search_type'] ?? 'keyword',
        ];

        $form['indexes'][$index_id]['pagination_depth'] = [
          '#type' => 'select',
          '#options' => [
            10 => $this->t('10'),
            20 => $this->t('20'),
            30 => $this->t('30'),
            40 => $this->t('40'),
            50 => $this->t('50'),
          ],
          '#default_value' => $stored_indexes[$index_id]['pagination_depth'] ?? 10,
          '#description' => $this->t('Select the pagination depth for search results.'),
        ];
        // Fetch pairs of mapping-embedding fields for the index.
        $pairs = $this->getMappingEmbeddingPairs($index_id);
        $pair_options = [];
        foreach ($pairs as $pair) {
          $pair_options[$pair['mapping_field'] . '|' . $pair['embedding_field']] = $this->t(
            '@mapping_field -> @embedding_field',
            [
              '@mapping_field' => $pair['mapping_field'],
              '@embedding_field' => $pair['embedding_field'],
            ]
          );
        }
        $selected_pair = $stored_indexes[$index_id]['mapping_embedding_pairs'] ?? '';
        // Add a multiselect field to display mapping-embedding pairs.
        $form['indexes'][$index_id]['mapping_embedding_pairs'] = [
          '#type' => 'select',
          '#options' => $pair_options,
          '#default_value' => $selected_pair,
          '#states' => [
            'visible' => [
              ':input[name="indexes[' . $index_id . '][enable_nlp]"]' => ['checked' => TRUE],
            ],
          ],
        ];

        // Add the nearest_neighbors field to the fieldset.
        $form['indexes'][$index_id]['nearest_neighbors'] = [
          '#type' => 'select',
          '#options' => [
            5 => $this->t('5'),
            10 => $this->t('10'),
            20 => $this->t('20'),
            30 => $this->t('30'),
            50 => $this->t('50'),
            60 => $this->t('60'),
            100 => $this->t('100'),
            200 => $this->t('200'),
            300 => $this->t('300'),
            500 => $this->t('500'),
            800 => $this->t('800'),
            1000 => $this->t('1000'),
          ],
          '#default_value' => $stored_indexes[$index_id]['nearest_neighbors'] ?? 10,
          '#states' => [
            'visible' => [
              ':input[name="indexes[' . $index_id . '][enable_nlp]"]' => ['checked' => TRUE],
            ],
          ],
        ];

        $form['indexes'][$index_id]['ingestion_pipeline_id'] = [
          '#type' => 'textfield',
          '#default_value' => $ingestion_pipeline_id ?: $index_id . '-nlp-ingestion-pipeline',
          '#size' => 20,
          '#disabled' => TRUE,
          '#states' => [
            'visible' => [
              ':input[name="indexes[' . $index_id . '][mapping_field]"]' => ['filled' => TRUE],
            ],
          ],
        ];

        $form['indexes'][$index_id]['search_pipeline_id'] = [
          '#type' => 'textfield',
          '#default_value' => $search_pipeline_id ?: $index_id . '-nlp-search-pipeline',
          '#size' => 20,
          '#disabled' => TRUE,
          '#states' => [
            'visible' => [
              ':input[name="indexes[' . $index_id . '][mapping_field]"]' => ['filled' => TRUE],
            ],
          ],
        ];

        // Get all available fields for this index.
        $index = Index::load($index_id);
        $fields = $index->getFields();
        $field_options = [];
        foreach ($fields as $field_id => $field) {
          $field_options[$field_id] = $field->getLabel() . ' (' . $field_id . ')';
        }

        $form['indexes'][$index_id]['exclude_from_search_results'] = [
          '#type' => 'select',
          '#options' => $field_options,
          '#default_value' => $stored_indexes[$index_id]['exclude_from_search_results'] ?? [],
          '#multiple' => TRUE,
          '#description' => $this->t('Select one or more fields to exclude from search results for this index.'),
          '#states' => [
            'visible' => [
              ':input[name="indexes[' . $index_id . '][enable_nlp]"]' => ['checked' => TRUE],
            ],
          ],
        ];
      }

      if (!empty($changed_indexes)) {
        $this->cleanNewIndexes($changed_indexes);
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Get mapping-embedding field pairs for a given index.
   *
   * @param string $index_id
   *   The ID of the index.
   *
   * @return array
   *   An array of mapping-embedding field pairs.
   */
  private function getMappingEmbeddingPairs(int|string $index_id): array {
    $index = Index::load($index_id);
    $fields = $index->getFields();
    $pairs = [];
    foreach ($fields as $field_id => $field) {
      if ($field->getType() === 'opensearch_text_embedding_vector') {
        // Check for corresponding mapping fields.
        foreach ($fields as $mapping_field_id => $mapping_field) {
          if ($this->isValidMappingField($mapping_field, $mapping_field_id, $field_id, $field->getPropertyPath())) {
            $pairs[] = [
              'mapping_field' => $mapping_field_id,
              'embedding_field' => $field_id,
            ];
          }
        }
      }
    }

    return $pairs;
  }

  /**
   * Cleans new embedding vector indexes.
   *
   * @param array $new_indexes
   *   An array of new indexes with their IDs as keys and labels as values.
   */
  private function cleanNewIndexes(array $new_indexes): void {
    foreach (array_keys($new_indexes) as $new_index_id) {
      // Delete pending server tasks for the new index.
      $this->nlpIngestionService->deletePendingServerTasks($new_index_id);
    }
  }

  /**
   * Get a list of Search API indexes that contain an embedding_vector field.
   */
  private function getIndexesWithEmbeddingVector(): array {
    $search_api_indexes = Index::loadMultiple();
    $index_options = [];

    foreach ($search_api_indexes as $index_id => $index) {
      $fields = $index->getFields();
      foreach ($fields as $field) {
        if ($field->getType() === 'opensearch_text_embedding_vector') {
          $index_options[$index_id] = $index->label();
          break;
        }
      }
    }

    return $index_options;
  }

  /**
   * Get common mapping and embedding fields across all indexes.
   *
   * @return array
   *   An array containing two keys:
   *   - 'mapping_fields': An array of common mapping fields.
   *   - 'embedding_fields': An array of common embedding fields.
   */
  private function getCommonMappingAndEmbeddingFields(): array {
    $search_api_indexes = Index::loadMultiple();
    $common_pairs = [];

    // Iterate through each index to find mapping-embedding field pairs.
    foreach ($search_api_indexes as $index_id => $index) {
      $fields = $index->getFields();
      $index_pairs = [];

      foreach ($fields as $field_id => $field) {
        if ($field->getType() === 'opensearch_text_embedding_vector') {
          // Check for corresponding mapping fields.
          foreach ($fields as $mapping_field_id => $mapping_field) {
            if ($this->isValidMappingField($mapping_field, $mapping_field_id, $field_id, $field->getPropertyPath())) {
              $index_pairs[] = [
                'mapping_field' => $mapping_field_id,
                'embedding_field' => $field_id,
              ];
            }
          }
        }
      }

      // Only consider indexes that have at least one mapping-embedding pair.
      if (!empty($index_pairs)) {
        $mapping_embedding_pairs[$index_id] = $index_pairs;
        // If this is the first valid index, initialize the common pairs.
        if (empty($common_pairs)) {
          $common_pairs = $index_pairs;
        }
        else {
          // Find the intersection of pairs with the previous indexes.
          $common_pairs = array_uintersect($common_pairs, $index_pairs, function (array $a, array $b): int {
            return strcmp($a['mapping_field'] . $a['embedding_field'], $b['mapping_field'] . $b['embedding_field']);
          });
        }
      }
    }

    // Extract mapping and embedding fields from the common pairs.
    $mapping_fields = [];
    $embedding_fields = [];
    foreach ($common_pairs as $pair) {
      $mapping_fields[$pair['mapping_field']] = $pair['mapping_field'];
      $embedding_fields[$pair['embedding_field']] = $pair['embedding_field'];
    }

    return [
      'mapping_fields' => $mapping_fields,
      'embedding_fields' => $embedding_fields,
    ];
  }

  /**
   * Checks if a field is valid for mapping.
   */
  private function isValidMappingField($field, $field_id, $selected_embedding_field, $embedding_property_path): bool {
    return $field->getType() !== 'opensearch_text_embedding_vector'
      && $field_id !== $selected_embedding_field
      && $field->getPropertyPath() == $embedding_property_path
      && in_array($field->getType(), ['string', 'text']);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // Check if 'is_externally_hosted_model' is checked.
    if ($form_state->getValue('is_externally_hosted_model')) {
      $connector_id = $form_state->getValue('connector_id');

      // Ensure the connector_id is not empty.
      if (empty($connector_id)) {
        $form_state->setErrorByName('connector_id', $this->t('The NLP Connector ID is required when using an externally hosted model.'));
      }
      else {
        // Validate the ML connector, if it exists or not.
        $response = $this->nlpIngestionService->getConnector($connector_id);
        if ($response === NULL || $response['hits']['hits'][0]['_id'] != $connector_id) {
          $form_state->setErrorByName('connector_id', $this->t('The entered NLP Connector ID is invalid or does not exist.'));
        }
      }
    }

    $indexes = $form_state->getValue('indexes');
    if (empty($indexes)) {
      return;
    }

    foreach ($indexes as $index_id => $settings) {
      foreach (['ingestion_pipeline_id', 'search_pipeline_id'] as $field) {
        $value = trim((string) $settings[$field]);
        // Check if the value is empty.
        if (empty($value)) {
          $form_state->setErrorByName(sprintf('indexes][%s][%s', $index_id, $field), $this->t('@field is required.', ['@field' => ucfirst(str_replace('_', ' ', $field))]));
        }
        elseif (!preg_match('/^[a-zA-Z0-9_-]+$/', $value)) {
          $form_state->setErrorByName(sprintf('indexes][%s][%s', $index_id, $field), $this->t('@field must contain only letters, numbers, underscores, and hyphens.', ['@field' => ucfirst(str_replace('_', ' ', $field))]));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->configFactory->getEditable('opensearch_nlp.nlp_settings');
    $config->set('enable_nlp', $form_state->getValue('enable_nlp'))
      ->set('is_externally_hosted_model', $form_state->getValue('is_externally_hosted_model'))
      ->set('connector_id', $form_state->getValue('connector_id'))
      ->set('model_type', $form_state->getValue('model_type'))
      ->set('model_path', $form_state->getValue('model_path'))
      ->set('model_version', $form_state->getValue('model_version'))
      ->set('pagination_depth', $form_state->getValue('pagination_depth'))
      ->set('model_format', $form_state->getValue('model_format'))
      ->set('model_group', $form_state->getValue('model_group'))
      ->set('model_description', $form_state->getValue('model_description'))
      ->set('model_group_description', $form_state->getValue('model_group_description'))
      ->set('search_type', $form_state->getValue('search_type'))
      ->set('universal_embedd_vector_field', $form_state->getValue('universal_embedd_vector_field'))
      ->set('universal_mapping_field', $form_state->getValue('universal_mapping_field'))
      ->set('nearest_neighbors', $form_state->getValue('nearest_neighbors'));
    // Save per-index settings.
    $is_external_model = $form_state->getValue('is_externally_hosted_model') ? 1 : 0;
    $connector_id = $form_state->getValue('connector_id');
    $indexes = $form_state->getValue('indexes');
    $stored_indexes = $config->get('indexes') ?? [];
    // Detect newly added indexes.
    $new_indexes = empty($indexes) ? [] : array_diff_key($indexes, $stored_indexes);
    // Save the last new index ID to the state system.
    if (!empty($new_indexes)) {
      $last_new_index_id = array_key_last($new_indexes);
      $this->state->set('opensearch_nlp.last_new_index', $last_new_index_id);
    }

    $stored_indexes = [];
    if (empty($indexes)) {
      $config->set('indexes', $stored_indexes)->save();
      parent::submitForm($form, $form_state);
      return;
    }

    foreach ($indexes as $index_id => $settings) {
      $stored_indexes[$index_id] = [
        'enable_nlp' => $settings['enable_nlp'],
        'mapping_embedding_pairs' => $settings['mapping_embedding_pairs'] ?? '',
        'ingestion_pipeline_id' => $settings['ingestion_pipeline_id'],
        'search_pipeline_id' => $settings['search_pipeline_id'],
        'search_type' => $settings['search_type'],
        'pagination_depth' => $settings['pagination_depth'],
        'nearest_neighbors' => $settings['nearest_neighbors'],
        'exclude_from_search_results' => $settings['exclude_from_search_results'] ?? [],
      ];
      // Register the search pipeline for this index.
      $this->nlpIngestionService->registerSearchPipeline($settings['search_pipeline_id']);
    }
    $this->state->delete('opensearch_nlp.last_new_index');
    $config->set('indexes', $stored_indexes)->save();
    parent::submitForm($form, $form_state);
    $this->nlpIngestionService->registerModelAndIngestPipeline($is_external_model, $connector_id);

    // Redirect to the "Index Fields" page of the last new index if it exists.
    if (!empty($new_indexes)) {
      $this->messenger()->addMessage($this->t('Please resave this index fields for the newly added NLP config changes to take place.'));
      // Delete the state before redirecting.
      $this->state->delete('opensearch_nlp.last_new_index');
      $form_state->setRedirect('entity.search_api_index.fields', ['search_api_index' => $last_new_index_id]);
    }
  }

}
