<?php

declare(strict_types=1);

namespace Drupal\opensearch_nlp\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\opensearch_nlp\Service\NLPIngestionService;

/**
 * Provides a form for testing NLP ingestion pipelines.
 */
class NLPIngestionPipelineTestForm extends FormBase {

  /**
   * Constructs a new NLPIngestionPipelineTestForm.
   *
   * @param \Drupal\opensearch_nlp\Service\NLPIngestionService $nlpIngestionService
   *   The NLP ingestion service.
   */
  public function __construct(protected NLPIngestionService $nlpIngestionService) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): NLPIngestionPipelineTestForm {
    return new self(
      $container->get('opensearch_nlp.nlp_ingestion')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'test_nlp_ingestion_pipeline_form';
  }

  /**
   * Builds the form for testing NLP ingestion pipelines.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param string|null $pipeline_name
   *   The name of the pipeline to test.
   * @param string|null $index
   *   The index to use for the test.
   * @param string|null $mapping_field
   *   The mapping field to use for the test.
   *
   * @return array
   *   The form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state, $pipeline_name = NULL, $index = NULL, $mapping_field = NULL): array {
    // Attach Drupal AJAX library.
    $form['#attached']['library'][] = 'core/drupal.ajax';

    // Hidden fields for pipeline_name, model_id, index, and mapping_field.
    $form['pipeline_name'] = [
      '#type' => 'hidden',
      '#value' => $pipeline_name,
    ];

    $form['index'] = [
      '#type' => 'hidden',
      '#value' => $index,
    ];

    $form['mapping_field'] = [
      '#type' => 'hidden',
      '#value' => $mapping_field,
    ];

    // Text area for user to input text.
    $form['input_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Enter Text'),
      '#description' => $this->t('Enter the text to predict using the selected model.'),
      '#required' => TRUE,
    ];

    // Submit button with AJAX callback.
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Predict'),
      '#ajax' => [
        'callback' => '::ajaxSubmit',
        'wrapper' => 'predict-result',
        'progress' => ['type' => 'throbber'],
      ],
    ];

    // Markup to show prediction result.
    $form['prediction_result'] = [
      '#type' => 'markup',
      '#markup' => '<div id="predict-result"></div>',
    ];

    return $form;
  }

  /**
   * AJAX callback to handle form submission and update the result container.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response object.
   */
  public function ajaxSubmit(array &$form, FormStateInterface $form_state): AjaxResponse {
    // Get the model ID and input text from the form state.
    $pipeline_name = $form_state->getValue('pipeline_name');
    $index = $form_state->getValue('index');
    $mapping_field = $form_state->getValue('mapping_field');
    $input_text = $form_state->getValue('input_text');
    // Create AJAX response to update the result container.
    $ajax_response = new AjaxResponse();
    // Clear previous result before setting new data.
    $ajax_response->addCommand(new ReplaceCommand('#predict-result', '<div id="predict-result"></div>'));

    if ($pipeline_name && $input_text) {
      // Call the NLP ingestion service to get the prediction.
      $response = $this->nlpIngestionService->testIngestionPipeline($pipeline_name, $index, $input_text, $mapping_field);
      // Process the response.
      if ($response) {
        $result = '<pre>' . json_encode($response, JSON_PRETTY_PRINT) . '</pre>';
      }
      else {
        $result = '<p><strong>Error:</strong> Failed to get a prediction.</p>';
      }
    }
    else {
      $result = '<p><strong>Error:</strong> Model ID or text is missing.</p>';
    }
    // Update the result container with new data.
    $ajax_response->addCommand(new ReplaceCommand('#predict-result', '<div id="predict-result">' . $result . '</div>'));

    return $ajax_response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Not used.
  }

}
