<?php

declare(strict_types=1);

namespace Drupal\opensearch_nlp\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\opensearch_nlp\Service\NLPIngestionService;

/**
 * Form to enter text for NLP model prediction.
 */
class NLPModelPredictForm extends FormBase {

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a new PredictForm.
   */
  public function __construct(
    /**
     * NLP ingestion service.
     */
    protected NLPIngestionService $nlpIngestionService,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    $this->logger = $loggerFactory->get('opensearch_nlp');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): NLPModelPredictForm {
    return new self(
      $container->get('opensearch_nlp.nlp_ingestion'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'opensearch_nlp_predict_form';
  }

  /**
   * Build the prediction form.
   */
  public function buildForm(array $form, FormStateInterface $form_state, $model_id = NULL): array {
    $form['#attached']['library'][] = 'core/drupal.ajax';

    $form['model_id'] = [
      '#type' => 'hidden',
      '#value' => $model_id,
    ];

    $form['input_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Enter Text'),
      '#description' => $this->t('Enter the text to predict using the selected model.'),
      '#required' => TRUE,
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Predict'),
      '#ajax' => [
        'callback' => '::ajaxSubmit',
        'wrapper' => 'predict-result',
        'progress' => ['type' => 'throbber'],
      ],
    ];

    $form['prediction_result'] = [
      '#type' => 'markup',
      '#markup' => '<div id="predict-result"></div>',
    ];

    return $form;
  }

  /**
   * AJAX callback function to handle prediction.
   */
  public function ajaxSubmit(array &$form, FormStateInterface $form_state): AjaxResponse {
    $model_id = $form_state->getValue('model_id');
    $input_text = $form_state->getValue('input_text');

    $ajax_response = new AjaxResponse();

    // Clear previous result before setting new data.
    $ajax_response->addCommand(new ReplaceCommand('#predict-result', '<div id="predict-result"></div>'));

    if ($model_id && $input_text) {
      $response = $this->nlpIngestionService->predictModel($model_id, $input_text);
      if ($response) {
        $result = '<pre>' . json_encode($response, JSON_PRETTY_PRINT) . '</pre>';
      }
      else {
        $result = '<p><strong>Error:</strong> Failed to get a prediction.</p>';
      }
    }
    else {
      $result = '<p><strong>Error:</strong> Model ID or text is missing.</p>';
    }

    // Update the result container with new data.
    $ajax_response->addCommand(new ReplaceCommand('#predict-result', '<div id="predict-result">' . $result . '</div>'));

    return $ajax_response;
  }

  /**
   * Placeholder function for model prediction logic.
   */
  public function predict($model_id, $text): string {
    return sprintf("Predicted output for '%s' using Model ID: %s", $text, $model_id);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // No default form submission as we're using AJAX.
  }

}
