<?php

declare(strict_types=1);

namespace Drupal\opensearch_nlp\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides a configuration form for OpenSearch KNN vector settings.
 */
class NLPVectorConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['opensearch_nlp.vector_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'nlp_vector_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('opensearch_nlp.vector_settings');

    // Vector Type (Fixed to knn_vector).
    $form['vector_type'] = [
      '#type' => 'item',
      '#title' => $this->t('Vector Type'),
      '#markup' => '<strong>knn_vector</strong>',
    ];

    // Dimension Selection.
    $form['dimension'] = [
      '#type' => 'select',
      '#title' => $this->t('Vector Dimension'),
      '#options' => [
        '128' => $this->t('128'),
        '256' => $this->t('256'),
        '384' => $this->t('384'),
        '512' => $this->t('512'),
        '768' => $this->t('768 (Default for BERT)'),
        '1024' => $this->t('1024'),
        '1536' => $this->t('1536'),
      ],
      '#default_value' => $config->get('dimension') ?? '768',
      '#description' => $this->t('Choose the dimension (size) of the vector. Higher dimensions (e.g., 768 or 1024) are suitable for more complex models (like BERT) but require more computational resources. Lower dimensions might reduce accuracy but are faster and more efficient.'),
    ];

    // Search Method Selection.
    $form['method'] = [
      '#type' => 'select',
      '#title' => $this->t('Search Method'),
      '#options' => [
        'hnsw' => $this->t('HNSW (Hierarchical Navigable Small World)'),
      ],
      '#default_value' => $config->get('method') ?? 'hnsw',
      '#description' => $this->t('Choose the search method used for finding the nearest neighbors. HNSW (Hierarchical Navigable Small World) is fast and scalable for large datasets.'),
    ];

    // Space Type Selection.
    $form['space_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Distance Metric (Space Type)'),
      '#options' => [
        'l2' => $this->t('L2 Distance (Euclidean)'),
        'cosinesimil' => $this->t('Cosine Similarity'),
        'dot_product' => $this->t('Dot Product'),
      ],
      '#default_value' => $config->get('space_type') ?? 'l2',
      '#description' => $this->t('Select the distance metric (space type) used for comparing vectors. "L2" (Euclidean) measures straight-line distance, "cosine" measures similarity in direction, and "dot product" calculates the degree of similarity between vectors. The choice impacts the accuracy and performance of the search.'),
    ];

    $form['engine'] = [
      '#type' => 'select',
      '#title' => $this->t('Engine'),
      '#default_value' => $config->get('engine') ?? 'lucene',
      '#options' => [
        'lucene' => $this->t('Lucene'),
        'faiss' => $this->t('Faiss'),
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->configFactory->getEditable('opensearch_nlp.vector_settings')
      ->set('dimension', $form_state->getValue('dimension'))
      ->set('method', $form_state->getValue('method'))
      ->set('space_type', $form_state->getValue('space_type'))
      ->set('engine', $form_state->getValue('engine'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
