<?php

declare(strict_types=1);

namespace Drupal\opensearch_nlp\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\opensearch_nlp\Service\NLPIngestionService;

/**
 * Provides a configuration form for the search pipeline settings.
 */
class SearchPipelineConfigForm extends ConfigFormBase {

  /**
   * Constructs a new NLPConfigForm.
   *
   * @param \Drupal\opensearch_nlp\Service\NLPIngestionService $nlpIngestionService
   *   The NLP ingestion service.
   */
  public function __construct(protected NLPIngestionService $nlpIngestionService) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): SearchPipelineConfigForm {
    return new self(
      $container->get('opensearch_nlp.nlp_ingestion')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['opensearch_nlp.search_pipeline_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'search_pipeline_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('opensearch_nlp.search_pipeline_settings');

    // Enable Multiindex Hybrid search.
    $form['enable_multi_index_hybrid_search'] = [
      '#type' => 'checkbox',
      "#disabled" => $this->nlpIngestionService->isExistingSearchPipeline('multi_index_hybrid_search_pipeline'),
      '#title' => $this->t('Create Multi-Index Hybrid Search Pipeline'),
      '#default_value' => $config->get('enable_multi_index_hybrid_search'),
    ];

    // Normalization Technique.
    $form['normalization_technique'] = [
      '#type' => 'select',
      '#title' => $this->t('Normalization Technique'),
      '#description' => $this->t('Choose how search scores are normalized for combination.'),
      '#options' => [
        'min_max' => $this->t('Min-Max Scaling (Default)'),
        'l2' => $this->t('L2 Normalization'),
      ],
      '#default_value' => $config->get('normalization_technique') ?? 'min_max',
    ];

    // Combination Technique.
    $form['combination_technique'] = [
      '#type' => 'select',
      '#title' => $this->t('Combination Technique'),
      '#description' => $this->t('Choose how to combine keyword and semantic scores.'),
      '#options' => [
        'arithmetic_mean' => $this->t('Arithmetic Mean (Default)'),
        'geometric_mean' => $this->t('Geometric Mean (Balances Differences)'),
        'harmonic_mean' => $this->t('Harmonic Mean (Penalizes Weak Matches)'),
      ],
      '#default_value' => $config->get('combination_technique') ?? 'arithmetic_mean',
    ];

    // Combination Weights.
    $form['combination_weights'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Combination Weights'),
      '#description' => $this->t('Enter weights for BM25 and k-NN (comma-separated, e.g., 0.5,0.5). Must sum to 1.0.'),
      '#default_value' => $config->get('combination_weights') ?? '0.5,0.5',
    ];

    // Tag.
    $form['tag'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Processor Tag'),
      '#description' => $this->t('Optional identifier for this processor.'),
      '#default_value' => $config->get('tag') ?? 'search-pipeline',
    ];

    // Description.
    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Processor Description'),
      '#description' => $this->t('Optional description for this processor.'),
      '#default_value' => $config->get('description') ?? 'description for this pipeline',
    ];

    // Add a table with explanations.
    $form['explanation_table'] = [
      '#type' => 'table',
      '#header' => [$this->t('Setting'), $this->t('Impact')],
      '#rows' => [
        [$this->t('Normalization: Min-Max'), $this->t('Scales all scores between 0 and 1 for fair combination.')],
        [
          $this->t('Normalization: L2'),
          $this->t('Adjusts scores based on vector magnitude, useful for embeddings.'),
        ],
        [
          $this->t('Combination: Arithmetic Mean'),
          $this->t('Balances keyword and semantic results equally.'),
        ],
        [
          $this->t('Combination: Geometric Mean'),
          $this->t('Works well when both keyword and semantic results are strong.'),
        ],
        [
          $this->t('Combination: Harmonic Mean'),
          $this->t('Penalizes weak matches, boosting strong matches.'),
        ],
        [
          $this->t('Weights: 0.5,0.5'),
          $this->t('Balanced approach between BM25 and k-NN.'),
        ],
        [
          $this->t('Weights: 0.7,0.3'),
          $this->t('Prioritizes keyword search more.'),
        ],
        [$this->t('Weights: 0.3,0.7'), $this->t('Prioritizes semantic search more.')],
      ],
      '#caption' => $this->t('Explanation of Search Pipeline Settings'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('opensearch_nlp.search_pipeline_settings')
      ->set('enable_multi_index_hybrid_search', $form_state->getValue('enable_multi_index_hybrid_search'))
      ->set('normalization_technique', $form_state->getValue('normalization_technique'))
      ->set('combination_technique', $form_state->getValue('combination_technique'))
      ->set('combination_weights', $form_state->getValue('combination_weights'))
      ->set('tag', $form_state->getValue('tag'))
      ->set('description', $form_state->getValue('description'))
      ->save();

    parent::submitForm($form, $form_state);
    if ($form_state->getValue('enable_multi_index_hybrid_search')) {
      $pipeline_id = 'multi_index_hybrid_search_pipeline';
      $this->nlpIngestionService->registerSearchPipeline($pipeline_id);
    }
    else {
      $this->messenger()->addStatus($this->t('Search Pipeline settings saved.'));
    }
  }

}
