<?php

declare(strict_types=1);

namespace Drupal\opensearch_nlp\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Provides an edit form for a specific search pipeline.
 */
class SearchPipelineEditForm extends FormBase {

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a new SearchPipelineEditForm.
   */
  public function __construct(
    /**
     * The NLP ingestion service.
     *
     * @var \Drupal\opensearch_nlp\Service\NLPIngestionService
     *   The NLP ingestion service.
     */
    protected $nlpIngestionService,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    $this->nlpIngestionService = $nlpIngestionService;
    $this->logger = $loggerFactory->get('opensearch_nlp');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): SearchPipelineEditForm {
    return new self(
      $container->get('opensearch_nlp.nlp_ingestion'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'search_pipeline_edit_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $pipeline_id = NULL) {
    if (!$pipeline_id) {
      $this->messenger()->addError($this->t('Invalid pipeline ID.'));
      return $form;
    }

    $pipeline = $this->nlpIngestionService->getSearchPipeline($pipeline_id);
    if (!$pipeline) {
      $this->messenger()->addError($this->t('Pipeline not found.'));
      return $form;
    }

    $details = $pipeline[$pipeline_id];

    $form['pipeline_id'] = [
      '#type' => 'hidden',
      '#value' => $pipeline_id,
    ];

    $form['normalization_technique'] = [
      '#type' => 'select',
      '#title' => $this->t('Normalization Technique'),
      '#options' => [
        'min_max' => $this->t('Min-Max Scaling (Default)'),
        'l2' => $this->t('L2 Normalization'),
      ],
      '#default_value' => $details['phase_results_processors'][0]['normalization-processor']['normalization']['technique'] ?? 'min_max',
    ];

    $form['combination_technique'] = [
      '#type' => 'select',
      '#title' => $this->t('Combination Technique'),
      '#options' => [
        'arithmetic_mean' => $this->t('Arithmetic Mean (Default)'),
        'geometric_mean' => $this->t('Geometric Mean'),
        'harmonic_mean' => $this->t('Harmonic Mean'),
      ],
      '#default_value' => $details['phase_results_processors'][0]['normalization-processor']['combination']['technique'] ?? 'arithmetic_mean',
    ];

    $form['combination_weights'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Combination Weights'),
      '#default_value' => $details['phase_results_processors'][0]['normalization-processor']['combination']['parameters']['weights'] ?? '0.5,0.5',
    ];

    $form['tag'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Processor Tag'),
      '#default_value' => $details['tag'] ?? '',
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Processor Description'),
      '#default_value' => $details['description'] ?? '',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save Changes'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $pipeline_id = $form_state->getValue('pipeline_id');
    $updated_values = [
      'normalization_technique' => $form_state->getValue('normalization_technique'),
      'combination_technique' => $form_state->getValue('combination_technique'),
      'combination_weights' => $form_state->getValue('combination_weights'),
      'tag' => $form_state->getValue('tag'),
      'description' => $form_state->getValue('description'),
    ];

    // Call API to update pipeline configuration.
    $success = $this->nlpIngestionService->updateSearchPipeline($pipeline_id, $updated_values);

    if ($success) {
      $this->messenger()->addMessage($this->t('Search pipeline updated successfully.'));

      // Redirect to ingestion pipelines list page.
      $form_state->setRedirect('opensearch_nlp.ingestion_pipelines');
    }
    else {
      $this->messenger()->addError($this->t('Failed to update search pipeline.'));
    }
  }

}
