<?php

declare(strict_types=1);

namespace Drupal\opensearch_nlp\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\State\StateInterface;
use Drupal\search_api\Entity\Server;
use GuzzleHttp\Exception\RequestException;
use OpenSearch\Common\Exceptions\Missing404Exception;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\search_api\Task\ServerTaskManagerInterface;
use OpenSearch\Client;

/**
 * Service for NLP ingestion and model registration in OpenSearch.
 */
class NLPIngestionService {

  /**
   * Maximum number of retries for model registration task polling.
   */
  private const int MAX_RETRIES = 10;

  /**
   * Retry interval in seconds for model registration task polling.
   */
  private const int RETRY_INTERVAL = 5;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  public $state;

  /**
   * The server task manager service.
   *
   * @var \Drupal\search_api\Task\ServerTaskManagerInterface
   */
  public $taskManager;

  /**
   * The OpenSearch client.
   *
   * @var \OpenSearch\Client
   */
  protected $client;

  /**
   * The test server.
   *
   * @var \Drupal\search_api\ServerInterface
   */
  protected $server;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructor.
   */
  public function __construct(
    LoggerChannelFactoryInterface $logger,
    /**
     * The configuration factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory,
    /**
     * The state service.
     *
     * @var \Drupal\Core\State\StateInterface
     */
    StateInterface $state,
    /**
     * The Search API task manager.
     *
     * @var \Drupal\search_api\Task\ServerTaskManagerInterface
     */
    ServerTaskManagerInterface $taskManager,
  ) {
    $this->state = $state;
    $this->taskManager = $taskManager;
    $this->client = $this->getClient();
    $this->logger = $logger->get('opensearch_nlp');
    $this->server = Server::load('opensearch');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): NLPIngestionService {
    return new self(
      $container->get('logger.factory'),
      $container->get('config.factory'),
      $container->get('state'),
      $container->get('search_api.server_task_manager')
    );
  }

  /**
   * Get the OpenSearch client.
   *
   * @return \OpenSearch\Client
   *   The OpenSearch client.
   */
  public function getClient(): Client {
    if (!$this->client) {
      $server = Server::load('opensearch');
      if ($server) {
        $backend = $server->getBackend();
        // @phpstan-ignore-next-line
        $this->client = $backend->getClient();
      }
      else {
        throw new \Exception('OpenSearch server not found.');
      }
    }
    return $this->client;
  }

  /**
   * Register a model group in OpenSearch if it isn't already registered.
   */
  public function registerModelGroup(): ?string {
    $model_groups = $this->getModelGroups();
    if (isset($model_groups['hits']['hits']) && !empty($model_groups['hits']['hits'])) {
      return $model_groups['hits']['hits'][0]['_id'];
    }
    else {
      $modelGroupId = $this->state->get('nlp_ingestion.model_group_id');
      // Fetch NLP configuration.
      $config = $this->configFactory->get('opensearch_nlp.nlp_settings');
      if ($modelGroupId !== NULL) {
        $this->logger->notice('Model group already registered with ID: @id', ['@id' => $modelGroupId]);
        return $modelGroupId;
      }
      try {
        $modelGroupResponse = $this->client->ml()->registerModelGroup([
          'body' => [
            'name' => $config->get('model_group') ?? 'nlp-model-group',
            'description' => $config->get('model_group_description') ?? 'Model group for NLP models',
          ],
        ]);

        $modelGroupId = $modelGroupResponse['model_group_id'] ?? NULL;

        if ($modelGroupId) {
          $this->state->set('nlp_ingestion.model_group_id', $modelGroupId);
          $this->logger->notice('Registered model group with ID: @id', ['@id' => $modelGroupId]);
        }
        return $modelGroupId;
      }
      catch (RequestException $exception) {
        $this->logger->error('Failed to register model group: @message', ['@message' => $exception->getMessage()]);
        return NULL;
      }
    }
  }

  /**
   * Register a model and ingest pipeline in OpenSearch.
   */
  public function registerModelAndIngestPipeline($is_external_model = FALSE, $connector_id = ''): ?string {
    try {
      $modelId = NULL;
      // Register the model group.
      $modelGroupId = $this->state->get('nlp_ingestion.model_group_id') != NULL ? $this->state->get('nlp_ingestion.model_group_id') : $this->registerModelGroup();
      if (!$modelGroupId) {
        $this->logger->error("Model group registration failed. Cannot proceed with model registration.");
        return NULL;
      }
      $deployed_modelId = '';
      $deployed_model = $this->getModelByGroupId($modelGroupId);
      if (!empty($deployed_model) && isset($deployed_model['hits']['hits']) && !empty($deployed_model['hits']['hits'])) {
        $deployed_modelId = $deployed_model['hits']['hits'][0]['_id'];
        $this->state->set('nlp_ingestion.model_id', $deployed_modelId);
      }
      // Fetch model path from configuration.
      $config = $this->configFactory->get('opensearch_nlp.nlp_settings');
      $modelPath = $config->get('model_path') ?? 'huggingface/sentence-transformers/msmarco-distilbert-base-tas-b';
      $model_version = $config->get('model_version') ?? '1.0.2';
      $model_format = $config->get('model_format') ?? 'TORCH_SCRIPT';
      $model_description = $config->get('model_description') ?? 'Model for text embedding';
      if ($this->state->get('nlp_ingestion.model_id') == NULL) {
        // Register the model.
        if ($is_external_model) {
          $registerModelResponse = $this->client->ml()->registerModel([
            'body' => [
              'name' => $modelPath,
              'function_name' => 'remote',
              'description' => $model_description,
              'model_group_id' => $modelGroupId,
              'connector_id' => $connector_id,
            ],
          ]);
        }
        else {
          $registerModelResponse = $this->client->ml()->registerModel([
            'body' => [
              'name' => $modelPath,
              'model_group_id' => $modelGroupId,
              'model_format' => $model_format,
              'version' => $model_version,
            ],
          ]);
        }
        if ($registerModelResponse['task_id']) {
          // Monitor the state of the register model task with polling.
          $taskId = $registerModelResponse['task_id'];
          for ($i = 0; $i < self::MAX_RETRIES; $i++) {
            $taskResponse = $this->client->ml()->getTask(['id' => $taskId]);
            if (isset($taskResponse['state']) && $taskResponse['state'] === 'FAILED') {
              $this->logger->error(
                'Model registration task for model ' . $modelPath . ' with task id ' . $taskId . ' failed: ' . ($taskResponse['error'] ?? 'Unknown error')
              );
              return NULL;
            }
            if ($taskResponse['state'] === 'COMPLETED') {
              $modelId = $taskResponse['model_id'];
              break;
            }
            $this->logger->info(sprintf('Task %s is still in progress. State: %s. Retrying in %d seconds...', $taskId, $taskResponse['state'], self::RETRY_INTERVAL));
            sleep(self::RETRY_INTERVAL);
          }
          $this->logger->error(sprintf('Task %s did not complete within the allowed retries.', $taskId));
        }
        // Store the model ID in state.
        $this->logger->notice('Registered model with ID: @id', ['@id' => $modelId]);
        // Deploy the model.
        $this->deployModel($modelId);
        // Store the model ID in state.
        $this->state->set('nlp_ingestion.model_id', $modelId);
        // Register the NLP ingest pipeline.
        $this->registerNlpIngestPipeline($modelId);

        return $modelId;
      }
      else {
        $this->logger->notice('Model already registered with ID: @id', ['@id' => $this->state->get('nlp_ingestion.model_id')]);
        // Register the NLP ingest pipeline.
        $this->registerNlpIngestPipeline($this->state->get('nlp_ingestion.model_id'));
        return $this->state->get('nlp_ingestion.model_id');
      }
    }
    catch (RequestException $requestException) {
      $this->logger->error('Failed to register model: @message', ['@message' => $requestException->getMessage()]);
      return NULL;
    }

  }

  /**
   * Deploy a model in OpenSearch.
   *
   * @param string $modelId
   *   The ID of the model to deploy.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function deployModel(string $modelId = '') {
    try {
      $response = $this->client->ml()->deployModel(['id' => $modelId]);
      $this->logger->notice('Successfully deployed model with id: @id', ['@id' => $modelId]);
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error('Failed to deploy model with id (@id): @message', [
        '@id' => $modelId,
        '@message' => $requestException->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Registers NLP ingest pipelines in OpenSearch for each index.
   *
   * @param string $modelId
   *   The ID of the model to use for text embedding.
   *
   * @return bool
   *   TRUE if all pipelines are registered successfully, FALSE otherwise.
   */
  public function registerNlpIngestPipeline(string $modelId = ''): bool {
    // Fetch NLP configuration.
    $config = $this->configFactory->get('opensearch_nlp.nlp_settings');
    $storedIndexes = $config->get('indexes') ?? [];
    if (empty($storedIndexes)) {
      $this->logger->error("No indexes found in configuration. Pipelines not registered.");
      return FALSE;
    }
    $allSuccessful = TRUE;
    foreach ($storedIndexes as $indexId => $indexConfig) {
      $pipelineId = $indexConfig['ingestion_pipeline_id'];
      $pipelineExists = FALSE;
      $ingestPipelines = $this->getIngestionPipelines();
      $fieldMap = [];
      // Construct the field_map from the single mapping-embedding pair.
      if (!empty($indexConfig['mapping_embedding_pairs'])) {
        [$mappingField, $embeddingField] = explode('|', (string) $indexConfig['mapping_embedding_pairs']);
        $fieldMap[$mappingField] = $embeddingField;
      }
      if ($ingestPipelines) {
        foreach ($ingestPipelines as $id => $pipeline) {
          if ($id === $pipelineId) {
            $pipelineExists = TRUE;
            // Check if the field_map matches the existing pipeline.
            $existingFieldMap = $pipeline['processors'][0]['text_embedding']['field_map'] ?? [];
            if ($existingFieldMap !== $fieldMap) {
              $this->logger->warning(sprintf('Field map mismatch for pipeline %s. Updating pipeline.', $pipelineId));
              $pipelineExists = FALSE;
            }
            break;
          }
        }
      }
      // Check if NLP is enabled and required configurations are present.
      if (!empty($indexConfig['enable_nlp']) && !empty($indexConfig['mapping_embedding_pairs']) && !empty($indexConfig['ingestion_pipeline_id']) && !$pipelineExists && $modelId !== '') {
        // Define the pipeline.
        $pipelineDefinition = [
          'description' => 'NLP ingest pipeline for index: ' . $indexId,
          'processors' => [
            [
              'text_embedding' => [
                'model_id' => $modelId,
                'field_map' => $fieldMap,
              ],
            ],
          ],
        ];
        try {
          // Register or update the pipeline in OpenSearch.
          $this->client->ingest()->putPipeline([
            'id' => $pipelineId,
            'body' => $pipelineDefinition,
          ]);
          $this->logger->notice(sprintf('Successfully registered or updated NLP ingest pipeline: %s for index: %s', $pipelineId, $indexId));
        }
        catch (RequestException $e) {
          $this->logger->error('Failed to register or update NLP ingest pipeline (@pipeline) for index (@index): @message', [
            '@pipeline' => $pipelineId,
            '@index' => $indexId,
            '@message' => $e->getMessage(),
          ]);
          $allSuccessful = FALSE;
        }
      }
      else {
        $this->logger->warning(sprintf('Skipping index %s due to missing required configurations or matching pipeline already exists.', $indexId));
      }
    }
    return $allSuccessful;
  }

  /**
   * Fetches the list of ingest pipelines from OpenSearch.
   *
   * @return array|null
   *   An array of pipeline definitions or NULL on failure.
   */
  public function getIngestionPipelines(): ?array {
    try {
      $response = $this->client->ingest()->getPipeline();
      $this->logger->notice("Successfully fetched ingest pipelines.");
      return $response;
    }
    catch (Missing404Exception) {
      $this->logger->warning("No ingestion pipelines found.");
      return [];
    }
    catch (RequestException $requestException) {
      $this->logger->error('Failed to fetch ingest pipelines: @message', ['@message' => $requestException->getMessage()]);
      return NULL;
    }
  }

  /**
   * Deletes an NLP ingest pipeline from OpenSearch.
   *
   * @param string $pipelineId
   *   The ID of the pipeline to delete.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function deletePipeline(string $pipelineId): array|NULL {
    try {
      $response = $this->client->ingest()->deletePipeline([
        'id' => $pipelineId,
      ]);
      $this->logger->notice('Successfully deleted NLP ingest pipeline: @id', ['@id' => $pipelineId]);
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf('Failed to delete NLP ingest pipeline (%s): %s', $pipelineId, $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Undeploy a model from OpenSearch.
   *
   * @param string $modelId
   *   The ID of the model to undeploy.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function undeployModel(string $modelId): array|NULL {
    try {
      $response = $this->client->ml()->undeployModel(['id' => $modelId]);
      $this->logger->notice('Successfully undeployed model with id: @id', ['@id' => $modelId]);
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf('Failed to undeploy model with id (%s): %s', $modelId, $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Delete a model from OpenSearch.
   *
   * @param string $modelId
   *   The ID of the model to delete.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function deleteModel(string $modelId): array|NULL {
    try {
      $response = $this->client->ml()->deleteModel(['id' => $modelId]);
      $this->state->delete('nlp_ingestion.model_id');
      $this->logger->notice('Successfully deleted model with id: @id', ['@id' => $modelId]);
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf('Failed to delete model with id (%s): %s', $modelId, $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Delete a model group from OpenSearch.
   *
   * @param string $modelGroupId
   *   The ID of the model group to delete.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function deleteModelGroup(string $modelGroupId): array|NULL {
    try {
      $response = $this->client->ml()->deleteModelGroup(['id' => $modelGroupId]);
      $this->logger->notice('Successfully deleted model group with id: @id', ['@id' => $modelGroupId]);
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf('Failed to delete model group with id (%s): %s', $modelGroupId, $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Register an NLP search pipeline in OpenSearch.
   *
   * @param string $pipelineId
   *   The ID of the pipeline to register.
   */
  public function registerSearchPipeline(string $pipelineId): void {
    if ($this->isExistingSearchPipeline($pipelineId)) {
      $this->logger->notice(sprintf('Search pipeline already registered: %s', $pipelineId));
      return;
    }
    $config = $this->configFactory->get('opensearch_nlp.search_pipeline_settings');
    // Retrieve saved values.
    $normalizationTechnique = $config->get('normalization_technique') ?? 'min_max';
    $combinationTechnique = $config->get('combination_technique') ?? 'arithmetic_mean';
    $weights = array_map('floatval', explode(',', $config->get('combination_weights') ?? '0.5,0.5'));
    $description = $config->get('description') ?? 'Post processor for hybrid search';

    // Build the request body.
    $pipelineDefinition = [
      'description' => $description,
      'phase_results_processors' => [
        [
          'normalization-processor' => [
            'normalization' => [
              'technique' => $normalizationTechnique,
            ],
            'combination' => [
              'technique' => $combinationTechnique,
              'parameters' => [
                'weights' => $weights,
              ],
            ],
          ],
        ],
      ],
    ];

    try {
      $this->client->searchPipeline()->put([
        'id' => $pipelineId,
        'body' => $pipelineDefinition,
      ]);

      $this->logger->notice(sprintf('Successfully registered NLP search pipeline: %s ', $pipelineId));
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf('Failed to register NLP search pipeline (%s): %s', $pipelineId, $requestException->getMessage()));
    }
  }

  /**
   * Update the index settings in OpenSearch.
   *
   * @param string $index_id
   *   The ID of the index to update.
   * @param string $search_pipeline_id
   *   The ID of the search pipeline to use.
   * @param string $ingestion_pipeline_id
   *   The ID of the ingestion pipeline to use.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function updateIndexSettings(string $index_id, string $search_pipeline_id, string $ingestion_pipeline_id): array|NULL {
    try {
      // Close the index before updating static settings.
      $this->client->indices()->close(['index' => $index_id]);
      // Update the index settings.
      $response = $this->client->indices()->putSettings([
        'index' => $index_id,
        'body' => [
          'index' => [
            'knn' => TRUE,
            'search.default_pipeline' => $search_pipeline_id,
            'default_pipeline' => $ingestion_pipeline_id,
          ],
        ],
      ]);
      // Reopen the index after updating.
      $this->client->indices()->open(['index' => $index_id]);
      $this->logger->notice('Successfully updated index settings for index: @id', ['@id' => $index_id]);
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf('Failed to update index settings for index: %s - %s', $index_id, $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Fetches the list of search pipelines from OpenSearch.
   *
   * @param string $pipelineId
   *   The pipeline ID.
   *
   * @return array|bool
   *   An array of pipeline definitions or FALSE on failure.
   */
  public function getSearchPipeline(string $pipelineId): array|bool {
    try {
      $response = $this->client->searchPipeline()->get(
        ['id' => $pipelineId],
      );
      $this->logger->notice("Successfully fetched search pipelines");
      // Check if the given search pipeline ID exists in the response.
      if (isset($response[$pipelineId])) {
        return $response;
      }
      else {
        $this->logger->warning(sprintf("Pipeline ID '%s' not found in response.", $pipelineId));
        return FALSE;
      }
    }
    catch (Missing404Exception) {
      $this->logger->warning("No search pipelines found.");
      return FALSE;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf("Failed to fetch search pipelines: %s", $requestException->getMessage()));
      return FALSE;
    }
  }

  /**
   * Fetches the index settings from OpenSearch.
   *
   * @param string $indexId
   *   The ID of the index to fetch settings for.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function getIndexsettings(string $indexId): array|NULL {
    try {
      $response = $this->client->indices()->getSettings(['index' => $indexId]);
      $this->logger->notice("Successfully fetched index settings");
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error('Failed to fetch index settings: @message', ['@message' => $requestException->getMessage()]);
      return NULL;
    }
  }

  /**
   * Checks if a search pipeline exists in OpenSearch.
   *
   * @param string $pipelineId
   *   The pipeline ID.
   *
   * @return bool
   *   TRUE if the pipeline exists, FALSE otherwise.
   */
  public function isExistingSearchPipeline(string $pipelineId): bool {
    try {
      $response = $this->client->searchPipeline()->get(
      ['id' => $pipelineId],
      );
      $this->logger->notice("Successfully fetched search pipelines");
      // Check if the given search pipeline ID exists in the response.
      if (isset($response[$pipelineId])) {
        return TRUE;
      }
      else {
        $this->logger->warning(sprintf("Pipeline ID '%s' not found in response.", $pipelineId));
        return FALSE;
      }
    }
    catch (Missing404Exception) {
      $this->logger->warning("No search pipelines found.");
      return FALSE;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf("Failed to fetch search pipelines: %s", $requestException->getMessage()));
      return FALSE;
    }

  }

  /**
   * Get search pipelines from OpenSearch.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function getSearchPipelines(): array|NULL {
    try {
      $response = $this->client->searchPipeline()->get();
      $this->logger->notice("Successfully fetched search pipelines");
      return $response;
    }
    catch (Missing404Exception) {
      $this->logger->warning("No ingestion pipelines found.");
      return [];
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf("Failed to fetch ingest pipelines: %s", $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Get model from OpenSearch.
   *
   * @param string $modelId
   *   The ID of the model to get.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function getModel(string $modelId): array|NULL {
    try {
      $response = $this->client->ml()->getModel(
      ['id' => $modelId],
      );
      $this->logger->notice("Successfully fetched model");
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf('Failed to fetch model: %s', $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Get a modellist using model group id.
   *
   * @param string $model_group_id
   *   The model group ID.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function getModelByGroupId(string $model_group_id): array|NULL {
    try {
      $params = [
        'body' => [
          'query' => [
            'term' => [
              'model_group_id' => $model_group_id,
            ],
          ],
          'size' => 1000,
        ],
      ];
      $response = $this->client->ml()->searchModels($params);
      $this->logger->notice("Successfully fetched model");
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf('Failed to fetch model: %s', $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Create a connector in OpenSearch.
   *
   * @param array $config
   *   The configuration for the connector.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function createConnector(array $config): array|NULL {
    try {
      $response = $this->client->ml()->createConnector([
        'body' => $config,
      ]);
      $this->logger->notice('Successfully created connector: @id', ['@id' => $response['connector_id']]);
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf("Failed to create connector: %s", $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Get a list of connectors from OpenSearch.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function getConnector(string $connector_id): array|NULL {
    try {
      $response = $this->client->ml()->getConnectors(
        ['id' => $connector_id],
      );
      $this->logger->notice('Successfully fetched connector: @id', ['@id' => $connector_id]);
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf("Failed to fetch connector: %s", $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Get a list of connectors from OpenSearch.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function getAllConnectors(): array|NULL {
    try {
      $response = $this->client->ml()->getConnectors();
      $this->logger->notice("Successfully feteched connectors");
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf("Failed to fetch connectors: %s", $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Delete a connector from OpenSearch.
   *
   * @param string $connector_id
   *   The ID of the connector to delete.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function deleteConnector(string $connector_id): array|NULL {
    try {
      $response = $this->client->ml()->deleteConnector(
        ['id' => $connector_id],
      );
      $this->logger->notice('Successfully deleted connector: @id', ['@id' => $connector_id]);
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf("Failed to delete connector: %s", $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Get a list of models which are in deployed state.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function getDeployedModels(): array|NULL {
    try {
      $params = [
        'body' => [
          'query' => [
            'term' => [
              'model_state' => 'DEPLOYED',
            ],
          ],
          'size' => 1000,
        ],
      ];
      $response = $this->client->ml()->searchModels(
        $params
      );
      $this->logger->notice("Successfully fetched deployed model");
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf('Failed to fetch model: %s', $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Get a list of models groups.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function getModelGroups(): array|NULL {
    try {
      $response = $this->client->ml()->getModelGroups();
      $this->logger->notice("Successfully fetched model group");
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf('Failed to fetch model group: %s', $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Delete a search pipeline from OpenSearch.
   *
   * @param string $pipelineId
   *   The ID of the pipeline to delete.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function deleteSearchPipeline(string $pipelineId): array|NULL {
    try {
      $response = $this->client->searchPipeline()->delete(
        ['id' => $pipelineId],
      );
      $this->logger->notice('Successfully deleted search pipeline: @id', ['@id' => $pipelineId]);
      return $response;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf("Failed to delete search pipeline (%s): %s", $pipelineId, $requestException->getMessage()));
      return NULL;
    }
  }

  /**
   * Update a search pipeline in OpenSearch.
   *
   * @param string $pipelineId
   *   The ID of the pipeline to update.
   * @param array $updatedValues
   *   The updated values for the pipeline.
   *
   * @return bool
   *   TRUE if the pipeline was updated successfully, FALSE otherwise.
   */
  public function updateSearchPipeline(string $pipelineId, array $updatedValues): bool {
    if (!$this->isExistingSearchPipeline($pipelineId)) {
      $this->logger->warning(sprintf('Search pipeline does not exist: %s. Attempting to register.', $pipelineId));
    }
    // Use updated values instead of config values.
    $normalizationTechnique = $updatedValues['normalization_technique'] ?? 'min_max';
    $combinationTechnique = $updatedValues['combination_technique'] ?? 'arithmetic_mean';
    $weights = array_map('floatval', explode(',', $updatedValues['combination_weights'] ?? '0.5,0.5'));
    $description = $updatedValues['description'] ?? 'Post processor for hybrid search';
    // Build updated pipeline definition.
    $pipelineDefinition = [
      'description' => $description,
      'phase_results_processors' => [
        [
          'normalization-processor' => [
            'normalization' => [
              'technique' => $normalizationTechnique,
            ],
            'combination' => [
              'technique' => $combinationTechnique,
              'parameters' => [
                'weights' => $weights,
              ],
            ],
          ],
        ],
      ],
    ];
    try {
      $this->client->searchPipeline()->put([
        'id' => $pipelineId,
        'body' => $pipelineDefinition,
      ]);
      $this->logger->notice('Successfully updated NLP search pipeline: @id', ['@id' => $pipelineId]);
      return TRUE;
    }
    catch (RequestException $requestException) {
      $this->logger->error(sprintf("Failed to update NLP search pipeline (%s): %s", $pipelineId, $requestException->getMessage()));
      return FALSE;
    }
  }

  /**
   * Validate/predict the model from OpenSearch.
   *
   * @param string $modelId
   *   The model ID.
   * @param string $text
   *   The text to predict.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function predictModel(string $modelId, string $text): array|NULL {
    $config = $this->configFactory->get('opensearch_nlp.nlp_settings');
    $model_type = $config->get('model_type');
    $body = $this->buildPredictionBody($text, $model_type);
    try {
      $response = $this->client->ml()->predict([
        'id' => $modelId,
        'body' => $body,
      ]);
      // Log or process the response.
      $this->logger->info('Prediction response: @response', ['@response' => json_encode($response)]);
      return $response;
    }
    catch (\Exception $exception) {
      $this->logger->error('ML prediction failed: @message', ['@message' => $exception->getMessage()]);
      return NULL;
    }
  }

  /**
   * Build the prediction body based on the model type.
   *
   * @param string $text
   *   The text to predict.
   * @param string $model_type
   *   The type of model (local or external).
   *
   * @return array
   *   The body for the prediction request.
   */
  protected function buildPredictionBody(string $text, string $model_type): array {
    switch ($model_type) {
      case 'openai':
        $messagesJson = json_encode([$text]);
        $body = [
          'parameters' => [
            'input' => $messagesJson,
          ],
        ];
        break;

      case 'rag':
        $messages = [
          [
            'role' => 'system',
            'content' => 'You are a helpful assistant.',
          ],
          [
            'role' => 'user',
            'content' => $text,
          ],
        ];
        $messagesJson = json_encode($messages);
        $body = [
          'parameters' => [
            'messages' => $messagesJson,
          ],
        ];
        break;

      default:
        $body = [
          'text_docs' => [$text],
          'return_number' => TRUE,
          'target_response' => ['sentence_embedding'],
        ];
        break;
    }
    return $body;
  }

  /**
   * Test the ingestion pipeline in OpenSearch.
   *
   * @param string $pipeline_name
   *   The pipeline name.
   * @param string $index
   *   The index name.
   * @param string $text
   *   The text to test.
   * @param string $mapping_field
   *   The mapping field.
   *
   * @return array|null
   *   The response from the OpenSearch server.
   */
  public function testIngestionPipeline(string $pipeline_name, string $index, string $text, string $mapping_field): array|NULL {
    try {
      $response = $this->client->ingest()->simulate([
        'id' => $pipeline_name,
        'body' => [
          'docs' => [
            [
              '_index' => $index,
              '_source' => [
                $mapping_field => $text,
              ],
            ],
          ],
        ],
      ]);
      // Log or process the response.
      $this->logger->info('Test response: @response', ['@response' => json_encode($response)]);
      return $response;
    }
    catch (\Exception $exception) {
      // Log error if any exception occurs.
      $this->logger->error('ML Test failed: @message', ['@message' => $exception->getMessage()]);
      return NULL;
    }
  }

  /**
   * Validate NLP search parameters and NLP is enabled with deployed models.
   *
   * @param string $query
   *   The search query to validate.
   *
   * @return bool
   *   True if NLP is enabled and models exist, False otherwise.
   *
   * @throws \Exception
   */
  public function canPerformNlpSearch(string $query): bool {
    // Validate query.
    if (empty($query)) {
      throw new \Exception("Invalid search parameters: query is required.");
    }

    // Load NLP settings.
    $nlp_settings = $this->configFactory->get('opensearch_nlp.nlp_settings');
    $enable_nlp = $nlp_settings->get('enable_nlp') ?? FALSE;

    // If NLP is disabled, return FALSE.
    if (!$enable_nlp) {
      return FALSE;
    }

    // Check if deployed NLP model exists.
    $deployed_model = $this->getDeployedModels();
    return !empty($deployed_model);
  }

  /**
   * Check semantic cache for a query and index.
   *
   * @return bool
   *   Returns true if semantic cache is enabled, otherwise false.
   */
  public function checkSemanticCache(): bool {
    $semantic_cache_settings = $this->configFactory->get('opensearch_nlp.semantic_cache_settings');
    $semantic_cache_enabled = $semantic_cache_settings->get('semantic_cache_enabled') ?? FALSE;
    return (bool) $semantic_cache_enabled;
  }

  /**
   * Delete all pending server tasks for the index.
   *
   * @param \Drupal\search_api\IndexInterface|null $index
   *   The index entity.
   */
  public function deletePendingServerTasks($index = NULL): void {
    $this->taskManager->delete($this->server, $index);
  }

  /**
   * Ensure required ML Commons cluster settings exist.
   *
   * @return bool
   *   TRUE if settings were ensured/updated, FALSE on error.
   */
  public function ensureMlCommonsSettings(): bool {
    $requiredSettings = [
      'plugins.ml_commons.only_run_on_ml_node' => FALSE,
      'plugins.ml_commons.native_memory_threshold' => 100,
      'plugins.ml_commons.agent_framework_enabled' => TRUE,
    ];
    try {
      $client = $this->getClient();
      // Get current cluster settings (include defaults as fallback).
      $current = $client->cluster()->getSettings([
        'include_defaults' => TRUE,
      ]);
      $toUpdate = [];
      foreach ($requiredSettings as $key => $expected) {
        $currentVal = $current['persistent'][$key] ?? $current['defaults'][$key] ?? NULL;
        if ($currentVal === NULL || $currentVal !== $expected) {
          $toUpdate[$key] = $expected;
        }
      }
      if (!empty($toUpdate)) {
        $client->cluster()->putSettings([
          'body' => [
            'persistent' => $toUpdate,
          ],
        ]);
        $this->logger->notice('Updated ML Commons cluster settings: @keys', [
          '@keys' => implode(', ', array_keys($toUpdate)),
        ]);
      }
      return TRUE;
    }
    catch (\Exception $exception) {
      $this->logger->error('Failed ensuring ML Commons settings: @msg', [
        '@msg' => $exception->getMessage(),
      ]);
      return FALSE;
    }
  }

}
