<?php

declare(strict_types=1);

namespace Drupal\opensearch_nlp\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\search_api\Entity\Server;
use Drupal\opensearch_nlp\Cache\SemanticCache;
use OpenSearch\Client;

/**
 * Provides NLP search functionality using OpenSearch.
 */
class NLPSearch {

  /**
   * The OpenSearch client.
   *
   * @var \OpenSearch\Client
   */
  protected $client;

  /**
   * NLPSearch constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config
   *   The configuration factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger
   *   Logger interface.
   * @param \Drupal\opensearch_nlp\Service\NLPIngestionService $nlpService
   *   The NLP ingestion service.
   * @param \Drupal\opensearch_nlp\Cache\SemanticCache $semanticCache
   *   The semantic cache service.
   */
  public function __construct(protected ConfigFactoryInterface $config, protected LoggerChannelFactoryInterface $logger, protected NLPIngestionService $nlpService, protected SemanticCache $semanticCache) {
    $this->client = $this->getClient();
  }

  /**
   * Search for documents.
   *
   * @param string $index_name
   *   The index name.
   * @param string $query
   *   The search query.
   * @param int $from
   *   The starting point for the search results.
   * @param int $size
   *   The number of results to return.
   * @param array $aggs
   *   Aggregations to apply to the search.
   * @param array $sort
   *   Sorting options for the search results.
   * @param array $body
   *   Additional body parameters for the search query.
   *
   * @return array
   *   The JSON response.
   *
   * @throws \Exception
   */
  public function search($index_name, string $query, $from = 0, $size = 10, $aggs = [], $sort = [], $body = []): array {
    try {
      // Validate query & NLP readiness.
      $is_nlp_enabled = $this->nlpService->canPerformNlpSearch($query);
      $index_names = (array) $index_name;

      // If NLP disabled or models not available, fallback to keyword search.
      if (!$is_nlp_enabled) {
        return $this->performSearch($this->keywordSearch($body), $index_names, NULL, $from, $size, $aggs, $sort, 'keyword');
      }

      // Check semantic cache first.
      $semantic_cache_enabled = $this->nlpService->checkSemanticCache();
      if ($semantic_cache_enabled) {
        $cached_results = $this->semanticCache->retrieveCachedResponse($query, $index_name, $body);
        if ($cached_results) {
          $this->logger->get('opensearch_nlp')->info('Cache hit for query: @query', ['@query' => $query]);
          return $cached_results;
        }
      }

      // Default to keyword search (fallback).
      $nlp_settings = $this->config->get('opensearch_nlp.nlp_settings');
      $indexes_config = $nlp_settings->get('indexes') ?? [];
      $search_type = 'keyword';
      $search_query = $this->keywordSearch($body);
      $search_pipeline_id = NULL;
      $used_fallback = FALSE;

      // semantic/hybrid logic in a separate try block.
      try {
        $deployed_model = $this->nlpService->getDeployedModels();
        $model_id = $deployed_model['hits']['hits'][0]['_id'];
        $k = (int) ($nlp_settings->get('nearest_neighbors') ?? 10);

        if (count($index_names) > 1) {
          // Multi-index search logic.
          $search_type = $nlp_settings->get('search_type') ?? 'keyword';
          $universal_mapping_field = $nlp_settings->get('universal_mapping_field') ?? '';
          $universal_embedd_vector_field = $nlp_settings->get('universal_embedd_vector_field') ?? '';
          $pagination_depth = (int) $nlp_settings->get('pagination_depth') ?? 10;
          $search_pipeline_id = 'multi_index_hybrid_search_pipeline';

          $search_query = match ($search_type) {
            'semantic' => $this->semanticSearch($query, $universal_embedd_vector_field, $model_id, body: $body, k: $k, multiindex: TRUE),
            'hybrid' => $this->hybridMultiIndexQuery($query, $universal_mapping_field, $universal_embedd_vector_field, $model_id, $k, $pagination_depth),
            'script_score' => $this->scriptScoreSearch($universal_embedd_vector_field, $model_id, $query, $k),
            default => $this->keywordSearch($body),
          };
        }
        else {
          // Single index search logic.
          $index_config = $indexes_config[$index_names[0]] ?? NULL;
          $search_type = $index_config['search_type'] ?? 'keyword';
          $pagination_depth = (int) $index_config['pagination_depth'] ?? 10;
          $mapping_embedding_pair = $index_config['mapping_embedding_pairs'] ?? '';
          [, $embedding_field] = explode('|', (string) $mapping_embedding_pair);
          $search_pipeline_id = $index_config['search_pipeline_id'] ?? NULL;
          $k = (int) ($index_config['nearest_neighbors'] ?? 10);

          $search_query = match ($search_type) {
            'semantic' => $this->semanticSearch($query, $embedding_field, $model_id, body: $body, k: $k),
            'hybrid' => $this->hybridSearch($query, $mapping_embedding_pair, $model_id, $k, $pagination_depth),
            'script_score' => $this->scriptScoreSearch($embedding_field, $model_id, $query, $k),
            default => $this->keywordSearch($body),
          };
        }
      }
      catch (\Exception $e) {
        // Fallback to keyword search on failure.
        $this->logger->get('opensearch_nlp')->warning('Semantic/Hybrid search failed, falling back to keyword: @message', [
          '@message' => $e->getMessage(),
        ]);
        $search_query = $this->keywordSearch($body);
        $search_type = 'keyword';
        $search_pipeline_id = NULL;
        $used_fallback = TRUE;
      }

      // Execute search.
      $response = $this->performSearch($search_query, $index_names, $search_pipeline_id, $from, $size, $aggs, $sort, $search_type);

      // Store in cache if enabled.
      if ($semantic_cache_enabled) {
        $this->semanticCache->storeResponse($query, $response, $index_name, $search_type, $body);
      }

      // Log result type.
      if ($used_fallback) {
        $this->logger->get('opensearch_nlp')->notice('Search for query "@query" completed using fallback keyword search.', [
          '@query' => $query,
        ]);
      }
      else {
        $this->logger->get('opensearch_nlp')->notice('Search for query "@query" completed using @type search.', [
          '@query' => $query,
          '@type' => $search_type,
        ]);
      }

      return $response;
    }
    catch (\Exception $exception) {
      $this->logger->get('opensearch_nlp')->error('Search error: @message', ['@message' => $exception->getMessage()]);
      return [];
    }
  }

  /**
   * Perform a keyword search.
   *
   * @param array $body
   *   The body of the search query.
   *
   * @return array
   *   The search query.
   */
  private function keywordSearch(array $body): array {
    return $body;
  }

  /**
   * Perform a semantic search using neural queries.
   *
   * @param string $query
   *   The search term.
   * @param string $embedding_field
   *   The combined embedding_field pair.
   * @param string $model_id
   *   The model ID.
   * @param array $body
   *   The body of the search query.
   * @param int $k
   *   Number of results.
   * @param bool $multiindex
   *   Whether to perform a multi-index search.
   *
   * @return array
   *   The search query.
   */
  private function semanticSearch(string $query, string $embedding_field, string $model_id, array $body, int $k = 5, bool $multiindex = FALSE): array {
    // Ensure bool and should clauses exist before building the main query.
    if ($multiindex) {
      $body['query']['bool']['should'][] = [
        'neural' => [
          $embedding_field => [
            'query_text' => $query,
            'model_id' => $model_id,
            'k' => $k,
          ],
        ],
      ];
      return $body;
    }
    if (!isset($body['query']['bool'])) {
      $body['query']['bool'] = [];
    }
    // Add the neural query to the should clause.
    if (!isset($body['query']['bool']['should'])) {
      $body['query']['bool']['should'] = [];
    }
    $body['query']['bool']['must'][] = [
      'neural' => [
        $embedding_field => [
          'query_text' => $query,
          'model_id' => $model_id,
          'k' => $k,
        ],
      ],
    ];

    return $body;
  }

  /**
   * Perform a script_score-based search using KNN.
   *
   * @param string $vector_field
   *   The vector field to search against.
   * @param string $model_id
   *   The model id.
   * @param string $query
   *   The query text.
   * @param int $k
   *   The number of results to return.
   * @param string $space_type
   *   The similarity metric (e.g., "cosinesimil").
   *
   * @return array
   *   The search results.
   */
  public function scriptScoreSearch(string $vector_field, string $model_id, string $query, int $k = 4, string $space_type = 'cosinesimil'): array {
    try {
      // Validate the model and extract the embedding data for the query.
      $query_embedding_response = $this->nlpService->predictModel($model_id, $query);
      $query_embedding = $this->extractEmbedding($query_embedding_response);
      if (!$query_embedding) {
        throw new \Exception('Failed to extract embedding for the query.');
      }

      // Build the search query.
      return [
        'query' => [
          'script_score' => [
            'query' => [
              'match_all' => (object) [],
            ],
            'script' => [
              'source' => 'knn_score',
              'lang' => 'knn',
              'params' => [
                'field' => $vector_field,
                'query_value' => $query_embedding,
                'space_type' => $space_type,
              ],
            ],
          ],
        ],
      ];
    }
    catch (\Exception $exception) {
      $this->logger->get('opensearch_nlp')->error('Script score search error: @message', ['@message' => $exception->getMessage()]);
      return [];
    }
  }

  /**
   * Extracts the embedding vector from the model validation response.
   *
   * @param array $response
   *   The response from the predictModel method.
   *
   * @return array|null
   *   The embedding vector, or NULL if extraction fails.
   */
  private function extractEmbedding(array $response) {
    if (isset($response['inference_results'][0]['output'][0]['data'])) {
      return $response['inference_results'][0]['output'][0]['data'];
    }
    return NULL;
  }

  /**
   * Perform a hybrid search combining keyword and semantic search.
   *
   * @param string $query
   *   The search query.
   * @param string $mapping_embedding_pair
   *   The combined mapping-embedding pair.
   * @param string $model_id
   *   The model ID.
   * @param int $k
   *   Number of results.
   * @param int $pagination_depth
   *   The pagination depth for the search.
   *
   * @return array
   *   The search results.
   */
  private function hybridSearch(string $query, string $mapping_embedding_pair, string $model_id, int $k = 5, int $pagination_depth = 10): array {
    [$mapping_field, $embedding_field] = explode('|', $mapping_embedding_pair);
    return [
      '_source' => ['excludes' => [$embedding_field]],
      'query' => [
        'hybrid' => [
          'pagination_depth' => $pagination_depth,
          'queries' => [
            [
              'match' => [
                $mapping_field => ['query' => $query],
              ],
            ],
            [
              'neural' => [
                $embedding_field => [
                  'query_text' => $query,
                  'model_id' => $model_id,
                  'k' => $k,
                ],
              ],
            ],
          ],
        ],
      ],
    ];
  }

  /**
   * Perform a hybrid search with multi-index.
   *
   * @param string $query
   *   The search query.
   * @param string $mapping_field
   *   The mapping field.
   * @param string $embedding_vector_field
   *   The embedding vector field.
   * @param string $model_id
   *   The model ID.
   * @param int $k
   *   Number of results.
   * @param int $pagination_depth
   *   The pagination depth for the search.
   *
   * @return array
   *   The search results.
   */
  public function hybridMultiIndexQuery(string $query, string $mapping_field, string $embedding_vector_field, string $model_id, int $k = 10, int $pagination_depth = 10): array {
    try {
      $pipelineId = 'multi_index_hybrid_search_pipeline';
      if ($this->nlpService->isExistingSearchPipeline($pipelineId)) {
        return [
          '_source' => [
            'exclude' => [$embedding_vector_field],
          ],
          'query' => [
            'hybrid' => [
              'pagination_depth' => $pagination_depth,
              'queries' => [
                [
                  'match' => [
                    $mapping_field => ['query' => $query],
                  ],
                ],
                [
                  'neural' => [
                    $embedding_vector_field => [
                      'query_text' => $query,
                      'model_id' => $model_id,
                      'k' => $k,
                    ],
                  ],
                ],
              ],
            ],
          ],
        ];
      }
      else {
        throw new \Exception("Hybrid multi-index search pipeline not found.");
      }
    }
    catch (\Exception $exception) {
      $this->logger->get('opensearch_nlp')->error('Hybrid multi-index search error: @message', ['@message' => $exception->getMessage()]);
      return [];
    }
  }

  /**
   * Performs the actual OpenSearch query.
   *
   * @param array $search_query
   *   The query payload.
   * @param array $index_name
   *   The index name.
   * @param string|null $pipeline_id
   *   The search pipeline ID, if applicable.
   * @param int $from
   *   The starting point for the search results.
   * @param int $size
   *   The number of results to return.
   * @param array $aggs
   *   Aggregations to apply to the search.
   * @param array $sort
   *   Sorting options for the search results.
   * @param string $search_type
   *   The type of search to perform (e.g., 'keyword', 'semantic', etc.).
   *
   * @return array
   *   The JSON response.
   */
  private function performSearch(array $search_query, array $index_name, $pipeline_id, $from = 0, $size = 10, $aggs = [], $sort = [], $search_type = 'keyword'): array {
    try {
      if ($search_type === 'keyword') {
        $searchParams = ['index' => $index_name, 'body' => $search_query];
        $response = $this->client->search($searchParams);
        unset($response['took'], $response['timed_out'], $response['_shards'], $response['hits']['max_score']);
        return $response;
      }
      $nlp_settings = $this->config->get('opensearch_nlp.nlp_settings');
      $indexes_config = $nlp_settings->get('indexes') ?? [];
      $all_exclude_fields = [];
      foreach ($index_name as $idx) {
        if (isset($indexes_config[$idx]['exclude_from_search_results'])) {
          $exclude_fields = $indexes_config[$idx]['exclude_from_search_results'];
          $all_exclude_fields = array_merge($all_exclude_fields, $exclude_fields);
        }
      }

      $search_query['_source'] = [
        'exclude' => $all_exclude_fields,
      ];
      $search_query['from'] = $from;
      $search_query['size'] = $size;

      if (!empty($aggs)) {
        $search_query['aggs'] = $aggs;
      }

      if (!empty($sort)) {
        $search_query['sort'] = $sort;
      }

      return $this->client->search([
        'index' => $index_name,
        'body' => $search_query,
        'search_pipeline' => $pipeline_id,
      ]);
    }
    catch (\Exception $exception) {
      $this->logger->get('opensearch_nlp')->error('Search error: @message', ['@message' => $exception->getMessage()]);
      return [];
    }
  }

  /**
   * Get the OpenSearch client.
   *
   * @return \OpenSearch\Client
   *   The OpenSearch client.
   */
  public function getClient(): Client {
    if (!$this->client) {
      $server = Server::load('opensearch');
      if ($server) {
        $backend = $server->getBackend();
        // @phpstan-ignore-next-line
        $this->client = $backend->getClient();
      }
      else {
        throw new \Exception('OpenSearch server not found.');
      }
    }
    return $this->client;
  }

}
