<?php

declare(strict_types=1);

namespace Drupal\Tests\opensearch_nlp\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests the NLP configuration forms.
 *
 * @group opensearch_nlp
 */
class NLPConfigFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['opensearch_nlp', 'search_api'];

  /**
   * A user with permission to administer search.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'administer search_api',
      'administer site configuration',
    ]);
  }

  /**
   * Tests the NLP configuration form can be accessed and submitted.
   */
  public function testNlpConfigFormAccess(): void {
    // Anonymous users cannot access the form.
    $this->drupalGet('/admin/config/search/opensearch-nlp');
    $this->assertSession()->statusCodeEquals(403);

    // Admin users can access the form.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/search/opensearch-nlp');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Enable/Disable NLP processing');
  }

  /**
   * Tests submitting the NLP configuration form.
   */
  public function testNlpConfigFormSubmit(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/search/opensearch-nlp');

    $edit = [
      'enable_nlp' => TRUE,
      'model_type' => 'openai',
      'model_path' => 'test-model-path',
      'model_description' => 'Test model description',
      'model_group' => 'test-group',
      'model_group_description' => 'Test group description',
      'search_type' => 'semantic',
      'nearest_neighbors' => '50',
    ];

    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify the configuration was saved.
    $config = $this->config('opensearch_nlp.nlp_settings');
    $this->assertTrue($config->get('enable_nlp'));
    $this->assertSame('openai', $config->get('model_type'));
    $this->assertSame('test-model-path', $config->get('model_path'));
    $this->assertSame('semantic', $config->get('search_type'));
    $this->assertSame('50', $config->get('nearest_neighbors'));
  }

  /**
   * Tests the vector configuration form.
   */
  public function testVectorConfigForm(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/search/opensearch-vector-nlp-config-form');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Vector Dimension');

    $edit = [
      'dimension' => '1536',
      'method' => 'hnsw',
      'space_type' => 'cosinesimil',
      'engine' => 'faiss',
    ];

    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify the configuration was saved.
    $config = $this->config('opensearch_nlp.vector_settings');
    $this->assertSame('1536', $config->get('dimension'));
    $this->assertSame('cosinesimil', $config->get('space_type'));
    $this->assertSame('faiss', $config->get('engine'));
  }

  /**
   * Tests the search pipeline configuration form.
   */
  public function testSearchPipelineConfigForm(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/search/opensearch-search-pipeline-config-form');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Normalization Technique');

    $edit = [
      'enable_multi_index_hybrid_search' => TRUE,
      'normalization_technique' => 'l2',
      'combination_technique' => 'geometric_mean',
      'combination_weights' => '0.7,0.3',
      'tag' => 'test-pipeline',
      'description' => 'Test pipeline description',
    ];

    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify the configuration was saved.
    $config = $this->config('opensearch_nlp.search_pipeline_settings');
    $this->assertTrue($config->get('enable_multi_index_hybrid_search'));
    $this->assertSame('l2', $config->get('normalization_technique'));
    $this->assertSame('geometric_mean', $config->get('combination_technique'));
    $this->assertSame('0.7,0.3', $config->get('combination_weights'));
  }

  /**
   * Tests the semantic cache settings form.
   */
  public function testSemanticCacheSettingsForm(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/semantic-cache/settings');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Enable Semantic Cache');

    $edit = [
      'semantic_cache_enabled' => TRUE,
    ];

    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify the configuration was saved.
    $config = $this->config('opensearch_nlp.semantic_cache_settings');
    $this->assertTrue($config->get('semantic_cache_enabled'));
  }

  /**
   * Tests form validation for model fields.
   */
  public function testNlpConfigFormValidation(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/search/opensearch-nlp');

    // Submit form with missing required fields.
    $edit = [
      'enable_nlp' => TRUE,
      'model_type' => 'local',
      // Missing model_path, model_description, model_group.
    ];

    $this->submitForm($edit, 'Save configuration');

    // Check for validation errors on required fields.
    $this->assertSession()->pageTextContains('field is required');
  }

  /**
   * Tests that default configuration values are correct.
   */
  public function testDefaultConfigurationValues(): void {
    // Check nlp_settings defaults.
    $nlp_config = $this->config('opensearch_nlp.nlp_settings');
    $this->assertFalse($nlp_config->get('enable_nlp'));
    $this->assertSame('local', $nlp_config->get('model_type'));
    $this->assertSame('TORCH_SCRIPT', $nlp_config->get('model_format'));
    $this->assertSame('keyword', $nlp_config->get('search_type'));
    $this->assertSame('10', $nlp_config->get('pagination_depth'));

    // Check vector_settings defaults.
    $vector_config = $this->config('opensearch_nlp.vector_settings');
    $this->assertSame('768', $vector_config->get('dimension'));
    $this->assertSame('hnsw', $vector_config->get('method'));
    $this->assertSame('l2', $vector_config->get('space_type'));
    $this->assertSame('lucene', $vector_config->get('engine'));

    // Check search_pipeline_settings defaults.
    $pipeline_config = $this->config('opensearch_nlp.search_pipeline_settings');
    $this->assertFalse($pipeline_config->get('enable_multi_index_hybrid_search'));
    $this->assertSame('min_max', $pipeline_config->get('normalization_technique'));
    $this->assertSame('arithmetic_mean', $pipeline_config->get('combination_technique'));
    $this->assertSame('0.5,0.5', $pipeline_config->get('combination_weights'));

    // Check semantic_cache_settings defaults.
    $cache_config = $this->config('opensearch_nlp.semantic_cache_settings');
    $this->assertFalse($cache_config->get('semantic_cache_enabled'));
  }

}
