<?php

declare(strict_types=1);

namespace Drupal\Tests\opensearch_nlp\Kernel;

use Drupal\KernelTests\KernelTestBase;

/**
 * Tests configuration schema for opensearch_nlp module.
 *
 * @group opensearch_nlp
 */
class ConfigSchemaTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'search_api',
    'search_api_opensearch',
    'opensearch_nlp',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['opensearch_nlp']);
  }

  /**
   * Tests opensearch_nlp.nlp_settings configuration schema.
   */
  public function testNlpSettingsSchema(): void {
    $config = $this->config('opensearch_nlp.nlp_settings');

    // Test default values.
    $this->assertFalse($config->get('enable_nlp'));
    $this->assertFalse($config->get('is_externally_hosted_model'));
    $this->assertSame('', $config->get('connector_id'));
    $this->assertSame('local', $config->get('model_type'));
    $this->assertSame('TORCH_SCRIPT', $config->get('model_format'));
    $this->assertSame('keyword', $config->get('search_type'));
    $this->assertSame('10', $config->get('pagination_depth'));
    $this->assertSame('10', $config->get('nearest_neighbors'));

    // Test setting and getting values.
    $config->set('enable_nlp', TRUE)
      ->set('model_type', 'openai')
      ->set('model_path', 'test/model')
      ->set('search_type', 'semantic')
      ->save();

    $this->assertTrue($config->get('enable_nlp'));
    $this->assertSame('openai', $config->get('model_type'));
    $this->assertSame('test/model', $config->get('model_path'));
    $this->assertSame('semantic', $config->get('search_type'));
  }

  /**
   * Tests opensearch_nlp.vector_settings configuration schema.
   */
  public function testVectorSettingsSchema(): void {
    $config = $this->config('opensearch_nlp.vector_settings');

    // Test default values.
    $this->assertSame('768', $config->get('dimension'));
    $this->assertSame('hnsw', $config->get('method'));
    $this->assertSame('l2', $config->get('space_type'));
    $this->assertSame('lucene', $config->get('engine'));

    // Test setting and getting values.
    $config->set('dimension', '1536')
      ->set('space_type', 'cosinesimil')
      ->set('engine', 'faiss')
      ->save();

    $this->assertSame('1536', $config->get('dimension'));
    $this->assertSame('cosinesimil', $config->get('space_type'));
    $this->assertSame('faiss', $config->get('engine'));
  }

  /**
   * Tests opensearch_nlp.search_pipeline_settings configuration schema.
   */
  public function testSearchPipelineSettingsSchema(): void {
    $config = $this->config('opensearch_nlp.search_pipeline_settings');

    // Test default values.
    $this->assertFalse($config->get('enable_multi_index_hybrid_search'));
    $this->assertSame('min_max', $config->get('normalization_technique'));
    $this->assertSame('arithmetic_mean', $config->get('combination_technique'));
    $this->assertSame('0.5,0.5', $config->get('combination_weights'));
    $this->assertSame('search-pipeline', $config->get('tag'));

    // Test setting and getting values.
    $config->set('enable_multi_index_hybrid_search', TRUE)
      ->set('normalization_technique', 'l2')
      ->set('combination_technique', 'geometric_mean')
      ->set('combination_weights', '0.7,0.3')
      ->save();

    $this->assertTrue($config->get('enable_multi_index_hybrid_search'));
    $this->assertSame('l2', $config->get('normalization_technique'));
    $this->assertSame('geometric_mean', $config->get('combination_technique'));
    $this->assertSame('0.7,0.3', $config->get('combination_weights'));
  }

  /**
   * Tests opensearch_nlp.semantic_cache_settings configuration schema.
   */
  public function testSemanticCacheSettingsSchema(): void {
    $config = $this->config('opensearch_nlp.semantic_cache_settings');

    // Test default value.
    $this->assertFalse($config->get('semantic_cache_enabled'));

    // Test setting and getting value.
    $config->set('semantic_cache_enabled', TRUE)->save();
    $this->assertTrue($config->get('semantic_cache_enabled'));
  }

  /**
   * Tests all configuration schemas are valid.
   */
  public function testConfigSchemasAreValid(): void {
    $typed_config = \Drupal::service('config.typed');

    $schemas = [
      'opensearch_nlp.nlp_settings',
      'opensearch_nlp.vector_settings',
      'opensearch_nlp.search_pipeline_settings',
      'opensearch_nlp.semantic_cache_settings',
    ];

    foreach ($schemas as $schema_name) {
      $definition = $typed_config->getDefinition($schema_name);
      $this->assertNotEmpty($definition, sprintf('Schema %s should be defined', $schema_name));
      $this->assertArrayHasKey('type', $definition, sprintf('Schema %s should have a type key', $schema_name));
    }
  }

  /**
   * Tests indexes configuration in nlp_settings.
   */
  public function testIndexesConfiguration(): void {
    $config = $this->config('opensearch_nlp.nlp_settings');

    $indexes = [
      'test_index' => [
        'enable_nlp' => '1',
        'mapping_embedding_pairs' => 'field1|field2',
        'ingestion_pipeline_id' => 'test-ingestion',
        'search_pipeline_id' => 'test-search',
        'search_type' => 'semantic',
        'pagination_depth' => '20',
        'nearest_neighbors' => '50',
        'exclude_from_search_results' => ['rendered_item'],
      ],
    ];

    $config->set('indexes', $indexes)->save();

    $saved_indexes = $config->get('indexes');
    $this->assertIsArray($saved_indexes);
    $this->assertArrayHasKey('test_index', $saved_indexes);
    $this->assertSame('1', $saved_indexes['test_index']['enable_nlp']);
    $this->assertSame('field1|field2', $saved_indexes['test_index']['mapping_embedding_pairs']);
    $this->assertSame('test-ingestion', $saved_indexes['test_index']['ingestion_pipeline_id']);
    $this->assertSame('semantic', $saved_indexes['test_index']['search_type']);
    $this->assertSame(['rendered_item'], $saved_indexes['test_index']['exclude_from_search_results']);
  }

}
