<?php

declare(strict_types=1);

namespace Drupal\Tests\opensearch_nlp\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Core\Site\Settings;

/**
 * Tests environment-specific configuration overrides via settings.php.
 *
 * @group opensearch_nlp
 */
class EnvironmentOverrideTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'search_api',
    'search_api_opensearch',
    'opensearch_nlp',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['opensearch_nlp']);
  }

  /**
   * Tests that default config values are loaded without overrides.
   */
  public function testDefaultConfigurationWithoutOverrides(): void {
    $config = $this->config('opensearch_nlp.nlp_settings');

    // Verify default values from config/install
    $this->assertFalse($config->get('enable_nlp'), 'NLP should be disabled by default');
    $this->assertFalse($config->get('is_externally_hosted_model'), 'External model should be disabled by default');
    $this->assertSame('', $config->get('connector_id'), 'Connector ID should be empty by default');
    $this->assertSame('local', $config->get('model_type'), 'Model type should be local by default');
    $this->assertSame(
      'huggingface/sentence-transformers/msmarco-distilbert-base-tas-b',
      $config->get('model_path'),
      'Default model path should be set'
    );
    $this->assertSame('1.0.2', $config->get('model_version'), 'Default model version should be 1.0.2');
    $this->assertSame('TORCH_SCRIPT', $config->get('model_format'), 'Default format should be TORCH_SCRIPT');
    $this->assertSame('nlp_model_group', $config->get('model_group'), 'Default model group should be set');
  }

  /**
   * Tests that settings.php overrides take precedence over config.
   */
  public function testSettingsPhpOverridesConfig(): void {
    // Simulate settings.php overrides (production environment)
    $settings = Settings::getAll();
    $settings['opensearch_nlp.enable_nlp'] = TRUE;
    $settings['opensearch_nlp.model_path'] = 'huggingface/sentence-transformers/all-mpnet-base-v2';
    $settings['opensearch_nlp.model_version'] = '2.0.0';
    $settings['opensearch_nlp.model_type'] = 'openai';
    $settings['opensearch_nlp.model_group'] = 'prod_nlp_models';
    $settings['opensearch_nlp.model_description'] = 'Production model for semantic search';

    new Settings($settings);

    // Verify that Settings::get() returns our overrides
    $this->assertTrue(Settings::get('opensearch_nlp.enable_nlp'), 'Enable NLP override should be TRUE');
    $this->assertSame(
      'huggingface/sentence-transformers/all-mpnet-base-v2',
      Settings::get('opensearch_nlp.model_path'),
      'Model path override should be set'
    );
    $this->assertSame('2.0.0', Settings::get('opensearch_nlp.model_version'), 'Model version override should be 2.0.0');
    $this->assertSame('openai', Settings::get('opensearch_nlp.model_type'), 'Model type override should be openai');
    $this->assertSame('prod_nlp_models', Settings::get('opensearch_nlp.model_group'), 'Model group override should be set');
  }

  /**
   * Tests environment-specific configuration (DEV vs STAGE vs PROD).
   */
  public function testMultiEnvironmentConfiguration(): void {
    $config = $this->config('opensearch_nlp.nlp_settings');

    // Scenario 1: DEV environment (smaller, faster model)
    $dev_settings = Settings::getAll();
    $dev_settings['opensearch_nlp.model_path'] = 'huggingface/sentence-transformers/all-MiniLM-L6-v2';
    $dev_settings['opensearch_nlp.model_version'] = '1.0.0';
    $dev_settings['opensearch_nlp.model_group'] = 'dev_nlp_models';
    $dev_settings['opensearch_nlp.model_description'] = 'Development model - fast and lightweight';

    new Settings($dev_settings);

    $this->assertSame(
      'huggingface/sentence-transformers/all-MiniLM-L6-v2',
      Settings::get('opensearch_nlp.model_path'),
      'DEV environment should use lightweight model'
    );
    $this->assertSame('dev_nlp_models', Settings::get('opensearch_nlp.model_group'), 'DEV should have dev model group');

    // Scenario 2: STAGE environment (same as production model)
    $stage_settings = Settings::getAll();
    $stage_settings['opensearch_nlp.model_path'] = 'huggingface/sentence-transformers/all-mpnet-base-v2';
    $stage_settings['opensearch_nlp.model_version'] = '1.5.0';
    $stage_settings['opensearch_nlp.model_group'] = 'stage_nlp_models';
    $stage_settings['opensearch_nlp.model_description'] = 'Staging model - production candidate';

    new Settings($stage_settings);

    $this->assertSame(
      'huggingface/sentence-transformers/all-mpnet-base-v2',
      Settings::get('opensearch_nlp.model_path'),
      'STAGE environment should use production-like model'
    );
    $this->assertSame('stage_nlp_models', Settings::get('opensearch_nlp.model_group'), 'STAGE should have stage model group');

    // Scenario 3: PROD environment (high-performance model)
    $prod_settings = Settings::getAll();
    $prod_settings['opensearch_nlp.model_path'] = 'huggingface/sentence-transformers/all-mpnet-base-v2';
    $prod_settings['opensearch_nlp.model_version'] = '2.0.0';
    $prod_settings['opensearch_nlp.model_group'] = 'prod_nlp_models';
    $prod_settings['opensearch_nlp.model_description'] = 'Production model - high accuracy';
    $prod_settings['opensearch_nlp.enable_nlp'] = TRUE;

    new Settings($prod_settings);

    $this->assertSame(
      'huggingface/sentence-transformers/all-mpnet-base-v2',
      Settings::get('opensearch_nlp.model_path'),
      'PROD environment should use high-performance model'
    );
    $this->assertSame('prod_nlp_models', Settings::get('opensearch_nlp.model_group'), 'PROD should have prod model group');
    $this->assertTrue(Settings::get('opensearch_nlp.enable_nlp'), 'NLP should be enabled in PROD');
  }

  /**
   * Tests external model configuration override.
   */
  public function testExternalModelOverride(): void {
    // Simulate external model configuration (e.g., OpenAI in production)
    $settings = Settings::getAll();
    $settings['opensearch_nlp.is_externally_hosted_model'] = TRUE;
    $settings['opensearch_nlp.model_type'] = 'openai';
    $settings['opensearch_nlp.connector_id'] = 'openai-connector-123';
    $settings['opensearch_nlp.model_path'] = 'gpt-3.5-turbo';

    new Settings($settings);

    $this->assertTrue(
      Settings::get('opensearch_nlp.is_externally_hosted_model'),
      'External model should be enabled'
    );
    $this->assertSame('openai', Settings::get('opensearch_nlp.model_type'), 'Model type should be openai');
    $this->assertSame(
      'openai-connector-123',
      Settings::get('opensearch_nlp.connector_id'),
      'Connector ID should be set'
    );
    $this->assertSame('gpt-3.5-turbo', Settings::get('opensearch_nlp.model_path'), 'Model path should be OpenAI model');
  }

  /**
   * Tests that config values are used when no settings.php override exists.
   */
  public function testFallbackToConfigWhenNoOverride(): void {
    $config = $this->config('opensearch_nlp.nlp_settings');

    // Update config values
    $config->set('model_path', 'custom/model/path')
      ->set('model_version', '3.0.0')
      ->set('model_group', 'custom_group')
      ->save();

    // Verify config values are returned when no settings override
    $this->assertSame('custom/model/path', $config->get('model_path'), 'Config value should be used');
    $this->assertSame('3.0.0', $config->get('model_version'), 'Config version should be used');
    $this->assertSame('custom_group', $config->get('model_group'), 'Config group should be used');

    // Verify Settings::get() returns NULL (no override)
    $this->assertNull(Settings::get('opensearch_nlp.model_path'), 'No settings override should exist');
    $this->assertNull(Settings::get('opensearch_nlp.model_version'), 'No settings override should exist');
  }

  /**
   * Tests partial overrides (some fields overridden, others from config).
   */
  public function testPartialSettingsOverride(): void {
    $config = $this->config('opensearch_nlp.nlp_settings');

    // Set config values
    $config->set('model_path', 'config/model/path')
      ->set('model_version', '1.0.0')
      ->set('model_group', 'config_group')
      ->set('model_description', 'Config description')
      ->save();

    // Override only some values in settings.php
    $settings = Settings::getAll();
    $settings['opensearch_nlp.model_path'] = 'settings/override/path';
    $settings['opensearch_nlp.model_group'] = 'settings_override_group';
    // Note: model_version and model_description are NOT overridden

    new Settings($settings);

    // Verify overridden values
    $this->assertSame(
      'settings/override/path',
      Settings::get('opensearch_nlp.model_path'),
      'Model path should come from settings override'
    );
    $this->assertSame(
      'settings_override_group',
      Settings::get('opensearch_nlp.model_group'),
      'Model group should come from settings override'
    );

    // Verify non-overridden values still come from config
    $this->assertNull(
      Settings::get('opensearch_nlp.model_version'),
      'Model version has no override, should be NULL'
    );
    $this->assertSame('1.0.0', $config->get('model_version'), 'Model version should come from config');
    $this->assertSame('Config description', $config->get('model_description'), 'Description should come from config');
  }

  /**
   * Tests environment variable integration via settings.php.
   */
  public function testEnvironmentVariableIntegration(): void {
    // Simulate environment variables being used in settings.php
    // Note: In real usage, this would be in settings.php like:
    // $settings['opensearch_nlp.model_path'] = getenv('OPENSEARCH_MODEL_PATH') ?: 'default';

    $settings = Settings::getAll();
    // Simulate what would happen if OPENSEARCH_MODEL_PATH env var was set
    $env_model_path = 'env/huggingface/model-from-env-var';
    $settings['opensearch_nlp.model_path'] = $env_model_path;
    $settings['opensearch_nlp.model_version'] = '5.0.0';

    new Settings($settings);

    $this->assertSame(
      'env/huggingface/model-from-env-var',
      Settings::get('opensearch_nlp.model_path'),
      'Model path should come from environment variable via settings.php'
    );
    $this->assertSame(
      '5.0.0',
      Settings::get('opensearch_nlp.model_version'),
      'Model version should come from environment variable via settings.php'
    );
  }

}
