<?php

declare(strict_types=1);

namespace Drupal\Tests\opensearch_nlp\Unit;

use PHPUnit\Framework\MockObject\MockObject;
use Drupal\opensearch_nlp\Service\NLPIngestionService;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\State\StateInterface;
use Drupal\search_api\Task\ServerTaskManagerInterface;
use OpenSearch\Client;
use OpenSearch\Namespaces\MlNamespace;

/**
 * Unit tests for NLP model registration.
 *
 * @coversDefaultClass \Drupal\opensearch_nlp\Service\NLPIngestionService
 * @group opensearch_nlp
 */
class ModelRegistrationTest extends UnitTestCase {

  /**
   * The mocked logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $loggerFactory;

  /**
   * The mocked logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $logger;

  /**
   * The mocked config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $configFactory;

  /**
   * The mocked config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $config;

  /**
   * The mocked state service.
   *
   * @var \Drupal\Core\State\StateInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $state;

  /**
   * The mocked task manager.
   *
   * @var \Drupal\search_api\Task\ServerTaskManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $taskManager;

  /**
   * The mocked OpenSearch client.
   *
   * @var \OpenSearch\Client|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $client;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $this->loggerFactory->method('get')->willReturn($this->logger);

    $this->config = $this->createMock(ImmutableConfig::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->configFactory->method('get')->willReturn($this->config);

    $this->state = $this->createMock(StateInterface::class);
    $this->taskManager = $this->createMock(ServerTaskManagerInterface::class);
    $this->client = $this->createMock(Client::class);
  }

  /**
   * Test registerModelGroup creates a new model group.
   *
   * @covers ::registerModelGroup
   */
  public function testRegisterModelGroupCreatesNewGroup(): void {
    $mlNamespace = $this->createMock(MlNamespace::class);

    // Mock that no model groups exist.
    $mlNamespace->expects($this->once())
      ->method('getModelGroups')
      ->willReturn([
        'hits' => ['hits' => []],
      ]);

    // State returns NULL (no existing group ID).
    $this->state->expects($this->once())
      ->method('get')
      ->with('nlp_ingestion.model_group_id')
      ->willReturn(NULL);

    // Config returns model group settings.
    $this->config->method('get')
      ->willReturnMap([
        ['model_group', NULL, 'test-model-group'],
        ['model_group_description', NULL, 'Test model group description'],
      ]);

    // Expect registerModelGroup to be called.
    $mlNamespace->expects($this->once())
      ->method('registerModelGroup')
      ->with([
        'body' => [
          'name' => 'test-model-group',
          'description' => 'Test model group description',
        ],
      ])
      ->willReturn([
        'model_group_id' => 'test-group-123',
      ]);

    // Expect state to save the new group ID.
    $this->state->expects($this->once())
      ->method('set')
      ->with('nlp_ingestion.model_group_id', 'test-group-123');

    $this->client->method('ml')->willReturn($mlNamespace);

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    $result = $service->registerModelGroup();
    $this->assertSame('test-group-123', $result);
  }

  /**
   * Test registerModelGroup returns existing group ID.
   *
   * @covers ::registerModelGroup
   */
  public function testRegisterModelGroupReturnsExistingGroup(): void {
    $mlNamespace = $this->createMock(MlNamespace::class);

    // Mock that a model group already exists.
    $mlNamespace->expects($this->once())
      ->method('getModelGroups')
      ->willReturn([
        'hits' => [
          'hits' => [
            ['_id' => 'existing-group-123'],
          ],
        ],
      ]);

    // registerModelGroup should NOT be called.
    $mlNamespace->expects($this->never())
      ->method('registerModelGroup');

    $this->client->method('ml')->willReturn($mlNamespace);

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    $result = $service->registerModelGroup();
    $this->assertSame('existing-group-123', $result);
  }

  /**
   * Test deployModel successfully deploys a model.
   *
   * @covers ::deployModel
   */
  public function testDeployModelSuccess(): void {
    $mlNamespace = $this->createMock(MlNamespace::class);

    $mlNamespace->expects($this->once())
      ->method('deployModel')
      ->with(['id' => 'model-123'])
      ->willReturn([
        'task_id' => 'deploy-task-123',
        'status' => 'DEPLOYED',
      ]);

    $this->client->method('ml')->willReturn($mlNamespace);

    $this->logger->expects($this->once())
      ->method('notice')
      ->with('Successfully deployed model with id: model-123');

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    $result = $service->deployModel('model-123');
    $this->assertIsArray($result);
    $this->assertSame('deploy-task-123', $result['task_id']);
  }

  /**
   * Test deployModel handles exceptions.
   *
   * @covers ::deployModel
   */
  public function testDeployModelHandlesException(): void {
    $mlNamespace = $this->createMock(MlNamespace::class);

    $mlNamespace->expects($this->once())
      ->method('deployModel')
      ->willThrowException(new \Exception('Deploy failed'));

    $this->client->method('ml')->willReturn($mlNamespace);

    $this->logger->expects($this->once())
      ->method('error')
      ->with($this->stringContains('Failed to deploy model'));

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    $result = $service->deployModel('model-123');
    $this->assertNull($result);
  }

  /**
   * Test undeployModel successfully undeploys a model.
   *
   * @covers ::undeployModel
   */
  public function testUndeployModelSuccess(): void {
    $mlNamespace = $this->createMock(MlNamespace::class);

    $mlNamespace->expects($this->once())
      ->method('undeployModel')
      ->with(['id' => 'model-123'])
      ->willReturn(['acknowledged' => TRUE]);

    $this->client->method('ml')->willReturn($mlNamespace);

    $this->logger->expects($this->once())
      ->method('notice')
      ->with('Successfully undeployed model with id: model-123');

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    $result = $service->undeployModel('model-123');
    $this->assertIsArray($result);
    $this->assertTrue($result['acknowledged']);
  }

  /**
   * Test deleteModel successfully deletes a model.
   *
   * @covers ::deleteModel
   */
  public function testDeleteModelSuccess(): void {
    $mlNamespace = $this->createMock(MlNamespace::class);

    $mlNamespace->expects($this->once())
      ->method('deleteModel')
      ->with(['id' => 'model-123'])
      ->willReturn(['acknowledged' => TRUE]);

    $this->client->method('ml')->willReturn($mlNamespace);

    // Expect state to be cleared.
    $this->state->expects($this->once())
      ->method('delete')
      ->with('nlp_ingestion.model_id');

    $this->logger->expects($this->once())
      ->method('notice')
      ->with('Successfully deleted model with id: model-123');

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    $result = $service->deleteModel('model-123');
    $this->assertIsArray($result);
    $this->assertTrue($result['acknowledged']);
  }

  /**
   * Test getDeployedModels returns deployed models.
   *
   * @covers ::getDeployedModels
   */
  public function testGetDeployedModels(): void {
    $mlNamespace = $this->createMock(MlNamespace::class);

    $expectedParams = [
      'body' => [
        'query' => [
          'term' => [
            'model_state' => 'DEPLOYED',
          ],
        ],
        'size' => 1000,
      ],
    ];

    $mlNamespace->expects($this->once())
      ->method('searchModels')
      ->with($expectedParams)
      ->willReturn([
        'hits' => [
          'hits' => [
            ['_id' => 'model-1', '_source' => ['model_state' => 'DEPLOYED']],
            ['_id' => 'model-2', '_source' => ['model_state' => 'DEPLOYED']],
          ],
        ],
      ]);

    $this->client->method('ml')->willReturn($mlNamespace);

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    $result = $service->getDeployedModels();
    $this->assertIsArray($result);
    $this->assertCount(2, $result['hits']['hits']);
  }

}
