<?php

declare(strict_types=1);

namespace Drupal\Tests\opensearch_nlp\Unit;

use PHPUnit\Framework\MockObject\MockObject;
use Drupal\opensearch_nlp\Service\NLPIngestionService;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\State\StateInterface;
use Drupal\search_api\Task\ServerTaskManagerInterface;
use OpenSearch\Client;
use OpenSearch\Namespaces\ClusterNamespace;

/**
 * Unit tests for NLPIngestionService.
 *
 * @coversDefaultClass \Drupal\opensearch_nlp\Service\NLPIngestionService
 * @group opensearch_nlp
 */
class NLPIngestionServiceTest extends UnitTestCase {

  /**
   * The mocked logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $loggerFactory;

  /**
   * The mocked logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $logger;

  /**
   * The mocked config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $configFactory;

  /**
   * The mocked state service.
   *
   * @var \Drupal\Core\State\StateInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $state;

  /**
   * The mocked task manager.
   *
   * @var \Drupal\search_api\Task\ServerTaskManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $taskManager;

  /**
   * The mocked OpenSearch client.
   *
   * @var \OpenSearch\Client|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $client;

  /**
   * The service under test.
   *
   * @var \Drupal\opensearch_nlp\Service\NLPIngestionService
   */
  protected $service;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $this->loggerFactory->method('get')->willReturn($this->logger);

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->state = $this->createMock(StateInterface::class);
    $this->taskManager = $this->createMock(ServerTaskManagerInterface::class);
    $this->client = $this->createMock(Client::class);
  }

  /**
   * Test ensureMlCommonsSettings updates missing settings.
   *
   * @covers ::ensureMlCommonsSettings
   */
  public function testEnsureMlCommonsSettingsUpdatesMissingSettings(): void {
    $clusterNamespace = $this->createMock(ClusterNamespace::class);

    // Mock getSettings to return empty persistent settings.
    $clusterNamespace->expects($this->once())
      ->method('getSettings')
      ->with(['include_defaults' => TRUE])
      ->willReturn([
        'persistent' => [],
        'defaults' => [],
      ]);

    // Expect putSettings to be called with all required settings.
    $clusterNamespace->expects($this->once())
      ->method('putSettings')
      ->with([
        'body' => [
          'persistent' => [
            'plugins.ml_commons.only_run_on_ml_node' => FALSE,
            'plugins.ml_commons.native_memory_threshold' => 100,
            'plugins.ml_commons.agent_framework_enabled' => TRUE,
          ],
        ],
      ]);

    $this->client->method('cluster')->willReturn($clusterNamespace);

    // Create a partial mock to override getClient().
    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    $this->logger->expects($this->once())
      ->method('notice')
      ->with(
        'Updated ML Commons cluster settings: @keys',
        $this->callback(function (array $context): bool {
          $keys = explode(', ', $context['@keys']);
          return count($keys) === 3
            && in_array('plugins.ml_commons.only_run_on_ml_node', $keys)
            && in_array('plugins.ml_commons.native_memory_threshold', $keys)
            && in_array('plugins.ml_commons.agent_framework_enabled', $keys);
        })
      );

    $result = $service->ensureMlCommonsSettings();
    $this->assertTrue($result, 'ensureMlCommonsSettings should return TRUE on success');
  }

  /**
   * Test ensureMlCommonsSettings skips when settings already correct.
   *
   * @covers ::ensureMlCommonsSettings
   */
  public function testEnsureMlCommonsSettingsSkipsWhenCorrect(): void {
    $clusterNamespace = $this->createMock(ClusterNamespace::class);

    // Mock getSettings to return correct settings.
    $clusterNamespace->expects($this->once())
      ->method('getSettings')
      ->with(['include_defaults' => TRUE])
      ->willReturn([
        'persistent' => [
          'plugins.ml_commons.only_run_on_ml_node' => FALSE,
          'plugins.ml_commons.native_memory_threshold' => 100,
          'plugins.ml_commons.agent_framework_enabled' => TRUE,
        ],
        'defaults' => [],
      ]);

    // Expect putSettings to NOT be called.
    $clusterNamespace->expects($this->never())
      ->method('putSettings');

    $this->client->method('cluster')->willReturn($clusterNamespace);

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    // Logger should not be called when no updates needed.
    $this->logger->expects($this->never())->method('notice');

    $result = $service->ensureMlCommonsSettings();
    $this->assertTrue($result, 'ensureMlCommonsSettings should return TRUE when settings already correct');
  }

  /**
   * Test ensureMlCommonsSettings handles exceptions.
   *
   * @covers ::ensureMlCommonsSettings
   */
  public function testEnsureMlCommonsSettingsHandlesException(): void {
    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')
      ->willThrowException(new \Exception('Connection failed'));

    $this->logger->expects($this->once())
      ->method('error')
      ->with(
        'Failed ensuring ML Commons settings: @msg',
        ['@msg' => 'Connection failed']
      );

    $result = $service->ensureMlCommonsSettings();
    $this->assertFalse($result, 'ensureMlCommonsSettings should return FALSE on error');
  }

  /**
   * Test ensureMlCommonsSettings updates only different settings.
   *
   * @covers ::ensureMlCommonsSettings
   */
  public function testEnsureMlCommonsSettingsUpdatesOnlyDifferentSettings(): void {
    $clusterNamespace = $this->createMock(ClusterNamespace::class);

    // Mock getSettings with one correct and two incorrect settings.
    $clusterNamespace->expects($this->once())
      ->method('getSettings')
      ->with(['include_defaults' => TRUE])
      ->willReturn([
        'persistent' => [
          'plugins.ml_commons.only_run_on_ml_node' => FALSE,
        ],
        'defaults' => [
          'plugins.ml_commons.native_memory_threshold' => 90,
        ],
      ]);

    // Expect putSettings to be called with only the different settings.
    $clusterNamespace->expects($this->once())
      ->method('putSettings')
      ->with([
        'body' => [
          'persistent' => [
            'plugins.ml_commons.native_memory_threshold' => 100,
            'plugins.ml_commons.agent_framework_enabled' => TRUE,
          ],
        ],
      ]);

    $this->client->method('cluster')->willReturn($clusterNamespace);

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    $this->logger->expects($this->once())
      ->method('notice')
      ->with(
        'Updated ML Commons cluster settings: @keys',
        $this->callback(function (array $context): bool {
          $keys = explode(', ', $context['@keys']);
          return count($keys) === 2
            && in_array('plugins.ml_commons.native_memory_threshold', $keys)
            && in_array('plugins.ml_commons.agent_framework_enabled', $keys);
        })
      );

    $result = $service->ensureMlCommonsSettings();
    $this->assertTrue($result);
  }

}
