<?php

declare(strict_types=1);

namespace Drupal\Tests\opensearch_nlp\Unit;

use PHPUnit\Framework\MockObject\MockObject;
use Drupal\opensearch_nlp\Service\NLPIngestionService;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\State\StateInterface;
use Drupal\search_api\Task\ServerTaskManagerInterface;
use OpenSearch\Client;
use OpenSearch\Namespaces\IngestNamespace;
use OpenSearch\Namespaces\SearchPipelineNamespace;

/**
 * Unit tests for NLP pipeline creation.
 *
 * @coversDefaultClass \Drupal\opensearch_nlp\Service\NLPIngestionService
 * @group opensearch_nlp
 */
class PipelineCreationTest extends UnitTestCase {

  /**
   * The mocked logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $loggerFactory;

  /**
   * The mocked logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $logger;

  /**
   * The mocked config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $configFactory;

  /**
   * The mocked config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $config;

  /**
   * The mocked state service.
   *
   * @var \Drupal\Core\State\StateInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $state;

  /**
   * The mocked task manager.
   *
   * @var \Drupal\search_api\Task\ServerTaskManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $taskManager;

  /**
   * The mocked OpenSearch client.
   *
   * @var \OpenSearch\Client|\PHPUnit\Framework\MockObject\MockObject
   */
  protected MockObject $client;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $this->loggerFactory->method('get')->willReturn($this->logger);

    $this->config = $this->createMock(ImmutableConfig::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->configFactory->method('get')->willReturn($this->config);

    $this->state = $this->createMock(StateInterface::class);
    $this->taskManager = $this->createMock(ServerTaskManagerInterface::class);
    $this->client = $this->createMock(Client::class);
  }

  /**
   * Test registerNlpIngestPipeline creates pipelines for enabled indexes.
   *
   * @covers ::registerNlpIngestPipeline
   */
  public function testRegisterNlpIngestPipelineCreatesForEnabledIndexes(): void {
    $ingestNamespace = $this->createMock(IngestNamespace::class);

    // Mock config with one enabled index.
    $this->config->method('get')
      ->with('indexes')
      ->willReturn([
        'test_index' => [
          'enable_nlp' => '1',
          'mapping_embedding_pairs' => 'title|title_embedding',
          'ingestion_pipeline_id' => 'test-ingestion-pipeline',
          'search_pipeline_id' => 'test-search-pipeline',
        ],
      ]);

    // Mock getIngestionPipelines returns empty (no existing pipelines).
    $ingestNamespace->expects($this->once())
      ->method('getPipeline')
      ->willReturn([]);

    // Expect putPipeline to be called with correct structure.
    $ingestNamespace->expects($this->once())
      ->method('putPipeline')
      ->with([
        'id' => 'test-ingestion-pipeline',
        'body' => [
          'description' => 'NLP ingest pipeline for index: test_index',
          'processors' => [
            [
              'text_embedding' => [
                'model_id' => 'model-123',
                'field_map' => [
                  'title' => 'title_embedding',
                ],
              ],
            ],
          ],
        ],
      ]);

    $this->client->method('ingest')->willReturn($ingestNamespace);

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient', 'getIngestionPipelines'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);
    $service->method('getIngestionPipelines')->willReturn([]);

    $result = $service->registerNlpIngestPipeline('model-123');
    $this->assertTrue($result);
  }

  /**
   * Test registerNlpIngestPipeline skips indexes without NLP enabled.
   *
   * @covers ::registerNlpIngestPipeline
   */
  public function testRegisterNlpIngestPipelineSkipsDisabledIndexes(): void {
    $ingestNamespace = $this->createMock(IngestNamespace::class);

    // Mock config with disabled index.
    $this->config->method('get')
      ->with('indexes')
      ->willReturn([
        'test_index' => [
          'enable_nlp' => '0',
          'mapping_embedding_pairs' => 'title|title_embedding',
          'ingestion_pipeline_id' => 'test-ingestion-pipeline',
        ],
      ]);

    // putPipeline should NOT be called.
    $ingestNamespace->expects($this->never())
      ->method('putPipeline');

    $this->client->method('ingest')->willReturn($ingestNamespace);

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient', 'getIngestionPipelines'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);
    $service->method('getIngestionPipelines')->willReturn([]);

    $result = $service->registerNlpIngestPipeline('model-123');
    $this->assertTrue($result);
  }

  /**
   * Test registerSearchPipeline creates search pipeline.
   *
   * @covers ::registerSearchPipeline
   */
  public function testRegisterSearchPipelineCreatesNewPipeline(): void {
    $searchPipelineNamespace = $this->createMock(SearchPipelineNamespace::class);

    // Mock search pipeline config.
    $pipelineConfig = $this->createMock(ImmutableConfig::class);
    $pipelineConfig->method('get')
      ->willReturnMap([
        ['normalization_technique', NULL, 'min_max'],
        ['combination_technique', NULL, 'arithmetic_mean'],
        ['combination_weights', NULL, '0.5,0.5'],
        ['description', NULL, 'Test pipeline description'],
      ]);

    $this->configFactory->method('get')
      ->willReturnMap([
        ['opensearch_nlp.nlp_settings', TRUE, $this->config],
        ['opensearch_nlp.search_pipeline_settings', TRUE, $pipelineConfig],
      ]);

    $expectedBody = [
      'description' => 'Test pipeline description',
      'phase_results_processors' => [
        [
          'normalization-processor' => [
            'normalization' => [
              'technique' => 'min_max',
            ],
            'combination' => [
              'technique' => 'arithmetic_mean',
              'parameters' => [
                'weights' => [0.5, 0.5],
              ],
            ],
          ],
        ],
      ],
    ];

    $searchPipelineNamespace->expects($this->once())
      ->method('put')
      ->with([
        'id' => 'test-pipeline',
        'body' => $expectedBody,
      ]);

    $this->client->method('searchPipeline')->willReturn($searchPipelineNamespace);

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient', 'isExistingSearchPipeline'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);
    $service->method('isExistingSearchPipeline')->willReturn(FALSE);

    $service->registerSearchPipeline('test-pipeline');
  }

  /**
   * Test registerSearchPipeline skips existing pipeline.
   *
   * @covers ::registerSearchPipeline
   */
  public function testRegisterSearchPipelineSkipsExisting(): void {
    $searchPipelineNamespace = $this->createMock(SearchPipelineNamespace::class);

    // Put should NOT be called.
    $searchPipelineNamespace->expects($this->never())
      ->method('put');

    $this->client->method('searchPipeline')->willReturn($searchPipelineNamespace);

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient', 'isExistingSearchPipeline'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);
    $service->method('isExistingSearchPipeline')->willReturn(TRUE);

    $this->logger->expects($this->once())
      ->method('notice')
      ->with($this->stringContains('already registered'));

    $service->registerSearchPipeline('test-pipeline');
  }

  /**
   * Test deletePipeline successfully deletes ingestion pipeline.
   *
   * @covers ::deletePipeline
   */
  public function testDeletePipelineSuccess(): void {
    $ingestNamespace = $this->createMock(IngestNamespace::class);

    $ingestNamespace->expects($this->once())
      ->method('deletePipeline')
      ->with(['id' => 'test-pipeline'])
      ->willReturn(['acknowledged' => TRUE]);

    $this->client->method('ingest')->willReturn($ingestNamespace);

    $this->logger->expects($this->once())
      ->method('notice')
      ->with('Successfully deleted NLP ingest pipeline: test-pipeline');

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    $result = $service->deletePipeline('test-pipeline');
    $this->assertIsArray($result);
    $this->assertTrue($result['acknowledged']);
  }

  /**
   * Test deleteSearchPipeline successfully deletes search pipeline.
   *
   * @covers ::deleteSearchPipeline
   */
  public function testDeleteSearchPipelineSuccess(): void {
    $searchPipelineNamespace = $this->createMock(SearchPipelineNamespace::class);

    $searchPipelineNamespace->expects($this->once())
      ->method('delete')
      ->with(['id' => 'test-search-pipeline'])
      ->willReturn(['acknowledged' => TRUE]);

    $this->client->method('searchPipeline')->willReturn($searchPipelineNamespace);

    $this->logger->expects($this->once())
      ->method('notice')
      ->with('Successfully deleted search pipeline: test-search-pipeline');

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    $result = $service->deleteSearchPipeline('test-search-pipeline');
    $this->assertIsArray($result);
    $this->assertTrue($result['acknowledged']);
  }

  /**
   * Test getIngestionPipelines returns pipelines.
   *
   * @covers ::getIngestionPipelines
   */
  public function testGetIngestionPipelines(): void {
    $ingestNamespace = $this->createMock(IngestNamespace::class);

    $ingestNamespace->expects($this->once())
      ->method('getPipeline')
      ->willReturn([
        'pipeline-1' => [
          'description' => 'Pipeline 1',
          'processors' => [],
        ],
        'pipeline-2' => [
          'description' => 'Pipeline 2',
          'processors' => [],
        ],
      ]);

    $this->client->method('ingest')->willReturn($ingestNamespace);

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);

    $result = $service->getIngestionPipelines();
    $this->assertIsArray($result);
    $this->assertArrayHasKey('pipeline-1', $result);
    $this->assertArrayHasKey('pipeline-2', $result);
  }

  /**
   * Test updateSearchPipeline updates pipeline configuration.
   *
   * @covers ::updateSearchPipeline
   */
  public function testUpdateSearchPipeline(): void {
    $searchPipelineNamespace = $this->createMock(SearchPipelineNamespace::class);

    $updatedValues = [
      'normalization_technique' => 'l2',
      'combination_technique' => 'geometric_mean',
      'combination_weights' => '0.7,0.3',
      'description' => 'Updated pipeline',
    ];

    $expectedBody = [
      'description' => 'Updated pipeline',
      'phase_results_processors' => [
        [
          'normalization-processor' => [
            'normalization' => [
              'technique' => 'l2',
            ],
            'combination' => [
              'technique' => 'geometric_mean',
              'parameters' => [
                'weights' => [0.7, 0.3],
              ],
            ],
          ],
        ],
      ],
    ];

    $searchPipelineNamespace->expects($this->once())
      ->method('put')
      ->with([
        'id' => 'test-pipeline',
        'body' => $expectedBody,
      ]);

    $this->client->method('searchPipeline')->willReturn($searchPipelineNamespace);

    $this->logger->expects($this->once())
      ->method('notice')
      ->with('Successfully updated NLP search pipeline: test-pipeline');

    $service = $this->getMockBuilder(NLPIngestionService::class)
      ->setConstructorArgs([
        $this->loggerFactory,
        $this->configFactory,
        $this->state,
        $this->taskManager,
      ])
      ->onlyMethods(['getClient', 'isExistingSearchPipeline'])
      ->getMock();

    $service->method('getClient')->willReturn($this->client);
    $service->method('isExistingSearchPipeline')->willReturn(TRUE);

    $result = $service->updateSearchPipeline('test-pipeline', $updatedValues);
    $this->assertTrue($result);
  }

}
