<?php

namespace Drupal\openwoo_publish\Plugin\OpenWooPublish;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Datetime\DateFormatter;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\file\FileInterface;
use Drupal\media\MediaInterface;
use Drupal\openwoo\OpenWooPublicationFieldsProviderInterface;
use Drupal\openwoo_publish\Attribute\OpenWooPublish;
use Drupal\openwoo_publish\Entity\OpenWooPublicationInterface;
use Drupal\openwoo_publish\Plugin\OpenWooPublishPluginBase;
use GuzzleHttp\Psr7\Utils;
use GuzzleHttp\RequestOptions;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides an OpenWoo Publish plugin for OpenWoo.app.
 */
#[OpenWooPublish(
  id: 'openwoo_app',
  label: new TranslatableMarkup('OpenWoo.app'),
  description: new TranslatableMarkup('Publish OpenWoo publications to OpenWoo.app'),
  configName: 'openwoo_publish.openwoo_app',
)]
class OpenWooApp extends OpenWooPublishPluginBase implements OpenWooPublicationFieldsProviderInterface {

  use StringTranslationTrait;

  /**
   * Array with the publication data.
   */
  protected array $publicationData;

  /**
   * The date formatter service.
   */
  protected DateFormatter $dateFormatter;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('http_client'),
      $container->get('plugin.manager.openwoo_publish'),
      $container->get('entity_type.manager'),
      $container->get('file_url_generator'),
      $container->get('openwoo.publication_fields'),
    );

    $instance->dateFormatter = $container->get('date.formatter');

    return $instance;
  }

  /**
   * Returns the API URL.
   *
   * @param string $entity_type
   *   The entity type to get the API URL for. Defaults to 'objects'.
   *
   * @returns string
   *   The API URL.
   */
  public function getApiUrl(string $entity_type = 'objects'): string {
    return rtrim($this->config()->get('api_url'), '/') . '/' . $entity_type;
  }

  /**
   * Returns the username for the API.
   *
   * @return string|null
   *   The username for the API.
   */
  public function getUsername(): ?string {
    return $this->configFactory->get($this->pluginDefinition['configName'])->get('username');
  }

  /**
   * Returns the password for the API.
   *
   * @return string|null
   *   The password for the API.
   */
  public function getPassword(): ?string {
    return $this->configFactory->get($this->pluginDefinition['configName'])->get('password');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormFields(?array $values): array {
    $elements = [];

    $elements['schema'] = [
      'type' => 'select',
      'label' => $this->t('Category'),
      'description' => $this->t('Choose the suitable category from the pull-down. The category determines the metadata fields that are available for the publication.'),
      'required' => TRUE,
      'options' => $this->getSchemaOptions($this->config()->get('register_id')),
    ];

    if (isset($values['schema'])) {
      $schema = $this->getSchema($values['schema']);
      if ($schema) {
        // Sort the properties by their order.
        uasort($schema['properties'], function ($a, $b) {
          if (!isset($a['order']) || !isset($b['order'])) {
            // We cannot determine an order. Consider both entries equal.
            return 0;
          }
          return $a['order'] <=> $b['order'];
        });

        foreach ($schema['properties'] as $key => $property) {
          // We do not support arbitrary values for now.
          if ('values' === $key) {
            continue;
          }

          // We use the entity label as the title.
          if ('titel' === $key) {
            continue;
          }

          // We use the schema as the category.
          if ('categorie' === $key) {
            continue;
          }

          // Skip fields that are configured to be invisible.
          if (isset($property['visible']) && !$property['visible']) {
            continue;
          }

          $elements[$key] = [
            'type' => 'string' === $property['type'] ? 'textfield' : $property['type'],
            'label' => mb_ucfirst($property['title']),
            'description' => $property['description'] ?? '',
            'required' => $property['required'] ?? FALSE,
            'default_value' => $values[$key] ?? '',
          ];
        }
      }
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function getReadableValue(string $property, mixed $value): mixed {
    if ('schema' === $property) {
      $schema = $this->getSchema($value);
      return $schema['title'];
    }

    return $value;
  }

  /**
   * {@inheritdoc}
   */
  public function addPublication(OpenWooPublicationInterface $openwoo_publication): string|null {
    $body = $this->buildRequestBody($openwoo_publication);

    $content = $this->sendRequest(
      $this->getApiUrl() . '/' . $this->config()->get('register_id') . '/' . $this->publicationData['schema'],
      $this->getOptions($body),
      Request::METHOD_POST,
    );

    $decoded_content = !empty($content) ? Json::decode($content) : NULL;
    if (NULL === $decoded_content) {
      return NULL;
    }
    $openwoo_publication->setExternalId($decoded_content['id']);

    // Upload files to OpenWoo.app.
    $this->uploadAttachments($openwoo_publication);

    // If the publication is published on our side, we need to publish it on
    // OpenWoo.app as well.
    if ($openwoo_publication->isPublished()) {
      $this->sendRequest(
        $this->getApiUrl() . '/' . $this->config()->get('register_id') . '/' . $this->publicationData['schema'] . '/' . $decoded_content['id'] . '/publish',
        $this->getOptions(),
        Request::METHOD_POST,
      );
    }

    return $content;
  }

  /**
   * {@inheritdoc}
   */
  public function updatePublication(OpenWooPublicationInterface $openwoo_publication, string $external_id): string|null {
    $body = $this->buildRequestBody($openwoo_publication);

    $content = $this->sendRequest(
      $this->getApiUrl() . '/' . $this->config()->get('register_id') . '/' . $this->publicationData['schema'] . '/' . $external_id,
      $this->getOptions($body),
      Request::METHOD_PUT,
    );

    $decoded_content = (!empty($content)) ? Json::decode($content) : NULL;
    if (NULL === $decoded_content) {
      return NULL;
    }

    // Upload files to OpenWoo.app.
    $this->uploadAttachments($openwoo_publication);

    // Check if we need to publish or unpublish the publication.
    $is_published = !empty($decoded_content['@self']['published']);
    if ($openwoo_publication->isPublished() && !$is_published) {
      $this->sendRequest(
        $this->getApiUrl() . '/' . $this->config()->get('register_id') . '/' . $this->publicationData['schema'] . '/' . $decoded_content['id'] . '/publish',
        $this->getOptions(),
        Request::METHOD_POST,
      );
    }
    elseif (!$openwoo_publication->isPublished() && $is_published) {
      $this->sendRequest(
        $this->getApiUrl() . '/' . $this->config()->get('register_id') . '/' . $this->publicationData['schema'] . '/' . $decoded_content['id'] . '/depublish',
        $this->getOptions(),
        Request::METHOD_POST,
      );
    }

    return $content;
  }

  /**
   * {@inheritdoc}
   */
  public function deletePublication(string $external_id, string $schema_id = ''): string|null {
    $content = $this->sendRequest(
      $this->getApiUrl() . '/' . $this->config()->get('register_id') . '/' . $schema_id . '/' . $external_id,
      $this->getOptions(),
      Request::METHOD_DELETE,
    );

    return $content;
  }

  /**
   * Returns the options for the request, setting the header and body.
   *
   * @param string|array|null $body
   *   The body for the request. If a string, the body is allowed to be JSON.
   *   If an array, the body is expected to be a multipart array, which is an
   *   array of associative array with (at least) name and contents keys.
   * @param array $headers
   *   The headers for the request.
   *
   * @return array
   *   The options to use for the request.
   */
  public function getOptions(string|array|null $body = NULL, array $headers = []): array {
    $options = [
      RequestOptions::HEADERS => $headers,
      RequestOptions::HTTP_ERRORS => FALSE,
    ];

    $username = $this->getUsername();
    $password = $this->getPassword();
    if ($username && $password) {
      $options[RequestOptions::AUTH] = [$username, $password];
    }

    if ($body) {
      $body_type = is_array($body) ? RequestOptions::MULTIPART : (json_validate($body) ? RequestOptions::JSON : RequestOptions::BODY);
      $options[$body_type] = RequestOptions::JSON === $body_type ? Json::decode($body) : $body;
    }

    return $options;
  }

  /**
   * Builds the request body for an API request based on the publication entity.
   *
   * @param \Drupal\openwoo_publish\Entity\OpenWooPublicationInterface $openwoo_publication
   *   The publication entity.
   *
   * @return string
   *   The JSON encoded request body.
   */
  protected function buildRequestBody(OpenWooPublicationInterface $openwoo_publication): string {
    $this->publicationData = unserialize(
      $openwoo_publication->get('data')->value,
      ['allowed_classes' => FALSE],
    );

    $body = $this->publicationData + ['titel' => $openwoo_publication->label()];

    return Json::encode($body);
  }

  /**
   * Uploads the attachments for the publication.
   *
   * @param \Drupal\openwoo_publish\Entity\OpenWooPublicationInterface $publication
   *   The publication entity.
   */
  protected function uploadAttachments(OpenWooPublicationInterface $publication): void {
    if (!$publication->hasField('attachments') || $publication->get('attachments')->isEmpty()) {
      return;
    }

    foreach ($publication->get('attachments') as $attachment) {
      // @todo How do we know the file is not uploaded already?
      $target_id = $attachment->get('target_id')->getValue();
      $media_entity = $this->entityTypeManager->getStorage('media')->load($target_id);
      assert($media_entity instanceof MediaInterface);
      $file = $media_entity->get('field_openwoo_media_file')->entity;
      assert($file instanceof FileInterface);

      if (!file_exists($file->getFileUri())) {
        continue;
      }

      $multipart = [
        [
          'name' => 'files[]',
          'contents' => Utils::tryFopen($file->getFileUri(), 'r'),
          'filename' => $file->getFilename(),
        ],
        [
          // (For now), we always publish the file. As long as we do not share
          // the public URL before the publication is published we are fine.
          'name' => 'share',
          'contents' => 'true',
        ],
      ];

      $this->sendRequest(
        $this->getApiUrl() . '/' . $this->config()->get('register_id') . '/' . $this->publicationData['schema'] . '/' . $publication->get('external_id')->value . '/filesMultipart',
        $this->getOptions($multipart),
        Request::METHOD_POST,
      );
    }
  }

  /**
   * Helper function to parse the date value.
   *
   * @param string|int $date
   *   The value to parse as iso date.
   *
   * @return string
   *   The iso date, defaults to an empty string if no valid date is found.
   */
  protected function getIsoDate(string|int $date): string {
    if (empty($date)) {
      return '';
    }

    if (is_string($date)) {
      $date = (new DrupalDateTime($date))->getTimestamp();
    }

    return $this->dateFormatter->format($date, 'custom', "Y-m-d\\TH:i:sO", NULL, 'nl');
  }

  /**
   * {@inheritDoc}
   */
  public function getElements(): array {
    $form['api_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API base URL'),
      '#description' => $this->t('The API base URL of your OpenWoo.app instance.'),
      '#default_value' => $this->config()->get('api_url') ?? '',
    ];

    $form['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username'),
      '#description' => $this->t('The API username for your OpenWoo.app instance. The username is not required, but the API responses are faster when authenticated.'),
      '#default_value' => $this->config()->get('username') ?? '',
    ];

    $password_description = $this->t('The password is not required, but the API responses are faster when authenticated.<br>The password must be saved in settings.php:<br>$config[&apos;openwoo_publish.openwoo_app&apos;][&apos;password&apos;] = &apos;PASSWORD&apos;;');
    $password_value = $this->configFactory->get($this->pluginDefinition['configName'])->get('password');
    $password = $password_value ? $this->maskValue($password_value) : $this->t('Not set');

    $form['password'] = [
      '#type' => 'item',
      '#title' => $this->t('Password'),
      '#plain_text' => $password,
      '#description' => $password_description,
    ];

    $form['register_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Woo register'),
      '#description' => $this->t('Select the Woo register in the OpenWoo.app instance.'),
      '#options' => $this->getRegisterOptions(),
      '#default_value' => $this->config()->get('register_id') ?? '',
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function submitElements(array $form, FormStateInterface $form_state) {
    $this->config()
      ->set('api_url', $form_state->getValue('api_url'))
      ->set('username', $form_state->getValue('username'))
      ->set('register_id', $form_state->getValue('register_id'))
      ->save();
  }

  /**
   * Masks a value.
   *
   * @param string $value
   *   The value to mask.
   *
   * @return string
   *   The masked value.
   */
  protected function maskValue(string $value): string {
    $mask = str_repeat('*', strlen($value) - 4);
    return $mask . substr($value, -4);
  }

  /**
   * Returns the register options.
   *
   * @return array
   *   The register options.
   */
  protected function getRegisterOptions(): array {
    $register_options = [];

    $response = $this->sendRequest(
      $this->getApiUrl('registers'),
      $this->getOptions(),
      Request::METHOD_GET,
    );

    $data = Json::decode($response);
    if (isset($data['results'])) {
      foreach ($data['results'] as $register) {
        $register_options[$register['id']] = $register['title'];
      }
    }

    return $register_options;
  }

  /**
   * Returns the schema options.
   *
   * @param int $register_id
   *   The register ID.
   *
   * @return array
   *   The schema options.
   */
  protected function getSchemaOptions(int $register_id): array {
    $schema_options = [];

    $response = $this->sendRequest(
      $this->getApiUrl('registers') . '/' . $register_id . '/schemas',
      $this->getOptions(),
      Request::METHOD_GET,
    );

    $data = Json::decode($response);
    if (isset($data['results'])) {
      foreach ($data['results'] as $schema) {
        $schema_options[$schema['id']] = $schema['title'];
      }
    }

    asort($schema_options);
    return $schema_options;
  }

  /**
   * Returns a schema by ID.
   *
   * @param int $schema_id
   *   The schema ID.
   *
   * @return array|null
   *   The schema data, or null if not found.
   */
  protected function getSchema(int $schema_id): ?array {
    $response = $this->sendRequest(
      $this->getApiUrl('schemas') . '/' . $schema_id,
      $this->getOptions(),
      Request::METHOD_GET,
    );

    $data = Json::decode($response);
    return $data ?? NULL;
  }

}
