<?php

namespace Drupal\openwoo_search\Plugin\OpenWooSearch;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\openwoo_search\Attribute\OpenWooSearch;
use Drupal\openwoo_search\Plugin\OpenWooSearchPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides an OpenWoo Search plugin for OpenWoo.app.
 */
#[OpenWooSearch(
  id: 'openwoo_app',
  label: new TranslatableMarkup('OpenWoo.app'),
  description: new TranslatableMarkup('Provides search for OpenWoo publications in OpenWoo.app.'),
  configName: 'openwoo_search.openwoo_app',
)]
class OpenWooApp extends OpenWooSearchPluginBase {

  use StringTranslationTrait;

  /**
   * The cache lifetime in seconds.
   */
  const CACHE_LIFETIME = 14400;

  /**
   * The date formatter service.
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * The time service.
   */
  protected TimeInterface $time;

  /**
   * The cache backend.
   */
  protected CacheBackendInterface $cache;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('http_client'),
      $container->get('plugin.manager.openwoo_search'),
      $container->get('openwoo.publication_fields'),
    );

    $instance->dateFormatter = $container->get('date.formatter');
    $instance->cache = $container->get('cache.default');
    $instance->time = $container->get('datetime.time');

    return $instance;
  }

  /**
   * Returns the API URL.
   *
   * @returns string
   *   The API URL.
   */
  public function getApiUrl(): string {
    return rtrim($this->config()->get('api_url'), '/') . '/publications';
  }

  /**
   * Returns the username for the API.
   *
   * @return string|null
   *   The username for the API.
   */
  public function getUsername(): ?string {
    return $this->configFactory->get($this->pluginDefinition['configName'])->get('username');
  }

  /**
   * Returns the password for the API.
   *
   * @return string|null
   *   The password for the API.
   */
  public function getPassword(): ?string {
    return $this->configFactory->get($this->pluginDefinition['configName'])->get('password');
  }

  /**
   * {@inheritdoc}
   */
  public function search(?string $search_text = NULL, ?string $year = NULL, ?string $category = NULL, ?int $page = NULL): ?array {
    $query = [
      '_search' => $search_text ?? NULL,
      '@self[published][gte]' => $year ? $year . '-01-01T00:00:00Z' : NULL,
      '@self[published][lte]' => $year ? $year . '-12-31T23:59:59Z' : NULL,
      '@self[schema]' => $category ?? NULL,
      '_limit' => $this->getItemsPerPage(),
      '_offset' => ($page ? $page - 1 : 0) * $this->getItemsPerPage(),
    ];
    $query = array_filter($query, fn($value) => !empty($value));
    $options = $this->getOptions($query);

    $content = $this->sendRequest($this->getApiUrl(), $options);
    if (empty($content)) {
      return NULL;
    }

    // @todo Rewrite base classes and interfaces to allow for arrays.
    return $this->parseResults((object) Json::decode($content), $search_text);
  }

  /**
   * {@inheritdoc}
   */
  public function getDetails(string $external_publication_id): ?array {
    $options = $this->getOptions();
    $content = $this->sendRequest($this->getApiUrl() . '/' . $external_publication_id, $options);
    // @todo Rewrite base classes and interfaces to allow for arrays.
    return (!empty($content)) ? $this->getPublicationValues((object) Json::decode($content)) : NULL;
  }

  /**
   * Returns publication values.
   *
   * @param object $publication
   *   The publication object.
   *
   * @return array
   *   An associative array with publication values.
   */
  public function getPublicationValues(object $publication): array {
    $values = [];
    $attachments = $this->getAttachmentsList($publication->id);

    foreach ($publication as $property => $value) {
      if (in_array($property, ['attachments', 'categorie', 'values', '@self'])) {
        continue;
      }

      $values[$property] = $value;
    }

    $values += [
      'title' => $publication->titel ?? $publication->{'@self'}['name'],
      // Schema is needed to get all field definitions.
      'schema' => $publication->{'@self'}['schema']['id'],
      'attachments' => $attachments,
    ];

    return $values;
  }

  /**
   * Returns the options for the request.
   *
   * @param array $query_params
   *   The query parameters.
   * @param array $headers
   *   The headers for the request.
   *
   * @return array
   *   Return the options to use for the request.
   */
  public function getOptions(array $query_params = [], array $headers = []): array {
    $options = [
      'query' => $query_params,
      'headers' => $headers,
      'http_errors' => FALSE,
    ];

    $username = $this->getUsername();
    $password = $this->getPassword();
    if ($username && $password) {
      $options['auth'] = [$username, $password];
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function getSearchStats(object $json, ?string $search_text = NULL): TranslatableMarkup {
    $page = (int) $json->page;
    $results_per_page = (int) $json->limit;
    $first = $json->total > 0 ? ($results_per_page * ($page - 1) + 1) : 0;
    $last = ($results_per_page * ($page - 1) + $results_per_page);
    $results = $first . ' - ' . min($last, $json->total);

    if (!isset($search_text)) {
      return $this->t("Result @results of @result_total results", [
        '@results' => $results,
        '@result_total' => $json->total,
      ]);
    }

    return $this->t("Result @results of @result_total results for search term: @search_term", [
      '@results' => $results,
      '@result_total' => $json->total,
      '@search_term' => $search_text,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function parseResults(object $json, ?string $search_text = NULL): array {
    $publications = [];

    foreach ($json->results as $item) {
      $result = new OpenWooSearchResult();

      $result->setId($item['id']);
      // We are not sure about the fields present in the schema. So we add a
      // fallback to the default object properties.
      $result->setTitle($item['titel'] ?? $item['@self']['name']);
      $result->setDescription($item['beschrijving'] ?? $item['@self']['description'] ?? '');
      $result->setSummary($item['samenvatting'] ?? $item['@self']['summary'] ?? '');

      $result->setCategory($item['@self']['schema']['title']);
      $result->setPublishDate($item['@self']['published']);
      $result->setPublishDateIso($item['@self']['published']);
      $result->setRemoteLink('/openwoo-search/result/' . $item['id']);

      $publications[] = $result;
    }

    return [
      'items' => $json->total,
      'pages' => $json->pages,
      'results_per_page' => $json->limit,
      'current_page' => $json->page,
      'number_of_pages' => $json->pages,
      'search_stats' => $this->getSearchStats($json, $search_text),
      'publications' => $publications,
    ];
  }

  /**
   * Returns the schema options.
   *
   * @todo We should abstract API communication for search and publish into a
   *   client that can be used by both submodules.
   *
   * @return array
   *   The schema options.
   */
  public function getCategoryOptions(): array {
    $cache_key = 'openwoo_search.category_options';
    if ($cache = $this->cache->get($cache_key)) {
      return $cache->data;
    }

    $schema_options = [];

    // We will probably factor this out soon. For now, we add an option to
    // override the schema ID, but we do not expose it in the UI.
    $register_id = $this->config()->get('register_id') ?? 2;
    $response = $this->sendRequest(
    // For now, we make some assumptions about the structure of the API URLs.
      str_replace('opencatalogi', 'openregister', rtrim($this->config()->get('api_url'), '/')) . '/registers/' . $register_id . '/schemas',
      $this->getOptions(),
      Request::METHOD_GET,
    );

    $data = Json::decode($response);
    if (isset($data['results'])) {
      foreach ($data['results'] as $schema) {
        $schema_options[$schema['id']] = $schema['title'];
      }
    }

    asort($schema_options);
    $this->storeInCache($cache_key, $schema_options);

    return $schema_options;
  }

  /**
   * {@inheritdoc}
   */
  public function getyearOptions(): array {
    $cache_key = 'openwoo_search.year_options';
    if ($cache = $this->cache->get($cache_key)) {
      return $cache->data;
    }

    $year_options = [];

    $request_options = [
      'query' => [
        '_facetable' => TRUE,
        '_facets' => [
          '@self' => [
            'published' => [
              'type' => 'date_histogram',
              'interval' => 'year',
            ],
          ],
        ],
      ],
    ];
    $response = $this->sendRequest(
    // For now, we make some assumptions about the structure of the API URLs.
      $this->getApiUrl(),
      $request_options + $this->getOptions(),
      Request::METHOD_GET,
    );

    $data = Json::decode($response);
    if (isset($data['facets']['facets']['@self']['published']['buckets'])) {
      foreach ($data['facets']['facets']['@self']['published']['buckets'] as $option) {
        $year_options[$option['key']] = $option['key'];
      }
    }

    krsort($year_options);
    $this->storeInCache($cache_key, $year_options);

    return $year_options;
  }

  /**
   * Returns a date in ISO format.
   *
   * @param string $date
   *   The value to parse as ISO date.
   *
   * @return string|null
   *   The ISO date, defaults to an empty string if no valid date is found.
   */
  protected function getIsoDate(string $date): ?string {
    if (empty($date)) {
      return NULL;
    }

    return $this->dateFormatter->format(strtotime($date), 'custom', "Y-m-d\\TH:i:sO", NULL, 'nl');
  }

  /**
   * {@inheritDoc}
   */
  public function getElements(): array {
    $form['api_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API base URL'),
      '#description' => $this->t('The API base URL of your OpenWoo.app instance.'),
      '#default_value' => $this->config()->get('api_url') ?? '',
    ];

    $form['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username'),
      '#description' => $this->t('The API username for your OpenWoo.app instance. The username is not required, but the API responses are faster when authenticated.'),
      '#default_value' => $this->config()->get('username') ?? '',
    ];

    $password_description = $this->t('The password is not required, but the API responses are faster when authenticated.<br>The password must be saved in settings.php:<br>$config[&apos;openwoo_search.openwoo_app&apos;][&apos;password&apos;] = &apos;PASSWORD&apos;;');
    $password_value = $this->configFactory->get($this->pluginDefinition['configName'])->get('password');
    $password = $password_value ? $this->maskValue($password_value) : $this->t('Not set');

    $form['password'] = [
      '#type' => 'item',
      '#title' => $this->t('Password'),
      '#plain_text' => $password,
      '#description' => $password_description,
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function submitElements(array $form, FormStateInterface $form_state): void {
    $this->config()
      ->set('api_url', $form_state->getValue('api_url'))
      ->set('username', $form_state->getValue('username'))
      ->save();
  }

  /**
   * Returns a list of attachments for the OpenWoo publication.
   *
   * @param int|string $publication_id
   *   The publication ID. This can be either the numeric ID or the UUID.
   *
   * @return array
   *   The array of attachments.
   */
  public function getAttachmentsList(int|string $publication_id): array {
    $attachments = [];

    $response = $this->sendRequest(
      $this->getApiUrl() . '/' . $publication_id . '/attachments',
      $this->getOptions(),
      Request::METHOD_GET,
    );

    $data = Json::decode($response);
    foreach ($data['results'] as $attachment) {
      $attachments[] = [
        'file_name' => $attachment['title'],
        'link' => $attachment['downloadUrl'],
        // Visibility is managed by the API.
        'status' => TRUE,
      ];
    }

    return $attachments;
  }

  /**
   * Masks a value.
   *
   * @param string $value
   *   The value to mask.
   *
   * @return string
   *   The masked value.
   */
  protected function maskValue(string $value): string {
    $mask = str_repeat('*', strlen($value) - 4);
    return $mask . substr($value, -4);
  }

  /**
   * Stores data in cache for the given cache key.
   *
   * @param string $cache_key
   *   The cache key.
   * @param mixed $data
   *   The data that needs to be stored.
   */
  protected function storeInCache(string $cache_key, mixed $data): void {
    $this->cache->set(
      $cache_key,
      $data,
      $this->time->getRequestTime() + self::CACHE_LIFETIME,
      [
        'config:openwoo_search.settings',
      ]
    );
  }

}
