<?php

declare(strict_types=1);

namespace Drupal\operation\Command;

use Composer\Console\Input\InputArgument;
use Drupal\operation\Data\OperationId;
use Drupal\operation\Exception\OperationNotFound;
use Drupal\operation\Exception\OperationStatusTransition;
use Drupal\operation\Operator;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

/**
 * {@selfdoc}
 */
#[AsCommand(
  name: 'operation:cancel',
  description: 'Cancels an operation.',
)]
final class CancelOperation extends Command {

  /**
   * {@inheritdoc}
   */
  public function __construct(private readonly Operator $operator) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  protected function configure(): void {
    $this->addArgument(
      name: 'operation',
      mode: InputArgument::REQUIRED,
      description: 'Operation ID.'
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function execute(InputInterface $input, OutputInterface $output): int {
    $io = new SymfonyStyle($input, $output);

    try {
      $operation = $this->operator->getOperation(
        // @phpstan-ignore argument.type
        new OperationId($input->getArgument('operation')),
      );
      $this->operator->cancelOperation($operation);
    }
    catch (OperationNotFound) {
      $io->getErrorStyle()->error('Operation not found.');
      return self::FAILURE;
    }
    catch (OperationStatusTransition $exception) {
      $io->getErrorStyle()->error($exception->getMessage());
      return self::FAILURE;
    }
    catch (\InvalidArgumentException) {
      $io->getErrorStyle()->error('Wrong operation ID');
      return self::FAILURE;
    }

    $io->success('Done.');

    return self::SUCCESS;
  }

}
