<?php

declare(strict_types=1);

namespace Drupal\operation\Data;

/**
 * {@selfdoc}
 */
final readonly class Operation {

  /**
   * @param positive-int $totalJobs
   * @param non-negative-int $processedJobs
   */
  public function __construct(
    public OperationId $id,
    public OperationStatus $status,
    public \DateTimeImmutable $created,
    public \DateTimeImmutable $updated,
    public int $totalJobs,
    public int $processedJobs,
    public mixed $result,
  ) {
    if ($this->totalJobs <= 0) {
      throw new \InvalidArgumentException('Total jobs value must be greater than zero.');
    }
    if ($this->processedJobs < 0) {
      throw new \InvalidArgumentException('Processed jobs value must be greater or equal to zero.');
    }
    if ($this->processedJobs > $this->totalJobs) {
      throw new \InvalidArgumentException('Processed jobs value must not be greater than total jobs value.');
    }
  }

  /**
   * {@selfdoc}
   */
  public function isActive(): bool {
    // All other statuses are terminal.
    return $this->status === OperationStatus::Pending || $this->status === OperationStatus::InProgress;
  }

  /**
   * @param non-negative-int $processed_jobs
   */
  public function with(OperationStatus $status, \DateTimeImmutable $updated, int $processed_jobs, mixed $result): self {
    return new self(
      id: $this->id,
      status: $status,
      created: $this->created,
      updated: $updated,
      totalJobs: $this->totalJobs,
      processedJobs: $processed_jobs,
      result: $result,
    );
  }

}
