<?php

declare(strict_types=1);

namespace Drupal\operation\Element;

use Drupal\Core\Render\Attribute\RenderElement;
use Drupal\Core\Render\Element\RenderElementBase;
use Drupal\Core\StringTranslation\TranslatableMarkup as TM;
use Drupal\operation\Data\OperationId;
use Drupal\operation\Operator;

/**
 * Provides a render element to display an operation.
 *
 * Properties:
 * - #operation: Operation ID.
 *
 * Usage Example:
 * @code
 * $build['operation'] = [
 *   '#type' => 'operation',
 *   '#operation' => '19aa3c79-8e03-4eea-867e-b7d54a7c1ad8',
 * ];
 * @endcode
 *
 * @RenderElement("operation")
 */
#[RenderElement('operation')]
final class Operation extends RenderElementBase  {

  /**
   * {@inheritdoc}
   */
  public function getInfo(): array {
    return [
      '#theme' => 'operation',
      '#attached' => ['library' => ['operation/operation']],
      '#pre_render' => [[self::class, 'preRender']],
    ];
  }

  /**
   * Operation element pre render callback.
   */
  public static function preRender(array $element): array {
    if (!isset($element['#operation'])) {
      throw new \LogicException('Missing operation ID.');
    }
    $operation = self::getOperator()->getOperation(
      new OperationId($element['#operation']),
    );
    $element['#percentage'] = \round(100 * $operation->processedJobs / $operation->totalJobs, 2) . ' %';
    $element['#message'] = new TM(
      'Processed items: @processed of @total.',
      ['@processed' => $operation->processedJobs, '@total' => $operation->totalJobs],
    );
    return $element;
  }

  /**
   * {@selfdoc}
   */
  private static function getOperator(): Operator {
    return \Drupal::service(Operator::class);
  }


}
