<?php

declare(strict_types=1);

namespace Drupal\operation;

use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\KeyValueStore\KeyValueExpirableFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreExpirableInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\operation\Data\JobId;
use Drupal\operation\Data\JobInput;
use Drupal\operation\Data\JobOutput;
use Drupal\operation\Data\Operation;
use Drupal\operation\Data\OperationId;
use Drupal\operation\Data\OperationStatus;
use Drupal\operation\Exception\OperationNotFound;
use Drupal\operation\Exception\OperationStatusTransition;
use Psr\Clock\ClockInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * {@selfdoc}
 */
final readonly class Operator {

  private const int EXPIRE = 3_600;

  /**
   * {@selfdoc}
   */
  public function __construct(
    #[Autowire(service: 'keyvalue.expirable')]
    public KeyValueExpirableFactoryInterface $keyValueFactory,
    public QueueFactory $queueFactory,
    public ClockInterface $clock,
    public UuidInterface $uuidGenerator,
  ) {}

  /**
   * {@selfdoc}
   */
  public function createOperation(string $queue_name, array $payloads): Operation {
    $operation = new Operation(
      id: new OperationId($this->generateId()),
      status: OperationStatus::Pending,
      created: $this->clock->now(),
      updated: $this->clock->now(),
      totalJobs: \count($payloads),
      processedJobs: 0,
      result: NULL,
    );
    $this->saveOperation($operation);
    $queue = $this->queueFactory->get($queue_name);
    foreach ($payloads as $payload) {
      $input = new JobInput(
        jobId: new JobId($this->generateId()),
        operationId: $operation->id,
        payload: $payload,
      );
      $queue->createItem($input);
    }
    return $operation;
  }

  /**
   * @return \Drupal\operation\Data\Operation[]
   */
  public function getAllOperations(): array {
    return \array_map(
      // @phpstan-ignore argument.type
      $this->getOperation(...),
      \array_column(\array_values($this->getOperationCollection()->getAll()), 'id'),
    );
  }

  /**
   * {@selfdoc}
   */
  public function getOperation(OperationId $operation_id): Operation {
    $operation = $this->getOperationCollection()->get($operation_id->value);
    if (!$operation instanceof Operation) {
      throw new OperationNotFound($operation_id);
    }
    if (!$operation->isActive()) {
      return $operation;
    }
    $job_outputs = \array_values($this->getJobOutputCollection($operation->id)->getAll());

    $operation = $operation->with(
      status: match(\count($job_outputs)) {
        $operation->totalJobs => OperationStatus::Completed,
        0 => OperationStatus::Pending,
        default => OperationStatus::InProgress,
      },
      updated: $this->clock->now(),
      processed_jobs: \count($job_outputs),
      result: $job_outputs,
    );

    $this->saveOperation($operation);
    return $operation;
  }

  /**
   * {@selfdoc}
   */
  public function deleteOperation(Operation $operation): void {
    $this->getOperationCollection()->delete($operation->id->value);
  }

  /**
   * {@selfdoc}
   */
  public function cancelOperation(Operation $operation): void {
    if (!$operation->isActive()) {
      throw new OperationStatusTransition($operation->status, OperationStatus::Cancelled);
    }
    $this->saveOperation(
      $operation->with(
        status: OperationStatus::Cancelled,
        updated: $this->clock->now(),
        processed_jobs: $operation->processedJobs,
        result: $operation->result,
      )
    );
  }

  /**
   * {@selfdoc}
   */
  public function failOperation(Operation $operation): void {
    if (!$operation->isActive()) {
      throw new OperationStatusTransition($operation->status, OperationStatus::Cancelled);
    }
    $this->saveOperation(
      $operation->with(
        status: OperationStatus::Failed,
        updated: $this->clock->now(),
        processed_jobs: $operation->processedJobs,
        result: $operation->result,
      )
    );
  }

  /**
   * {@selfdoc}
   */
  public function submitJobOutput(JobOutput $job_output): void {
    $this->getJobOutputCollection($job_output->operationId)
      ->set($job_output->jobId->value, $job_output->output);
  }

  /**
   * {@selfdoc}
   */
  private function getOperationCollection(): KeyValueStoreExpirableInterface {
    return $this->keyValueFactory->get('operation');
  }

  /**
   * {@selfdoc}
   */
  private function getJobOutputCollection(OperationId $operation_id): KeyValueStoreExpirableInterface {
    return $this->keyValueFactory->get('op_job_result.' . $operation_id->value);
  }

  /**
   * {@selfdoc}
   */
  private function saveOperation(Operation $operation): void {
    $this->getOperationCollection()->setWithExpire($operation->id->value, $operation, self::EXPIRE);
  }

  /**
   * {@selfdoc}
   */
  private function generateId(): string {
    return $this->uuidGenerator->generate();
  }

}
