<?php

declare(strict_types=1);

namespace Drupal\operation\Plugin\QueueWorker;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\operation\Data\JobInput;
use Drupal\operation\Data\JobOutput;
use Drupal\operation\Operator;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for operation workers.
 */
abstract class OperationWorkerBase extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * {@selfdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, private Operator $operator) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    // @phpstan-ignore new.static
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get(Operator::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    if (!$data instanceof JobInput) {
      throw new \InvalidArgumentException(
        \sprintf('Wrong worker input. Expected %s, but %s was given.', JobInput::class, \get_debug_type($data)),
      );
    }
    // Ensure that the operation is still active.
    if (!$this->operator->getOperation($data->operationId)->isActive()) {
      return;
    }

    $output = $this->doProcessItem($data->payload);

    $this->operator->submitJobOutput(
      JobOutput::fromJobInput($data, $output),
    );
  }

  /**
   * {@selfdoc}
   */
  abstract protected function doProcessItem(mixed $payload): mixed;

}
