<?php

declare(strict_types=1);

namespace Drupal\Tests\operation\Unit;

use Drupal\operation\Data\Operation;
use Drupal\operation\Data\OperationId;
use Drupal\operation\Data\OperationStatus;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\TestWith;

/**
 * {@selfdoc}
 */
#[CoversClass(Operation::class)]
final class OperationTest extends UnitTestCase {

  /**
   * {@selfdoc}
   */
  #[TestWith([0, 0, NULL])]
  #[TestWith([-1, 0, 'Total jobs value must be greater or equal to zero.'])]
  #[TestWith([0, -1, 'Processed jobs value must be greater or equal to zero.'])]
  #[TestWith([10, 10, NULL])]
  #[TestWith([10, 11, 'Processed jobs value must not be greater than total jobs value.'])]
  public function testValidation(int $total_jobs, int $processed_jobs, ?string $expected_exception): void {
    if ($expected_exception) {
      self::expectExceptionObject(new \InvalidArgumentException($expected_exception));
    }
    new Operation(
      id: new OperationId('66e61b48-fc4e-4609-b0ae-791145aee2ad'),
      status: OperationStatus::Pending,
      created: new \DateTimeImmutable(),
      updated: new \DateTimeImmutable(),
      // @phpstan-ignore argument.type
      totalJobs: $total_jobs,
      // @phpstan-ignore argument.type
      processedJobs: $processed_jobs,
      result: NULL,
    );
    // @phpstan-ignore staticMethod.alreadyNarrowedType
    self::assertTrue(TRUE);
  }

  /**
   * {@selfdoc}
   */
  #[TestWith([OperationStatus::Pending, TRUE])]
  #[TestWith([OperationStatus::InProgress, TRUE])]
  #[TestWith([OperationStatus::Completed, FALSE])]
  #[TestWith([OperationStatus::Failed, FALSE])]
  #[TestWith([OperationStatus::Cancelled, FALSE])]
  public function testCanBeUpdated(OperationStatus $status, bool $expected_result): void {
    $operation = new Operation(
      id: new OperationId('66e61b48-fc4e-4609-b0ae-791145aee2ad'),
      status: $status,
      created: new \DateTimeImmutable(),
      updated: new \DateTimeImmutable(),
      totalJobs: 10,
      processedJobs: 5,
      result: NULL,
    );
    self::assertSame($expected_result, $operation->isActive());
  }

  /**
   * {@selfdoc}
   */
  public function testUpdate(): void {
    $original_operation = new Operation(
      id: new OperationId('a2c79890-8dbd-4210-93f7-ea859d452971'),
      status: OperationStatus::Pending,
      created: new \DateTimeImmutable('2024-01-01 00:00:00+00:00'),
      updated: new \DateTimeImmutable('2024-01-01 00:00:00+00:00'),
      totalJobs: 10,
      processedJobs: 0,
      result: NULL,
    );
    $expected_operation = new Operation(
      id: new OperationId('a2c79890-8dbd-4210-93f7-ea859d452971'),
      status: OperationStatus::InProgress,
      created: new \DateTimeImmutable('2024-01-01 00:00:00+00:00'),
      updated: new \DateTimeImmutable('2024-01-01 15:30:00+00:00'),
      totalJobs: 10,
      processedJobs: 5,
      result: [1, 2, 3, 4, 5],
    );
    $actual_operation = $original_operation->with(
      status: OperationStatus::InProgress,
      updated: new \DateTimeImmutable('2024-01-01 15:30:00+00:00'),
      processed_jobs: 5,
      result: [1, 2, 3, 4, 5],
    );
    self::assertEquals($expected_operation, $actual_operation);
    self::assertNotSame($expected_operation, $actual_operation);
  }

}
