<?php

namespace Drupal\optimize_database_tables\Commands;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\optimize_database_tables\Service\DbHandler;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for the optimize_database_tables module.
 */
class OptimizeDatabaseTablesCommands extends DrushCommands {

  use StringTranslationTrait;

  /**
   * Config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * DB handler service.
   *
   * @var \Drupal\optimize_database_tables\Service\DbHandler
   */
  protected DbHandler $dbHandler;

  /**
   * Constructs the Drush command with injected services.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    DbHandler $dbHandler,
  ) {
    parent::__construct();
    $this->configFactory = $configFactory;
    $this->dbHandler = $dbHandler;
  }

  /**
   * Run database optimization with progress output.
   *
   * Optimizes either all tables or the configured subset, according to
   * optimize_database_tables.settings. Displays a SymfonyStyle progress bar
   * via Drush IO. Shows total DB size before/after and saved space; with
   * --details prints per-table breakdown.
   *
   * @command optimize_database_tables:run
   * @aliases optimize-dbt:run
   * @option details Show per-table size details before/after
   * @usage drush optimize-dbt:run
   *   Optimize database tables with a progress bar.
   * @usage drush optimize-dbt:run --details
   *   Optimize and show per-table size details.
   */
  public function run(array $options = ['details' => FALSE]): void {
    $io = $this->io();

    $io->title('Optimize database tables');

    $config = $this->configFactory->get('optimize_database_tables.settings');
    $allTables = (bool) $config->get('all_tables');

    $tables = $allTables ?
      $this->dbHandler->getTablesList() :
      (array) $config->get('table_list');

    if (empty($tables)) {
      $link = Url::fromRoute('optimize_database_tables.settings')->toString();
      $io->warning((string) $this->t(
        'No tables to optimize. Check configuration at @link.',
        ['@link' => $link]),
      );
      return;
    }

    $count = count($tables);
    $io->text($allTables
      ? $this->t('Optimizing all tables (@count found).', ['@count' => $count])
      : $this->t('Optimizing @count selected table(s).', ['@count' => $count])
    );

    // Measure before sizes.
    $beforeTotal = $this->dbHandler->getTotalSizeBytes($tables);
    $perTableBefore = [];
    $showDetails = !empty($options['details']);
    if ($showDetails) {
      foreach ($tables as $t) {
        $name = is_string($t) ? $t : (string) $t;
        $perTableBefore[$name] = $this->dbHandler->getTableSizeBytes($name);
      }
    }

    $io->newLine();
    $io->progressStart($count);

    $start = microtime(TRUE);

    foreach ($tables as $table) {
      // Ensure $table is the table name, whether options are keyed or not.
      $tableName = is_string($table) ? $table : (string) $table;
      $this->dbHandler->optimizeTable($tableName);
      $io->progressAdvance();
    }

    $io->progressFinish();

    // Measure after sizes.
    $afterTotal = $this->dbHandler->getTotalSizeBytes($tables);

    $duration = microtime(TRUE) - $start;
    $saved = max(0, $beforeTotal - $afterTotal);
    $percent = $beforeTotal > 0 ? ($saved / $beforeTotal * 100) : 0;

    $io->success($this->t(
      'Optimization finished for @count table(s) in @seconds seconds.',
      ['@count' => $count, '@seconds' => number_format($duration, 2)]
    ));

    $io->definitionList(
      $this->t('Before'), $this->dbHandler->formatBytes($beforeTotal),
      $this->t('After'), $this->dbHandler->formatBytes($afterTotal),
      $this->t('Saved'), $this->dbHandler->formatBytes($saved) . ' (' . number_format($percent, 2) . '%)'
    );

    if ($showDetails) {
      $rows = [];
      foreach ($tables as $t) {
        $name = is_string($t) ? $t : (string) $t;
        $before = $perTableBefore[$name] ?? $this->dbHandler->getTableSizeBytes($name);
        $after = $this->dbHandler->getTableSizeBytes($name);
        $gain = max(0, $before - $after);
        $rows[] = [
          $name,
          $this->dbHandler->formatBytes($before),
          $this->dbHandler->formatBytes($after),
          $this->dbHandler->formatBytes($gain),
          ($before > 0) ? (number_format($gain / $before * 100, 2) . '%') : '—',
        ];
      }
      $io->newLine();
      $io->table([
        (string) $this->t('Table'),
        (string) $this->t('Before'),
        (string) $this->t('After'),
        (string) $this->t('Saved'),
        (string) $this->t('% Saved'),
      ], $rows);
    }
  }

}
